/*
*	 This file contains classes that represents image returned from camera.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/

#include <windows.h>
#include <vector>
using namespace std;

//FORWARD DECLARATIONS
namespace NSPipeline
{
	class TImageARGB;
	class TImageRGB;
	class TImageGray;

	class TImageSetReal;
	class TImageSetManager;
	class TFrameReal;
};

namespace NSSimpleUnit
{
	class TImageRGBResult;
}

#ifndef _PIPELINE_IMAGE_HH_
#define _PIPELINE_IMAGE_HH_

#include "pipeline/ImageAbstract.h"
#include "pipeline/CriticalSection.h"

namespace NSPipeline
{
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImageARGB

/** Tato trida obsahuje obrazy typu ARGB kde kazdy pixel zabira 32 bitu - tedy 8 bitu na kanal.
*	
*	V teto tride jsou obrazova data ulozena spojite - PITCH je tedy roven poctu pixelu v radku
*	nasobenych velikosti pixelu ( 4 bajtu ).
*
*	Kazdy pixel je reprezentovan jako DWORD 0xAARRGGBB.
*	Tedy v pameti je postupne ulozen nejprve kanal BLUE, GREEN, RED, ALPHA.
*
*	Objekt teto tridy muze byt vytvoren pouze tridou TImageSetReal.
*/
class TImageARGB : public TImage
{
friend class TImageSetReal;

//PUBLIC OVERRIDEN METHODS
public:
	virtual EnumImageType	GetImageType(void) const;
	virtual	DWORD			GetPixelSize(void) const;

	virtual DWORD			GetWidth(void) const;
	virtual DWORD			GetHeight(void) const;
	virtual DWORD			GetPitch(void) const;

	virtual DWORD			GetDataSize(void) const;
	virtual const void*		GetData(void) const;

//PRIVATE OVERRIDEN METHODS
private:
	virtual					~TImageARGB(void);

//PUBLIC METHODS
public:
	void*					GetDataWrite(void);
	void					FromGray( const TImageGray * image );
	void					FromRGB( const TImageRGB * image );

	bool					IsRectInside( DWORD x, DWORD y, DWORD width, DWORD height ) const;
	void					CopyFrom( const TImageARGB * src, DWORD x, DWORD y, DWORD width, DWORD height);

//PRIVATE METHODS
private:
							TImageARGB(DWORD width, DWORD height);

//PRIVATE FAKE METHODS
private:
							TImageARGB( const TImageARGB & orig );	///< falesny kopirovaci konstruktor
	void					operator=( const TImageARGB & orig );	///< falesny operator prirazeni

//PRIVATE COMPONENTS
private:
	unsigned char*			m_data;			///< data obrazu

	DWORD					m_width;		///< sirka v pixelech
	DWORD					m_height;		///< vyska v pixelech
};
//OK 2007-08-25 14:51:16 B04-315B\Tom

/** Vrati typ tohoto obrazku.
*/
inline EnumImageType TImageARGB::GetImageType(void) const
{
	return ENUM_IMAGETYPE_ARGB; 
}
//OK 2007-08-25 14:51:23 B04-315B\Tom

/** Vrati ukazatel na data.
*/
inline void* TImageARGB::GetDataWrite(void)
{
	return m_data;
}
//OK 2007-08-25 14:51:27 B04-315B\Tom

/** Tato metoda urci jestli je zadany obdelnik uvnitr tohoto obrazu.
*
*	\param	x		[in] x-souradnice leveho horniho rohu obdelniku
*	\param	y		[in] y-souradnice leveho horniho rohu obdelniku
*	\param	width	[in] sirka obdelniku v pixelech
*	\param	height	[in] vysla obdelniku v pixelech
*/
inline bool TImageARGB::IsRectInside( DWORD x, DWORD y, DWORD width, DWORD height ) const
{
	return 
		( x + width ) <= ( m_width ) &&
		( y + height ) <= ( m_height );		
}
//OK 2007-08-26 17:08:05 B04-315B\Tom

// TImageARGB
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImageRGB

/** Tato trida obsahuje obrazy typu RGB kde kazdy pixel zabira 24 bitu - tedy 8 bitu na kanal.
*	
*	V teto tride jsou obrazova data ulozena spojite - PITCH je tedy roven poctu pixelu v radku
*	nasobenych velikosti pixelu ( 4 bajty ).
*
*	Kazdy pixel je reprezentovan jako BYTE[3].
*	Tedy v pameti je postupne ulozen nejprve kanal BLUE, GREEN, RED.
*
*	Objekt teto tridy muze byt vytvoren pouze tridou TImageSetReal.
*/
class TImageRGB : public TImage
{
friend class TImageSetReal;
friend class NSSimpleUnit::TImageRGBResult;

//PUBLIC OVERRIDEN METHODS
public:
	virtual EnumImageType	GetImageType(void) const;
	virtual	DWORD			GetPixelSize(void) const;

	virtual DWORD			GetWidth(void) const;
	virtual DWORD			GetHeight(void) const;
	virtual DWORD			GetPitch(void) const;

	virtual DWORD			GetDataSize(void) const;
	virtual const void*		GetData(void) const;

//PRIVATE OVERRIDEN METHODS
private:
	virtual					~TImageRGB(void);

//PUBLIC METHODS
public:
	void*					GetDataWrite(void);
	void					FromGray( const TImageGray * image );
	void					FromARGB( const TImageARGB * image );

	bool					IsRectInside( DWORD x, DWORD y, DWORD width, DWORD height ) const;
	void					CopyFrom( const TImageRGB * src, DWORD x, DWORD y, DWORD width, DWORD height);

//PRIVATE METHODS
private:
							TImageRGB(DWORD width, DWORD height);

//PRIVATE FAKE METHODS
private:
							TImageRGB( const TImageRGB & orig );	///< falesny kopirovaci konstruktor
	void					operator=( const TImageRGB & orig );	///< falesny operator prirazeni

//PRIVATE COMPONENTS
private:
	unsigned char*			m_data;			///< data obrazu

	DWORD					m_width;		///< sirka v pixelech
	DWORD					m_height;		///< vyska v pixelech
};
//OK 2007-09-04 12:09:37 B04-315B\Tom

/** Vrati typ tohoto obrazku.
*/
inline EnumImageType TImageRGB::GetImageType(void) const
{
	return ENUM_IMAGETYPE_RGB; 
}
//OK 2007-09-04 12:09:42 B04-315B\Tom

/** Vrati ukazatel na data.
*/
inline void* TImageRGB::GetDataWrite(void)
{
	return m_data;
}
//OK 2007-09-04 12:10:02 B04-315B\Tom

/** Tato metoda urci jestli je zadany obdelnik uvnitr tohoto obrazu.
*
*	\param	x		[in] x-souradnice leveho horniho rohu obdelniku
*	\param	y		[in] y-souradnice leveho horniho rohu obdelniku
*	\param	width	[in] sirka obdelniku v pixelech
*	\param	height	[in] vysla obdelniku v pixelech
*/
inline bool TImageRGB::IsRectInside( DWORD x, DWORD y, DWORD width, DWORD height ) const
{
	return 
		( x + width ) <= ( m_width ) &&
		( y + height ) <= ( m_height );		
}
//OK 2007-08-26 17:08:05 B04-315B\Tom

// TImageRGB
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImageGray

/** Tato trida obsahuje monochromaticke obrazy s 8 bity na pixel.
*	
*	V teto tride jsou obrazova data ulozena spojite - PITCH je tedy roven poctu pixelu v radku
*	nasobenych velikosti pixelu ( 1 bajt ).
*
*	Kazdy pixel je reprezentovan jako 8-bitova hodnota.
*
*	Objekt teto tridy muze byt vytvoren pouze tridou TImageSetReal.
*/
class TImageGray : public TImage
{
friend class TImageSetReal;

//PUBLIC OVERRIDEN METHODS
public:
	virtual EnumImageType	GetImageType(void) const;
	virtual	DWORD			GetPixelSize(void) const;

	virtual DWORD			GetWidth(void) const;
	virtual DWORD			GetHeight(void) const;
	virtual DWORD			GetPitch(void) const;

	virtual DWORD			GetDataSize(void) const;
	virtual const void*		GetData(void) const;

//PRIVATE OVERRIDEN METHODS
private:
	virtual					~TImageGray(void);

//PUBLIC METHODS
public:
	void*					GetDataWrite(void);
	void					FromARGB( const TImageARGB * image );
	void					FromRGB( const TImageRGB * image );

	bool					IsRectInside( DWORD x, DWORD y, DWORD width, DWORD height ) const;
	void					CopyFrom( const TImageGray * src, DWORD x, DWORD y, DWORD width, DWORD height);

//PRIVATE METHODS
private:
							TImageGray(DWORD width, DWORD height);

//PRIVATE FAKE METHODS
private:
							TImageGray( const TImageGray & orig );	///< falesny kopirovaci konstruktor
	void					operator=( const TImageGray & orig );	///< falesny operator prirazeni

//PRIVATE COMPONENTS
private:
	unsigned char*			m_data;			///< data obrazu

	DWORD					m_width;		///< sirka v pixelech
	DWORD					m_height;		///< vyska v pixelech
};
//OK 2007-08-25 14:52:16 B04-315B\Tom

/** Vrati typ tohoto obrazku.
*/
inline EnumImageType TImageGray::GetImageType(void) const
{
	return ENUM_IMAGETYPE_GRAY; 
}
//OK 2007-08-25 14:52:40 B04-315B\Tom

/** Vrati ukazatel na data.
*/
inline void* TImageGray::GetDataWrite(void)
{
	return m_data;
}
//OK 2007-08-25 14:52:41 B04-315B\Tom

/** Tato metoda urci jestli je zadany obdelnik uvnitr tohoto obrazu.
*
*	\param	x		[in] x-souradnice leveho horniho rohu obdelniku
*	\param	y		[in] y-souradnice leveho horniho rohu obdelniku
*	\param	width	[in] sirka obdelniku v pixelech
*	\param	height	[in] vysla obdelniku v pixelech
*/
inline bool TImageGray::IsRectInside( DWORD x, DWORD y, DWORD width, DWORD height ) const
{
	return 
		( x + width ) <= ( m_width ) &&
		( y + height ) <= ( m_height );		
}
//OK 2007-08-26 17:08:05 B04-315B\Tom

// TImageGray
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/** Tato trida obsahuje ruzne verze jednoho obrazu ulozene jako objekty
*	typu TImage a odvozenych trid.
*
*	Objekty tohoto typu nejsou konstruovany primo, ale pomoci manageru techto objektu
*	typu TImageSetManager. Tento manager je zodpovedny za spravu techto objektu.
*/
class TImageSetReal : public TImageSet
{
friend class TImageSetManager;
friend class TFrameReal;

//PUBLIC OVERRIDEN METHODS
public:
	virtual const TImage*	GetARGB(void) const;
	virtual const TImage*	GetGray(void) const;
	virtual const TImage*	GetRGB(void) const;	

//PRIVATE OVERRIDEN METHODS
private:
	virtual					~TImageSetReal(void);

//PUBLIC METHODS
public:
	TImageARGB*				GetARGBWrite(void) const;
	TImageGray*				GetGrayWrite(void) const;
	TImageRGB*				GetRGBWrite(void) const;	

//PRIVATE METHODS
private:
							TImageSetReal(TImageSetManager * manager, DWORD id, DWORD width, DWORD height);

	DWORD					GetID(void) const;
	TImageSetManager*		GetManager(void);

//PRIVATE FAKE METHODS
private:
							TImageSetReal( const TImageSetReal & orig );///< falesny kopirovaci konstruktor
	void					operator=( const TImageSetReal & orig );	///< falesny operator prirazeni

//PRIVATE COMPONENTS
private:
	TImageARGB*				m_image_ARGB;		///< obraz typu ARGB - 32 bitu na pixel
	TImageGray*				m_image_gray;		///< obraz sedotonovy - 8 bitu na pixel
	TImageRGB*				m_image_RGB;		///< obraz typu RGB - 24 bitu na pixel

	TImageSetManager*		m_manager;			///< manazer kde byl tento objekt vytvoren
	DWORD					m_managerID;		///< jednoznacny identifikator v manageru
												///< podle nej manager pozna objekt pri vraceni do manageru
};
//OK 2007-08-25 14:53:42 B04-315B\Tom

/** Vrati obraz ARGB s moznosti upravit ho.
*/
inline TImageARGB* TImageSetReal::GetARGBWrite(void) const
{
	return m_image_ARGB;
}
//OK 2007-08-25 14:56:33 B04-315B\Tom

/** Vrati monochromaticky obraz s moznosti upravit ho.
*/
inline TImageGray* TImageSetReal::GetGrayWrite(void) const
{
	return m_image_gray;
}
//OK 2007-08-25 14:56:35 B04-315B\Tom

/** Vrati obraz RGB s moznosti upravit ho.
*/
inline TImageRGB* TImageSetReal::GetRGBWrite(void) const
{
	return m_image_RGB;
}
//OK 2007-09-04 14:13:57 B04-315B\Tom

/** Vrati jednoznacny identifikator vraceny manazerem techto objektu.
*/
inline DWORD TImageSetReal::GetID(void) const
{
	return m_managerID;
}
//OK 2007-08-25 14:56:36 B04-315B\Tom

/** Vrati manazer kde byl tento snimek vytvoren.
*/
inline TImageSetManager* TImageSetReal::GetManager(void)
{
	return m_manager;
}
//OK 2007-08-25 14:56:38 B04-315B\Tom

// TImageSetReal
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImageSetManager

/** Tato trida slouzi jako manazer pro tvorbu objektu typu TImageSetReal.
*
*	Objekty jsou zde alokovany a dealokovany. Jinym zpusobem nelze tyto objekty vytvorit.
*
*	Trida obsahuje predalokovane objekty, takze aplikace neni zatezovana neustalou alokaci
*	a dealokaci objektu.
*
*	Pri vytvareni je nutne zadat maximalni pocet predalokovanych objektu. Vetsi mnozstvi
*	objektu manager nealokuje a pokud jiz neni zadny dalsi objekt k dispozici pak vraci
*	manager NULL.
*/
class TImageSetManager
{
//PUBLIC METHODS
public:
							TImageSetManager( DWORD width, DWORD height, DWORD initSize );
							~TImageSetManager(void);

	TImageSetReal*			GetImageSet(void);
	void					ReleaseImageSet(const TImageSetReal * img);

//PRIVATE FAKE METHODS
private:
							TImageSetManager( const TImageSetManager & orig );	///< falesny kopirovaci konstruktor
	void					operator=( const TImageSetManager & orig );			///< falesny operator prirazeni

//PRIVATE COMPONENTS
private:
	TCriticalSection		m_critical;		///< synchronizace pro GetImage() a ReleaseImage()

	DWORD					m_width;		///< zakladni sirka obrazku v pixelech
	DWORD					m_height;		///< zakladni vyska obrazku v pixelech	

	vector<TImageSetReal*>	m_imageSets;	///< pole s s predalokovanymi objekty pro obrazova data
	vector<BOOL>			m_imgFlags;		///< pole stejne velikosti jako m_imageSets
											///< pro kazdy objekt obsahuje informaci zda je
											///< nebo neni aktualne pouzit
};
//OK 2007-08-25 14:57:44 B04-315B\Tom

// TImageSetManager
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TFrameReal

/** Tato trida reprezentuje snimek ziskany z kamery.
*
*	Snimek muze obsahovat vice obrazu (v ruznych formatech), ktere jsou dostupne pres metodu
*	TFrame::GetImageSet().
*
*	Kazdy snimek ma prirazenu casovou znacku jeho vytvoreni, takze lze jednotlive snimky
*	snadno identifikovat. Casova znacka obsahuje take identifikator, kterym lze rozpoznat
*	poradi snimku ve kterem byly generovany.
*/
class TFrameReal : public TFrame
{
//PUBLIC OVERRIDEN METHODS
public:
	virtual const TImageSet*	GetImageSet(void) const;

	virtual const TTimeStamp &	GetTimestamp(void) const;

	virtual void				AddRefs(void);
	virtual void				Release(void);
	virtual DWORD				GetRefs(void) const;

	virtual DWORD				AddLock(void);
	virtual DWORD				ReleaseLock(void);
	virtual DWORD				GetLockCount(void) const;

//PRIVATE OVERRIDEN METHODS
private:
	virtual						~TFrameReal(void);

//PUBLIC STATIC METHODS
public:							
	static TFrameReal*			Create( TImageSetReal * images );

//PRIVATE FAKE METHODS
private:
								TFrameReal( const TFrameReal & orig );	///< falesny kopirovaci konstruktor
	void						operator=( const TFrameReal & orig );	///< falesny operator prirazeni

//PRIVATE METHODS
private:
								TFrameReal( TImageSetReal * images );

//PRIVATE COMPONENTS
private:	
	DWORD						m_refs;			///< pocet referenci na snimek
	DWORD						m_locks;		///< pocet zamku na snimku
	TCriticalSection			m_critSection;
	TCriticalSection			m_critSection2;

	TTimeStamp					m_time;			///< cas vytvoreni snimku
	TImageSetReal*				m_image;		///< data v tomto snimku
};

/** Vytvoreni snimku z pripravene mnoziny obrazu.
*
*	\param	images	[in] platna mnozina obrazu
*/
inline TFrameReal* TFrameReal::Create( TImageSetReal * images )
{
	return new TFrameReal( images );
}
//OK 2007-08-25 15:03:48 B04-315B\Tom

// TFrameReal
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

}; //end of NSPipeline
using namespace NSPipeline;

#endif