/*
*	This file contains abstract class which represents working units.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/

// FORWARD DECLARATIONS
namespace NSPipeline
{
	class TUnit;
};

#ifndef _PIPELINE_UNIT_HH_
#define _PIPELINE_UNIT_HH_

#include "pipeline/UnitTypes.h"
#include "pipeline/ImageAbstract.h"

namespace NSPipeline
{

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TUnitInterface

/** Tato trida reprezentuje rozhrani pracovni jednotky.
*
*	Toto rozhrani (nebo trida z nej odvozena) je dostupne aplikaci.
*
*	U metod Start(), Stop() a Release() se predpoklada, ze budou volany pouze
*	z jednoho vlakna a tedy neni vyzadovana synchronizace pro vicevlaknovy pristup.
*
*	Metoda GetType() typicky vraci konstantu a v takovem pripade soubezny pristup
*	nemusi byt synchronizovan.
*
*	Metoda GetResult() vraci vysledky pro zadane snimky (podle ID snimku) - je volana pouze
*	jednim vlaknem, ale je nutne zajistit, ze vnitrni vlakno jednotky ukladajici vysledky 
*	bude synchronizovano s vlaknem, ktere pristupuje k temto vysledkum.
*/
class TUnitInterface
{
//PUBLIC METHODS
public:
	/** Vrati typ navratove hodnoty teto jednotky.
	*/
	virtual EnumUnitType			GetType(void) = 0;

	/** Vysledek pro snimek s pozadovanym identifikatorem.
	*
	*	Muze vracet NULL pokud vysledek neni k dispozici.
	*
	*	\param	id	[in] identifikator vysledku (z casove znacky)
	*/
	virtual TUnitRetTypeInterface*	GetResult(DWORD id) = 0;

	/** Vraci hodnotu udavajici pocet snimku, ktere jednotce
	*	"utecou" behem zpracovani dvou snimkou.
	*
	*	Tato hodnota se meni behem zpracovani a jednotka ji pocita jako 
	*	rozdil mezi ID poslednich dvou snimku.
	*/
	virtual DWORD			GetFrameInterval() = 0;

	/** Spusteni vypoctu jednotky.
	*
	*	Vraci TRUE pokud jednotka jiz bezi nebo byla uspesne spustena.
	*/
	virtual BOOL			Start(void) = 0;

	/** Ukonceni vypoctu jednotky.
	*
	*	\warning	Musi uvolnit vsechny vysledky ktere jsou ulozene v jednotce!
	*
	*	Vraci TRUE v pripade zastaveni jednotky, FALSE pri kriticke chybe.
	*/
	virtual BOOL			Stop(void) = 0;

	/** Odstrani jednotku.
	*
	*	Pred odstranenim jednotky by mela byt jednotka zastavena volanim metody Stop.
	*/
	virtual void			Release(void) = 0;

//PROTECTED METHODS
protected:
							TUnitInterface(void){};
							virtual ~TUnitInterface(void){};

//PROTECTED FAKE METHODS
protected:
							TUnitInterface( const TUnitInterface & orig );	///< falesny kopirovaci konstruktor
	void					operator=( const TUnitInterface & orig );		///< falesny prirazovaci operator

};
//OK 2007-08-25 18:19:20 B04-315B\Tom

// TUnitInterface
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/** Typ DLL knihovny.
*
*	Typ urcuje parametry funkci CreateUnit v DLL pres kterou se vytvari vypocetni jednotka.
*/
enum EnumDLLType
{
	/**
	*	Zakladni typ DLL - funkce CreateUnit vyzaduje pouze 
	*	ukazatel na rozhrani dispatcheru.
	*
	*	CreateUnit( TDispatcherInterface * dispatcher )	
	*/
	ENUM_DLLTYPE_BASIC = 0,

	/**
	*	Funkce CreateUnit vyzaduje rozhrani dispatcheru
	*	a jako druhy parametr retezec zakonceny nulou.
	*
	*	CreateUnit( TDispatcherInterface * dispatcher, const char * string )
	*/	
	ENUM_DLLTYPE_STRING = 1,											

	ENUM_DLLTYPE_FORCED32 = 0x7fffffff		///< zarovnani na 32-bitu
};
//OK 2007-08-25 18:19:18 B04-315B\Tom

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/** Ukazatel na funkci pro ziskani informaci o typu navratove funkce CreateUnit.
*
*	Funkce se jmenuje "GetType".
*/
typedef EnumDLLType (*PTR_DLLTYPE) (void);

/** Ukazatel na funkci z DLL knihovny, pres kterou lze ziskat vypocetni jednotku.
*
*	Funkce se jmenuje "CreateUnit".
*/
typedef TUnitInterface*	(*PTR_UNITGET_BASIC)( TDispatcherInterface * dispatcher );

/** Ukazatel na funkce z DLL knihovny, pres kterou lze ziskat vypocetni jednotku.
*
*	Funkce se jmenuje "CreateUnit".
*/
typedef TUnitInterface*	(*PTR_UNITGET_STRING)( TDispatcherInterface * dispatcher, const char * str );

}; //end of NSPipeline
using namespace NSPipeline;

#endif