/*
*	This file contains class that load images from specified video sequence.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/


#include "StdAfx.h"
#include <tchar.h>
#include <strsafe.h>
#include <cstdlib>
#include "CameraSource.h"


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TDSSampler

/** Vytvori filter s jednou referenci.
*
*	\param	unk		[in] povinne predavane rozhrani pro bazovou tridu CBaseVideoRenderer - muze byt NULL
*	\param	hr		[in] promenna pro ulozeni vysledku
*/
TDSSampler::TDSSampler( IUnknown* unk, HRESULT *hr ):
		CBaseVideoRenderer(__uuidof(CLSID_Sampler), NAME("RGB Sampler"), unk, hr)
{	
	this->AddRef();

	m_data1 = NULL;
	m_data2 = NULL;
};

/** Destruktor.
*
*	Uvolni vsechny prostredky.
*/
TDSSampler::~TDSSampler()
{
	delete [] m_data1;
	delete [] m_data2;
}

/** Overeni typu pripojeneho datoveho zdroje.
*
*	\param	media	[in] parametry vstupniho datoveho zdroje zaslane DirectShow grafem.
*/
HRESULT TDSSampler::CheckMediaType( const CMediaType* media ) 
{
	if ( NULL == media )
	{
		return E_FAIL;
	}

	//mozne formaty jsou pouze VIDEOINFO a VIDEOINFO2
	//FORMAT_VideoInfo
	if ( FORMAT_VideoInfo == ( *media->FormatType() ) )
	{
		VIDEOINFOHEADER* vi;

		if (
			! IsEqualGUID( *( media->Subtype() ), MEDIASUBTYPE_RGB24) ||	// musi byt RGB format
			! ( vi = (VIDEOINFOHEADER *)media->Format() )					// musi existovat platna hlavicka
		) 
		{
			//podminka nesplnena => filtr nelze pripojit
			return E_FAIL;
		}
		else
		{
			//pouzitelny format
			m_info = vi->bmiHeader;
		}
	}
	else
	{
		//FORMAT_VideoInfo2
		if ( FORMAT_VideoInfo2 == ( *media->FormatType() ) )
		{
			VIDEOINFOHEADER2* vi;

			if (
				! IsEqualGUID( *( media->Subtype() ), MEDIASUBTYPE_RGB24) ||	// musi byt RGB format
				! ( vi = (VIDEOINFOHEADER2 *)media->Format() )					// musi existovat platna hlavicka
			) 
			{
				//podminka nesplnena => filtr nelze pripojit
				return E_FAIL;
			}
			else
			{
				//pouzitelny format
				m_info = vi->bmiHeader;
			}
		}
		else
		{
			// nepouzitelny format
			return E_FAIL;
		}
	}	

	//ulozime informace o datovem formatu

	m_rgb_width		= m_info.biWidth;
	m_rgb_height	= abs(m_info.biHeight);
	m_rgb_size		= m_rgb_width * m_rgb_height * 3;

	m_rowsFliped	= m_info.biHeight >= 0;

	//vypocitame zarovnovaci bajty - tj. delku celeho radku v puvodnim obraze
	int tmp_width = m_rgb_width * 3;	//sirka v bajtech
	if ( (tmp_width % 4) > 0 )
	{
		m_img_align = 4 - ( tmp_width % 4 );
	}
	else
	{
		m_img_align = 0;
	}
	m_img_stride = tmp_width + m_img_align;

	//vytvorime pole pro vysledky
	delete [] m_data1;
	m_data1 = NULL;
	delete [] m_data2;
	m_data2 = NULL;

	m_data1 = new BYTE[m_rgb_size];
	m_data2 = new BYTE[m_rgb_size];

	return  S_OK;
}

/** Ziskani dalsiho snimku z datoveho toku.
*
*	Metoda je volana grafem DirectShow.
*
*	\param	sample	[in] dalsi ziskany vzorek
*/
HRESULT TDSSampler::DoRenderSample(IMediaSample *sample)
{
	//zapiseme data do druheho snimku
	BYTE * src = NULL;
	sample->GetPointer( &src );

	//kopirujeme radek po radku
	//musime zvolit kopirovani podle poradi radku
	DWORD offsetSrc = 0;
	DWORD offsetDst = 0;	
	if ( m_rowsFliped )
	{
		//radky odspoda nahoru
		for ( int r = (m_rgb_height - 1) ; r >= 0 ; r-- )
		{
			offsetDst = r * m_rgb_width * 3;

			for ( int c = 0 ; c < m_rgb_width ; c++ ) //sloupce
			{
				m_data2[offsetDst++] = src[offsetSrc++];	// B
				m_data2[offsetDst++] = src[offsetSrc++];	// G
				m_data2[offsetDst++] = src[offsetSrc++];	// R				
			}
			offsetSrc += m_img_align;
		}
	}
	else
	{
		//radky odshoda dolu
		for ( int r = 0 ; r < m_rgb_height ; r++ )
		{
			for ( int c = 0 ; c < m_rgb_width ; c++ ) //sloupce
			{
				m_data2[offsetDst++] = src[offsetSrc++];	// B
				m_data2[offsetDst++] = src[offsetSrc++];	// G
				m_data2[offsetDst++] = src[offsetSrc++];	// R				
			}
			offsetSrc += m_img_align;
		}
	}

	//vymenime ukazatele
	m_cs.Lock();
		BYTE * tmp = m_data1;
		m_data1 = m_data2;
		m_data2 = tmp;
	m_cs.Unlock();

	return  S_OK;
}

/** Metoda grafu DirectShow pro overeni co udelat se zadanym snimkem.
*
*	V tomto pripade se kazdy snimek bude spravne zobrazovat podle jeho zadaneho casu.
*	
*	\param	sample		[in] snimek o kterem se rozhoduje
*	\param	start		[in] cas pro pocatek zobrazeni snimku
*	\param	stop		[in] cas pro konec zobrazeni snimku
*/
HRESULT TDSSampler::ShouldDrawSampleNow(
	IMediaSample *sample,
	REFERENCE_TIME *start,
	REFERENCE_TIME *stop
	) 
{
	//zadny snimek nebude zahozen - kazdy snimek bude zobrazen podle jeho casu
	// vracenim S_OK dosahneme okamziteho zobrazeni snimku !!!
	return S_FALSE;
}

/** Nakopiruje data do pripraveneho bufferu.
*
*	\param	data	[in out] buffer pro ulozeni dat ve formatu RGB 
*/
void TDSSampler::GetData( void * data )
{
	//posledni snimek je ulozen v M_DATA1
	m_cs.Lock();

		//prekopirujeme data
		memcpy( data, m_data1, m_rgb_size );

	m_cs.Unlock();
}

// TDSSampler
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/** Konstruktor.
*
*	Inicializuje vse na NULL.
*
*	Inicializace musi byt provedena metodou Initialize() !!!
*/
TVideoDSFile::TVideoDSFile(void)
{
	m_graph = NULL;
	m_ctrl = NULL;
	m_sampler = NULL;
}

/** Destruktor.
*
*	Odstrani DirectShow graf a deinicializuje COM.
*/
TVideoDSFile::~TVideoDSFile(void)
{
	this->Destroy();
}

/** Najde prvni kameru v systemu a jeji vystupni pin.
*
*	Vraci TRUE pri nalezeni kamery. Jinak jsou vystupni parametry neplatne.
*
*	\param	camera		[out] adresa pro ulozeni ukazatele na kameru
*	\param	cameraOut	[out] adresa pro ulozeni ukazatele na vystup z kamery
*/
bool TVideoDSFile::EnumCamera( IBaseFilter ** camera, IPin ** cameraOut )
{
	HRESULT hr;

    ICreateDevEnum* enums = NULL;
	hr = CoCreateInstance ( CLSID_SystemDeviceEnum, 0, CLSCTX_INPROC, IID_ICreateDevEnum, (void **) &enums );
	if ( FAILED ( hr ) )
	{
		return false;
	}

    IEnumMoniker* enumsMon = NULL; 
	hr = enums->CreateClassEnumerator ( CLSID_VideoInputDeviceCategory, &enumsMon, 0 );  
	if ( FAILED ( hr ) )
	{
		enums->Release();
		return false;
	}

    IMoniker* moniker = NULL; 
	hr = enumsMon->Next (1, &moniker, 0 );
	if ( FAILED( hr ) )
	{
		enumsMon->Release();
		enums->Release();
		return false;
	}

    IBaseFilter* lcamera = NULL;
	hr = moniker->BindToObject( 0, 0, IID_IBaseFilter, (void**) &lcamera );
	if ( FAILED ( hr ) )
	{
		moniker->Release();
		enumsMon->Release();
		enums->Release();
		return false;
	}

    IEnumPins* enumsPins = NULL; 
	hr = lcamera->EnumPins( &enumsPins ); 
	if ( FAILED( hr ) )
	{
		lcamera->Release();
		moniker->Release();
		enumsMon->Release();
		enums->Release();
		return false;
	}

    IPin* lcameraOut  = 0;
	hr = enumsPins->Next( 1, &lcameraOut, 0);
	if ( FAILED( hr ) )
	{
		enumsPins->Release();
		lcamera->Release();
		moniker->Release();
		enumsMon->Release();
		enums->Release();
		return false;
	}

	*camera = lcamera;
	*cameraOut = lcameraOut;

	return true;
}

/** Inicializace DirectShow grafu.
*
*	Vraci TRUE pokud probehla inicializace uspesne nebo pokud byl jiz graf otevren.
*
*/
bool TVideoDSFile::Initialize( void )
{
	if ( ! m_graph )
	{
		//vytvorime novy graf

		//inicializace COM
		if ( FAILED( CoInitialize( 0 ) ) )
		{
			return false;
		}

		//vytvorime graf a ziskame jeho ovladani
		HRESULT hr = 
			CoCreateInstance( CLSID_FilterGraph, 0, CLSCTX_INPROC, IID_IGraphBuilder, (void **)  &m_graph );
		if ( FAILED( hr ) )
		{
			m_graph = NULL;
			CoUninitialize();
			return false;
		}

		hr = m_graph->QueryInterface( IID_IMediaControl, (void **) &m_ctrl );
		if ( FAILED( hr ) )
		{
			m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}

		//pomocny objekt pro tvorbu grafu
		ICaptureGraphBuilder2 * builder = NULL;
		hr = CoCreateInstance( CLSID_CaptureGraphBuilder2, 0, CLSCTX_INPROC, IID_ICaptureGraphBuilder2, (void **)  &builder );
		if ( FAILED( hr ) )
		{
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}

		hr = builder->SetFiltergraph( m_graph );
		if ( FAILED( hr ) )
		{
			builder->Release(); builder = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}

		//vytvorime sampler - ma jednu referenci ziskanou v kontruktoru
		m_sampler = new TDSSampler( 0, &hr );

		//ziskame zdroj videa
		
		IBaseFilter* video  = NULL; 
		IPin* videoOut  = NULL;

		if ( ! this->EnumCamera( &video, &videoOut ) )
		{
			m_sampler->Release(); m_sampler = NULL;
			builder->Release(); builder = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}
		//pin nebudeme potrebovat
		videoOut->Release();
		videoOut = NULL;
		
		//pridame kameru do grafu
		hr = m_graph->AddFilter( video, L"DirectShow camera" );
		if ( FAILED ( hr ) )
		{			
			video->Release(); video = NULL;
			m_sampler->Release(); m_sampler = NULL;
			builder->Release(); builder = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}		
		
		//sestavime graf
		hr = m_graph->AddFilter( (IBaseFilter*)m_sampler, L"Sampler" );
		if ( FAILED( hr ) )
		{
			video->Release(); video = NULL;
			m_sampler->Release(); m_sampler = NULL;
			builder->Release(); builder = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}

		hr = builder->RenderStream( &PIN_CATEGORY_CAPTURE, &MEDIATYPE_Video, video, NULL, m_sampler );
		if ( FAILED( hr ) )
		{
			video->Release(); video = NULL;
			m_sampler->Release(); m_sampler = NULL;
			builder->Release(); builder = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}

		//bulider uz neni dale potreba
		builder->Release();
		builder = NULL;

		//kameru uz take ne - je v grafu
		video->Release();
		video = NULL;	

		//nyni by mel mit sampler informace o videoformatu
		{
			m_width = m_sampler->GetWidth();
			m_height = m_sampler->GetHeight();
		}

		//spustime graf
		hr = m_ctrl->Run();
		if ( FAILED( hr ) )
		{
			m_sampler->Release(); m_sampler = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}

		return true;
	}
	else
	{
		//graf uz existuje
		return true;
	}
}

/** Odstraneni aktualniho grafu (pokud existuje) a deinicializace COM.
*/
bool TVideoDSFile::Destroy(void)
{
	if ( m_graph )
	{
		//nejprve graf radeji zastavime
		m_ctrl->Stop();

		//odstranime graf
		if ( m_sampler )
		{
			m_sampler->Release();
			m_sampler = NULL;
		}

		if ( m_ctrl )
		{
			m_ctrl->Release();
			m_ctrl = NULL;
		}

		m_graph->Release();
		m_graph = NULL;

		CoUninitialize();
	}

	return true;
}

/** Nakopiruje data ve formatu RGB z posledniho zachyceneho obrazku (pokud existuje)
*	do pripraveneho bufferu, jehoz velikost musi byt minimalne 
*
*		GetWidth() * GetHeight() * 3 bajty
*
*	\param	data	[in out] buffer pro pixely ve formatu RGB
*/
void TVideoDSFile::GetDataRGB( void * data )
{
	m_sampler->GetData( data );
}

// TVideoDSFile
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
