/*
*	This file contains class that load images from specified file.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/

#include "StdAfx.h"
#include "CamImage.h"

#include <cstdlib>
#include <cstring>
#include <cmath>

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TCameraImage

/** Funkce pro vytvoreni tridy a preneseni ven z DLL.
*
*	\warning	Muze vratit NULL pri chybe.
*
*	\param	filename	[in] jmeno souboru s obrazkem
*/
extern "C" TCameraAbstract*	CreateCamera(const char * filename)
{
	return TCameraImage::CreateCamera(filename);
}

/** Tato staticka metoda vytvori kameru.
*
*	\warning	Vraci NULL pri chybe.
*
*	\param	filename	[in] jmeno souboru s obrazkem
*/
TCameraImage* TCameraImage::CreateCamera(const char * filename)
{
	//vytvorime objekt
	TCameraImage * cam = new TCameraImage;

	//inicializujeme
	if ( cam->Initialize( filename ) )
	{
		//ok - kamera inicializovana v poradku
		return cam;
	}
	else
	{
		//nejaka chyba
		delete cam;
		return NULL;
	}
}

/** Soukromy konstruktor.
*
*	Pouze vytvori tridu. Ta musi byt inicializovana metodou Initialize() !!!
*/
TCameraImage::TCameraImage(void)
{
	m_data = NULL;
}

/** Soukromy destruktor.
*/
TCameraImage::~TCameraImage(void)
{
	delete [] m_data;
	m_data = NULL;
}

/** Tato metoda inicializuje tridu po vytvoreni.
*
*	\param	filename	[in] jmeno souboru s nacitanym obrazkem
*/
bool TCameraImage::Initialize( const char * filename)
{
	// inicializace GDI+
	GdiplusStartupInput gdiplusStartupInput;
	ULONG_PTR           gdiplusToken;
	GdiplusStartup(&gdiplusToken, &gdiplusStartupInput, NULL);

	//prevod na WCHAR
	size_t len = strlen( filename );
	size_t bytes = mbstowcs( NULL, filename, strlen( filename ) ) + 1;
	wchar_t * str_dst = new wchar_t[bytes];
	mbstowcs( str_dst, filename, strlen( filename ) + 1 );

	//musime nacist obrazek
	Bitmap * bmp = Bitmap::FromFile( str_dst );

	delete [] str_dst;

	if ( ! bmp )
	{
		//nejaky chyba pri nacitani obrazku

		GdiplusShutdown(gdiplusToken);

		return false;
	}

	//ok - mame obraz
	int width = (int)bmp->GetWidth();
	int height = (int)bmp->GetHeight();
	
	//vytvorime pomocny obraz
	Bitmap * bmpTemp = new Bitmap( width, height, PixelFormat24bppRGB );
	if ( ! bmpTemp )
	{
		delete bmp;

		GdiplusShutdown(gdiplusToken);

		return false;
	}

	//nyni vykreslime nacteny obraz na nas obraz
	Graphics * graph = Graphics::FromImage( bmpTemp );
	if ( ! graph )
	{
		delete bmp;
		delete bmpTemp;		

		GdiplusShutdown(gdiplusToken);

		return false;
	}

	graph->DrawImage( bmp, 0, 0 );

	delete graph;
	delete bmp;

	//precteme data z obrazku ve formatu RGB
	Rect rect( 0, 0, width, height );
	BitmapData data;
	if ( Ok == bmpTemp->LockBits( &rect, ImageLockModeRead, PixelFormat24bppRGB , &data ) )
	{
		const DWORD rowSize = data.Width * 3;	//delka radku obrazu v bajtech
		const DWORD pitch = (DWORD) abs(data.Stride);

		m_width		= (int)data.Width;
		m_height	= (int)data.Height;

		//kopirujeme data
		m_data = new unsigned char[ 3 * m_width * m_height ];
		const char * m_source = (const char *) data.Scan0;

		//zalezi na usporadani radku
		if ( data.Stride >= 0 )
		{
			//radky jsou ulozeny odshora dolu
			for ( int y = 0 ; y < height ; y++ ) // pro kazdy radek
			{			
				memcpy( &m_data[ y * m_width * 3 ], &m_source[ y * pitch ], rowSize );
			}
		}
		else
		{
			//radky jsou ulozeny odspoda nahoru
			DWORD offsetSrc = 0;
			for ( int y = (height - 1)  ; y >= 0 ; y-- ) // pro kazdy radek
			{	
				memcpy( &m_data[ y * m_width * 3 ], &m_source[ offsetSrc ], rowSize );

				offsetSrc += pitch;
			}
		}

		bmpTemp->UnlockBits( &data );
		delete bmpTemp;

		GdiplusShutdown(gdiplusToken);

		return true;
	}
	else
	{
		//nepodarilo se zamknout bitmapu
		delete bmpTemp;

		GdiplusShutdown(gdiplusToken);

		return false;
	}
}

/** Velikost obrazku v bajtech.
*/
DWORD TCameraImage::GetDataSize(void)
{
	return this->GetWidth() * this->GetHeight() * 3;
}

/** Ulozi data do pripraveneho bufferu.
*
*	\param	dest	[in out] buffer pro data obrazku o velikosti minimalne TCameraImage::GetDataSize
*/
void TCameraImage::GetData(void * dest)
{
	memcpy( dest, m_data, this->GetDataSize() );
}

/** Vrati sirku obrazku v pixelech.
*/
int TCameraImage::GetWidth(void)
{
	return m_width;
}

/** Vrati vysku obrazku v pixelech.
*/
int TCameraImage::GetHeight(void)
{
	return m_height;
}

/** Zrusi tento objekt.
*/
void TCameraImage::Destroy(void)
{
	delete this;
}

// TCameraImage
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////