/*
*	This file contains class that load images from specified video sequence.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/


#include "StdAfx.h"
#include <tchar.h>
#include <strsafe.h>
#include <cstdlib>
#include "VideoDS.h"


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TDSSampler

/** Vytvori filter s jednou referenci.
*
*	\param	unk		[in] povinne predavane rozhrani pro bazovou tridu CBaseVideoRenderer - muze byt NULL
*	\param	hr		[in] promenna pro ulozeni vysledku
*/
TDSSampler::TDSSampler( IUnknown* unk, HRESULT *hr ):
		CBaseVideoRenderer(__uuidof(CLSID_Sampler), NAME("RGB Sampler"), unk, hr)
{	
	this->AddRef();

	m_data1 = NULL;
	m_data2 = NULL;
};

/** Destruktor.
*
*	Uvolni vsechny prostredky.
*/
TDSSampler::~TDSSampler()
{
	delete [] m_data1;
	delete [] m_data2;
}

/** Overeni typu pripojeneho datoveho zdroje.
*
*	\param	media	[in] parametry vstupniho datoveho zdroje zaslane DirectShow grafem.
*/
HRESULT TDSSampler::CheckMediaType( const CMediaType* media ) 
{    
	VIDEOINFO* vi; 
	
	if (
		! IsEqualGUID( *media->Subtype(), MEDIASUBTYPE_RGB24) ||	// musi byt RGB format
		! ( vi = (VIDEOINFO *)media->Format() )						// musi se existovat platna hlavicka
	) 
	{
		//podminka nesplnena => filtr nelze pripojit
		return E_FAIL;
	}

	//ulozime informace o datovem formatu
	m_info = vi->bmiHeader;	

	m_rgb_width		= m_info.biWidth;
	m_rgb_height	= abs(m_info.biHeight);
	m_rgb_size		= m_rgb_width * m_rgb_height * 3;

	m_rowsFliped = m_info.biHeight >= 0;

	//vypocitame zarovnovaci bajty - tj. delku celeho radku v puvodnim obraze
	int tmp_width = m_rgb_width * 3;	//sirka v bajtech
	if ( (tmp_width % 4) > 0 )
	{
		m_img_align = 4 - ( tmp_width % 4 );
	}
	else
	{
		m_img_align = 0;
	}
	m_img_stride = tmp_width + m_img_align;

	//vytvorime pole pro vysledky
	delete [] m_data1;
	m_data1 = NULL;
	delete [] m_data2;
	m_data2 = NULL;

	m_data1 = new BYTE[m_rgb_size];
	m_data2 = new BYTE[m_rgb_size];

	return  S_OK;
}

/** Ziskani dalsiho snimku z datoveho toku.
*
*	Metoda je volana grafem DirectShow.
*
*	\param	sample	[in] dalsi ziskany vzorek
*/
HRESULT TDSSampler::DoRenderSample(IMediaSample *sample)
{
	//zapiseme data do druheho snimku
	BYTE * src = NULL;
	sample->GetPointer( &src );

	//kopirujeme radek po radku
	//musime zvolit kopirovani podle poradi radku
	DWORD offsetSrc = 0;
	DWORD offsetDst = 0;
	if ( m_rowsFliped )
	{
		//radky odspoda nahoru
		for ( int r = (m_rgb_height - 1) ; r >= 0 ; r-- )
		{
			offsetDst = r * m_rgb_width * 3;

			for ( int c = 0 ; c < m_rgb_width ; c++ ) //sloupce
			{
				m_data2[offsetDst++] = src[offsetSrc++];	// B
				m_data2[offsetDst++] = src[offsetSrc++];	// G
				m_data2[offsetDst++] = src[offsetSrc++];	// R				
			}
			offsetSrc += m_img_align;
		}
	}
	else
	{
		//radky odshoda dolu
		for ( int r = 0 ; r < m_rgb_height ; r++ )
		{
			for ( int c = 0 ; c < m_rgb_width ; c++ ) //sloupce
			{
				m_data2[offsetDst++] = src[offsetSrc++];	// B
				m_data2[offsetDst++] = src[offsetSrc++];	// G
				m_data2[offsetDst++] = src[offsetSrc++];	// R				
			}
			offsetSrc += m_img_align;
		}
	}

	//vymenime ukazatele
	m_cs.Lock();
		BYTE * tmp = m_data1;
		m_data1 = m_data2;
		m_data2 = tmp;
	m_cs.Unlock();

	return  S_OK;
}

/** Metoda grafu DirectShow pro overeni co udelat se zadanym snimkem.
*
*	V tomto pripade se kazdy snimek bude spravne zobrazovat podle jeho zadaneho casu.
*	
*	\param	sample		[in] snimek o kterem se rozhoduje
*	\param	start		[in] cas pro pocatek zobrazeni snimku
*	\param	stop		[in] cas pro konec zobrazeni snimku
*/
HRESULT TDSSampler::ShouldDrawSampleNow(
	IMediaSample *sample,
	REFERENCE_TIME *start,
	REFERENCE_TIME *stop
	) 
{
	//zadny snimek nebude zahozen - kazdy snimek bude zobrazen podle jeho casu
	return S_FALSE;
}

/** Nakopiruje data do pripraveneho bufferu.
*
*	\param	data	[in out] buffer pro ulozeni dat ve formatu RGB 
*/
void TDSSampler::GetData( void * data )
{
	//posledni snimek je ulozen v M_DATA1
	m_cs.Lock();

		//prekopirujeme data
		memcpy( data, m_data1, m_rgb_size );

	m_cs.Unlock();
}

// TDSSampler
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/** Konstruktor.
*
*	Inicializuje vse na NULL.
*
*	Inicializace musi byt provedena metodou Initialize() !!!
*/
TVideoDSFile::TVideoDSFile(void)
{
	m_graph = NULL;
	m_ctrl = NULL;
	m_sampler = NULL;
	m_seek = NULL;
}

/** Destruktor.
*
*	Odstrani DirectShow graf a deinicializuje COM.
*/
TVideoDSFile::~TVideoDSFile(void)
{
	this->Destroy();
}

/** Inicializace DirectShow grafu.
*
*	Vraci TRUE pokud probehla inicializace uspesne nebo pokud byl jiz graf otevren.
*
*	\param	filename	[in] jmeno otviraneho souboru
*/
bool TVideoDSFile::Initialize( const char * filename )
{
	if ( ! m_graph )
	{
		//vytvorime novy graf

		//inicializace COM
		if ( FAILED( CoInitialize( 0 ) ) )
		{
			return false;
		}

		//vytvorime graf a ziskame jeho ovladani
		HRESULT hr = 
			CoCreateInstance( CLSID_FilterGraph, 0, CLSCTX_INPROC, IID_IGraphBuilder, (void **)  &m_graph );
		if ( FAILED( hr ) )
		{
			m_graph = NULL;
			CoUninitialize();
			return false;
		}

		//vytvorime graf a ziskame jeho ovladani
		hr = m_graph->QueryInterface( IID_IMediaControl, (void **) &m_ctrl );
		if ( FAILED( hr ) )
		{
			m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}

		//vytvorime sampler - ma jednu referenci ziskanou v kontruktoru
		m_sampler = new TDSSampler( 0, &hr );

		//ziskame zdroj videa
		size_t len =  mbstowcs( NULL, filename, strlen( filename ) + 1 );
		wchar_t * tmp_filename = new wchar_t[len + 1];
		mbstowcs( tmp_filename, filename, strlen( filename ) + 1 );
		
		IBaseFilter* video  = NULL; 
		hr = m_graph->AddSourceFilter ( tmp_filename, L"File Source", &video );
		delete [] tmp_filename;
		if ( FAILED( hr ) )
		{
			m_sampler->Release(); m_sampler = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}		

		//vystupni pin z videosignalu
		IPin* videoOut  = NULL; 
		//nejrpve pokus o nacteni bezneho videa
		hr = video->FindPin( L"Output", &videoOut );
		if ( FAILED( hr ) )
		{
			//zkusime nacist jako WMV nebo ASF
			hr = video->FindPin( L"Raw Video 1", &videoOut );
			if ( FAILED( hr ) )
			{
				video->Release();
				m_sampler->Release(); m_sampler = NULL;
				m_ctrl->Release(); m_ctrl = NULL;
				m_graph->Release(); m_graph = NULL;
				CoUninitialize();
				return false;
			}
		}		
		video->Release();

		//vstup sampleru
		IPin* samplerInput = NULL; 
		hr = m_sampler->FindPin( L"In", &samplerInput );
		if ( FAILED ( hr ) )
		{
			videoOut->Release();
			m_sampler->Release(); m_sampler = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}
		
		//sestavime graf
		hr = m_graph->AddFilter( (IBaseFilter*)m_sampler, L"Sampler" );
		if ( FAILED( hr ) )
		{
			samplerInput->Release();
			videoOut->Release();
			m_sampler->Release(); m_sampler = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}

		hr = m_graph->Connect( videoOut, samplerInput );
		if ( FAILED( hr ) )
		{
			samplerInput->Release();
			videoOut->Release();
			m_sampler->Release(); m_sampler = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}

		hr = m_graph->Render( videoOut );
		//TODO - zde je chyba v navratove hodnote
		//ackoliv vse funguje je vracena chyba
		/*
		if ( FAILED( hr ) )
		{
			samplerInput->Release();
			videoOut->Release();
			m_sampler->Release(); m_sampler = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}
		*/
		videoOut->Release();
		samplerInput->Release();

		//nyni by mel mit sampler informace o videoformatu
		{
			m_width = m_sampler->GetWidth();
			m_height = m_sampler->GetHeight();
		}

		//ovladani pozice
		hr = m_graph->QueryInterface ( IID_IMediaSeeking, (void **) &m_seek );
		if ( FAILED ( hr ) )
		{
			m_sampler->Release(); m_sampler = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}

		//zjisteni delky videa
		m_length = 0;
		m_seek->GetDuration( &m_length );

		//spustime graf
		hr = m_ctrl->Run();
		if ( FAILED( hr ) )
		{
			m_seek->Release();
			m_sampler->Release(); m_sampler = NULL;
			m_ctrl->Release(); m_ctrl = NULL;
			m_graph->Release(); m_graph = NULL;
			CoUninitialize();
			return false;
		}

		return true;
	}
	else
	{
		//graf uz existuje
		return true;
	}
}

/** Odstraneni aktualniho grafu (pokud existuje) a deinicializace COM.
*/
bool TVideoDSFile::Destroy(void)
{
	if ( m_graph )
	{
		//nejprve graf radeji zastavime
		m_ctrl->Stop();

		if ( m_seek )
		{
			m_seek->Release();
			m_seek = NULL;
		}

		//odstranime graf
		if ( m_sampler )
		{
			m_sampler->Release();
			m_sampler = NULL;
		}

		if ( m_ctrl )
		{
			m_ctrl->Release();
			m_ctrl = NULL;
		}

		m_graph->Release();
		m_graph = NULL;

		CoUninitialize();
	}

	return true;
}

/** Nakopiruje data ve formatu RGB z posledniho zachyceneho obrazku (pokud existuje)
*	do pripraveneho bufferu, jehoz velikost musi byt minimalne 
*
*		GetWidth() * GetHeight() * 3 bajty
*
*	\param	data	[in out] buffer pro pixelu ve formatu RGB
*/
void TVideoDSFile::GetDataRGB( void * data )
{
	m_sampler->GetData( data );

	//kontrola pozice a pripadne "pretoceni"
	LONGLONG pos;
	m_seek->GetCurrentPosition( &pos );
	if ( pos == m_length )
	{
		pos = 0;
		m_seek->SetPositions( &pos, AM_SEEKING_AbsolutePositioning, NULL, AM_SEEKING_NoPositioning );
	}
}

// TVideoDSFile
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
