/**
*	Main program.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*/

#define _WIN32_WINNT	0x0501
#define WINVER			0x0501

#include <windows.h>
#include <windowsx.h>
#include <gdiplus.h>
using namespace Gdiplus;

#include "pipeline/Dispatcher.h"
#include "pipeline/FrameQueue.h"
#include "pipeline/CameraThread.h"
#include "pipeline/CameraLoader.h"
#include "pipeline/UnitLoader.h"
using namespace NSPipeline;

#define	APPNAME		"Image Processing Pipeline Example"

//FORWARD DECLARATIONS
HWND InitWindow(void);
LRESULT CALLBACK MsgProc(HWND hWnd,UINT uMsg,WPARAM wParam,LPARAM lParam);

/** Zobrazeni dialogu s chybovym hlasenim.
*
*	\param	mess		[in] zobrazena zprava
*	\param	caption		[in] nadpis zpravy
*/
void ErrorMessage(const char * mess, const char * caption)
{
	MessageBox( NULL, mess, caption, MB_OK | MB_ICONERROR );
}

/** Upraveni velikosti okna tak, aby klientska oblast mela pozadovane rozmery.
*
*	\param	wnd		[in] menene okno
*	\param	width	[in] pozdovana sirka klientske oblasti
*	\param	height	[in] pozadovana vyska klientske oblasti
*/
void AdjustWindowSize(HWND wnd, int width, int height)
{
	DWORD style = WS_POPUP| WS_CAPTION| WS_SYSMENU;	

	RECT rc = {0,0,width,height};	
	BOOL res = AdjustWindowRect( &rc, style, FALSE );

	WINDOWPLACEMENT place;
	GetWindowPlacement( wnd, &place );
	SetWindowPlacement( wnd, &place );
}

/** Funkce pro zapis dat RGB obrazku do GDI+ bitmapy.
*
*	\param	bmp		[in] bitmapa GDI+ ve formatu RGB 24 bitu
*	\param	data	[in] data ve formatu FGB ulozena spojite
*/
void WriteToBitmap24bpp( Bitmap * bmp, const void * data )
{

#pragma pack(1)
	struct LocalPixel
	{
		BYTE red;
		BYTE green;
		BYTE blue;
	};
#pragma pack()

	Rect rect( 0,0, bmp->GetWidth(), bmp->GetHeight() );
	BitmapData bmp_data;

	if ( Ok == bmp->LockBits( &rect, ImageLockModeWrite, PixelFormat24bppRGB, &bmp_data) )
	{
		for ( UINT y = 0; y < bmp->GetHeight(); y++ )
		{
			LocalPixel * row = (LocalPixel*) &((char*)bmp_data.Scan0)[bmp_data.Stride * y];

			memcpy( row, &((const char*)data)[y * bmp->GetWidth() * 3], bmp->GetWidth() * 3 );
		}		

		bmp->UnlockBits( &bmp_data );
	}
}

/**	Hlavni smycka zprav.
*/
void MainLoop( HWND wnd )
{
	BOOL res;
	BOOL cont = TRUE;

	//INICIALIZACE RETEZCE

	//nejprve vytvorime staticke prvky (neobsahuji sve vypocetni vlakno) celeho retezce
	TFrameQueue * pipeline_queue = new TFrameQueue;
	TDispatcher * pipeline_dispatcher = new TDispatcher( pipeline_queue );


	//nyni musime vybrat zdroj obrazovych dat - nacteme tedy kameru ze zvolene DLL
	//musime nacist kameru
	TCameraLoader * pipeline_cameraLoader = new TCameraLoader;
	TCameraAbstract * pipeline_camera = 
		pipeline_cameraLoader->GetCamera_DirectShow( "../libs/CameraDirectShow.dll" );
	if ( ! pipeline_camera )
	{
		delete pipeline_cameraLoader;
		delete pipeline_queue;
		delete pipeline_dispatcher;
		ErrorMessage( "Nelze nacist kameru ze zadane DLL knihovny!", "Kamera nenalezena!" );
		return;
	}
	
	// nastavime spravnou velikost okna
	AdjustWindowSize( wnd, pipeline_camera->GetWidth(), pipeline_camera->GetHeight() );

	//nastavime informace o vlastnostech snimku do dispatcheru, ten bude tyto informace
	//poskytovat vypocetnim jednotkam
	pipeline_dispatcher->SetFramesInfo( pipeline_camera->GetWidth(), pipeline_camera->GetHeight() );

	//vytvorime vlakno pro kameru, ktery bude ziskavat snimky kazdych 25 milisekund (40 FPS)
	//a bude disponovat maximalne 100 snimky, ktere smi poslat do retezce.
	TCameraThread * pipeline_cameraThread = 
		new TCameraThread( pipeline_dispatcher, pipeline_camera, 25, 100 );

	//RETEZEC JE VYTVOREN - ZDE SE PRIDAVAJI JEDNOTKY

	//objekt pro nacteni jedne jednotky
	TUnitLoader pipeline_unit01_loader;
		
	//budeme nacitat jednotku, ktera neprijima zadne dodatecne parametry
	TUnitInterface * pipeline_unit01 =
	pipeline_unit01_loader.GetUnit_BASIC( "../units/AverageIntensity.dll", pipeline_dispatcher );
	if ( ! pipeline_camera )
	{
		delete pipeline_cameraThread;
		delete pipeline_cameraLoader;
		delete pipeline_queue;
		delete pipeline_dispatcher;
		ErrorMessage( "Nelze nacist vypocetni jednotku ze zadane DLL knihovny!", "Jednotka nenalezena!" );
		return;
	}

	//muzeme jednotku spustit
	pipeline_unit01->Start();	

	// ZDE SPUSTIME VLAKNO GENERUJICI SNIMKY
	pipeline_cameraThread->Start();

	//vytvorime si bitmapu pro vykreslovani pozadi
	Bitmap * bmp_bg = new Bitmap( pipeline_camera->GetWidth(), pipeline_camera->GetHeight(), PixelFormat24bppRGB );
	Bitmap * bmp_bb = new Bitmap( pipeline_camera->GetWidth(), pipeline_camera->GetHeight(), PixelFormat24bppRGB );
	
	//vykreslovani textu
	Font gdi_font(L"Arial", 16);	
	SolidBrush gdi_brush(Color(255, 255, 255, 255));

	//aktualni text
	wstring text = L"NO RESULT";

	//MAIN LOOP
	while ( cont )
	{
		MSG msg;

		//OBSLUHA ZPRAV
		while ( PeekMessage( &msg, NULL , 0, 0, PM_REMOVE) )
		{
			//UKONCENI APLIKACE
			if ( msg.message == WM_QUIT )
			{
				cont = FALSE;
				break;
			}

			//STISK KLAVESY
			if ( msg.message == WM_KEYDOWN )
			{
				// ukonceni aplikace
				if ( msg.wParam == VK_ESCAPE )
				{
					cont = FALSE;
				}
			}

			if ( msg.message == WM_PAINT )
			{
				break;
			}

			//vyridime tuto zpravu
			DispatchMessage( &msg );
		}		

		//aktualizujeme obrazek
		TFrameReal * frame = pipeline_queue->GetRenderableFrame();
		if ( frame )
		{
			//zapiseme snimek
			WriteToBitmap24bpp( bmp_bg, frame->GetImageSet()->GetRGB()->GetData() );

			//pokusime se ziskat vysledek vypoctu pro tento snimek
			TUnitRetTypeInterface * res = pipeline_unit01->GetResult( frame->GetTimestamp().GetID() );
			if ( res )
			{
				TUnitRetType_integer * res_int = (TUnitRetType_integer*) res;
				
				int value = res_int->GetValue();

				res->Release();

				wchar_t tmp[255];
				swprintf( tmp, L"Average intensity for FRAME with ID %u is %d", frame->GetTimestamp().GetID(), value );
				text = tmp;
			}

			frame->Release();
		}

		Graphics * go_bmp = Graphics::FromImage( bmp_bb );
		go_bmp->DrawImage( bmp_bg, 0, 0 );
		go_bmp->DrawString( text.c_str(), text.length(), &gdi_font, PointF(0,0), &gdi_brush );
		delete go_bmp;

		Graphics * go = Graphics::FromHWND( wnd );
		go->DrawImage( bmp_bb, 0, 0 );						
		delete go;

		Sleep( 10 );
	}

	// UKONCENI APLIKACE

	delete bmp_bb;
	delete bmp_bg;

	// NEJPRVE MUSIME UKONCIT VSECHNY JEDNOTKY
		
	pipeline_unit01->Stop();
	pipeline_unit01->Release();
	pipeline_unit01 = NULL;

	// NYNI ZRUSIME CELY VYPOCETNI RETEZEC

	//nejprve zastavime generovani novych snimku
	pipeline_cameraThread->Stop();

	//nyni musime uvolnit vsechny snimky z retezce
	pipeline_queue->Free();
	pipeline_dispatcher->Free();

	//odstranime cely system kamery
	delete pipeline_cameraThread;
	pipeline_camera->Destroy();
	pipeline_camera = NULL;
	delete pipeline_cameraLoader;

	//odstranime staticke slozky retezce
	delete pipeline_dispatcher;
	delete pipeline_queue;

	//APLIKACE JE SPRAVNE UKONCENA
}

/** Vstupni funkce programu.
*
* \param h_instance			[in] handle tohoto procesu
* \param h_prev_instance	[in] handle predchazejiciho procesu 
* \param cmdline			[in] parametry prikazove radky
* \param how_to_show		[in] definuje jak zobrazit okno
*
* \return 0 if program runs correctly
*
*/
int _stdcall WinMain(HINSTANCE h_instance,HINSTANCE h_prev_instance,LPSTR cmdline,int how_to_show)
{
	//inicializace GDI+
	GdiplusStartupInput gdiplusStartupInput;
	ULONG_PTR           gdiplusToken;
	GdiplusStartup(&gdiplusToken, &gdiplusStartupInput, NULL);

	//vytvoreni okna
	HWND wnd = InitWindow();
	ShowWindow( wnd, SW_NORMAL );

	// hlavni smycka programu
	MainLoop( wnd );

	//radne ukoncime GDI+
	GdiplusShutdown(gdiplusToken);

	return 0;
}

/** Inicializace okna.
*/
HWND InitWindow(void)
{
HWND hWnd;

	WNDCLASSEX twc = 
	{
			sizeof(WNDCLASSEX),
			CS_CLASSDC,
			MsgProc,
			0L,
			0L,
			GetModuleHandle(NULL),
			NULL,
			NULL,
			(HBRUSH)COLOR_WINDOW,
			NULL,
			APPNAME,
			NULL
	};


	if(!RegisterClassEx( &twc )) return NULL;

	DWORD style = WS_POPUP| WS_CAPTION| WS_SYSMENU;

	RECT rc = {0,0,640,480};
	BOOL res = AdjustWindowRect( &rc, style, FALSE );
	if ( !res )
	{
		return NULL;
	}	

    hWnd = 
		CreateWindow
		(
			APPNAME,
			APPNAME,
			style ,
			0,
			0,
			rc.right - rc.left,
			rc.bottom - rc.top,
			GetDesktopWindow(),
			NULL,
			twc.hInstance,
			NULL
		);

	return hWnd;
}

/** Funkce pro zpracovani aplikacnich zprav.
*/
LRESULT CALLBACK MsgProc(HWND hWnd,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
	switch ( uMsg )
	{
		case WM_DESTROY:
		{
			PostQuitMessage(0);
			return 0;
		}
		break;

		default:
			return DefWindowProc( hWnd, uMsg, wParam, lParam ); 
	}

	return DefWindowProc( hWnd, uMsg, wParam, lParam ); 
}