/*
*	This file contains class that represents thread with camera input.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/
#include <cassert>

#include "pipeline/CameraThread.h"

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TCameraThread

/** Staticka metoda pro beh vlakna kamery.
*
*	Tato metoda je spustena metodou start a v ramci jejiho provadeni dochazi ke generovani novych
*	snimku, ktere jsou zasilany dispatcheru.
*
*	Jedine spravne ukonceni vykonani teto metody je zavolani metody Stop objektu, ktery
*	je predan do tela teto metody jako parametr.
*
*	\param	ptr		[in] ukazatel na objekt typu TCameraThread reprezentujici
*					objekt zodpovedny za beh tohoto vlakna
*/
DWORD WINAPI TCameraThread::TCameraDLLRun(void* ptr)
{
	//ziskame ukazatel na kameru
	TCameraThread * cam = (TCameraThread*)ptr;

	//smycka bezi dokud neni vlakno ukonceno pres metodu stop
	while( cam->m_isRunning )
	{
		//tady ziskavame snimky a posilame je do dispacheru

		//ziskame volny snimek z alokatoru
		TImageSetReal * img = cam->m_manager->GetImageSet();
		if ( img )
		{
			//mame snimek

			//naplnime snimek daty
			cam->m_camera->GetData( img->GetRGBWrite()->GetDataWrite() );
			
			//TODO OPTIMIZE - here are 2 loops inside following methods

			//nyni musime vytvorit dalsi datove formaty
			img->GetGrayWrite()->FromRGB( img->GetRGBWrite() );
			img->GetARGBWrite()->FromRGB( img->GetRGBWrite() );

			//vytvorime obrazovy snimek a posleme do dispatcheru
			TFrameReal * frm = TFrameReal::Create(img);		
			cam->m_dispatcher->SetFrame(frm);
			//dispatcher si udrzuje vlastni referenci - proto tuto jiz muzeme uvolnit
			frm->Release();
		}
		else
		{
			//zadny snimek neni k dispozici - nic se nedeje, pouze se ceka
		}


		//cekame zadany casovy interval
		Sleep( cam->m_sleepTime );

		//TODO - mozna vypocitat zdrzeni ve funkci a upravit cekaci cas
	}

	return 0;
}
//OK 2007-09-04 21:29:20 B04-315B\Tom

/** Konstruktor.
*
*	Pripravi kameru k behu. Kameru je nutne spustit metodou Start().
*
*	\param	dispatcher		[in] ukazatel na platny dispatcher
*	\param	camera			[in] ukazatel na platnou kameru
*	\param	sleepTime		[in] prodleva mezi ziskanim dvou snimku v milisekundach
*	\param	reservedFrames	[in] pocet predalokovanych snimku v alokatoru snimku
*/
TCameraThread::TCameraThread( TDispatcher * dispatcher, TCameraAbstract * camera, DWORD sleepTime, DWORD reservedFrames )
{
	assert( dispatcher && camera );

	m_thread		= NULL;		
	m_isRunning		= false;	

	m_camera		= camera;		
	m_sleepTime		= sleepTime;	

	m_dispatcher	= dispatcher;

	//vytvorime manager snimku
	if ( 0 == reservedFrames ) reservedFrames = 1;	//chceme alespon 1 snimek
	m_manager		= new TImageSetManager( (DWORD)camera->GetWidth(), (DWORD)camera->GetHeight(), reservedFrames ) ;		
}
//OK 2007-08-25 15:25:29 B04-315B\Tom

/** Zruseni objektu.
*
*	\warning	Destruktor znici manager snimku. Je tedy dulezite, aby se v aplikaci
*				v dobe volani teto metody nevyskytovaly zadne snimky.
*				Pred zrusenim by tedy kamera mela byt zastavena metodou Stop().
*/
TCameraThread::~TCameraThread(void)
{
	if(m_thread)
	{
		//kamera bezi - musime zastavit
		Stop();
	}

	//odstranime manager snimku
	delete m_manager;
}
//OK 2007-08-25 15:25:51 B04-315B\Tom

/** Spusteni kamery.
*
*	Metoda vraci TRUE pokud byla kamera uspesne spustena nebo jiz bezi.
*/
BOOL TCameraThread::Start(void)
{
	if( ! m_thread )
	{
		//kamera nebezi

		//spustime vypocetni vlakno
		m_thread = new TThread;
		m_isRunning = true;

		BOOL res = m_thread->Run((FUNC_PTR)TCameraDLLRun,(void*)this);

		if(res)
		{
			return TRUE;
		}
		else
		{
			m_isRunning = false;
			delete m_thread;
			m_thread = NULL;

			return FALSE;
		}
	}
	else
	{
		//jednotka uz bezi
		return TRUE;
	}
}
//OK 2007-08-25 15:26:03 B04-315B\Tom

/** Zastaveni kamery.
*
*	Metoda znici vypocetni vlakno.
*
*	Vraci TRUE pokud bylo vlakno uspesne zastaveno.
*/
BOOL TCameraThread::Stop(void)
{
	if( m_thread )
	{
		//kamera bezi - nastavime stav pro zastaveni
		m_isRunning = false;

		//cekame na ukonceni vlakna
		while( ! m_thread->IsTerminated() )
		{
			Sleep(20);
		};		

		//znicime vlakno
		//nyni jednotka stoji - ukoncime ji
		m_thread->Finish();
		delete m_thread;
		m_thread = NULL;

		return TRUE;
	}
	else
	{
		//vlakno neexistuje
		return TRUE;
	}
}
//OK 2007-08-25 15:26:09 B04-315B\Tom

// TCameraThread
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
