/*
*	This file contains class that represents queue of frames that comes from
*	dispatcher and wait for displaying in renderer.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/
#include <cassert>

#include "pipeline/FrameQueue.h"

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TFrameQueue

/** Konstruktor.
*
*	Vytvori frontu s vnitrnim bufferem s delkou 0.
*	Snimky tedy lze zobrazit okamzite pote co nemaji zadny zamek.
*/
TFrameQueue::TFrameQueue(void)
{
	m_length = 0;
}
//OK 2007-08-25 12:07:17 B04-315B\Tom

/** Destruktor.
*
*	Odstrani vsechny snimky z fronty - tj. na kazdy snimek zavola Release().
*/
TFrameQueue::~TFrameQueue(void)
{
	m_critical.Enter();

	while( ! m_frames.empty() )
	{
		TFrameReal * f = m_frames.front();
		if ( f ) f->Release();
		m_frames.pop();
	}

	m_critical.Leave();
}
//OK 2007-08-25 12:09:13 B04-315B\Tom

/** Prida snimek do fronty. Zvysi pocet referenci na tento snimek.
*
*	Fronta udrzuje pouze referenci na snimek, nikdy neudrzuje zamek.
*
*	\note	Metoda by mela byt volana z dispatcheru.
*
*	\param	frame	[in] pridavany snimek, fronta zvysi pocet referenci na tento snimek o 1
*/
void TFrameQueue::AddFrame( TFrameReal * frame )
{
	m_critical.Enter();

		//pridame snimek do vnitrniho bufferu
		frame->AddRefs();
		m_buffer.push( frame );

		//pokud vstupni buffer presahl svou velikost pak presuneme snimky do
		//vystupni fronty
		while( m_buffer.size() > m_length )
		{
			m_frames.push( m_buffer.front() );
			m_buffer.pop();
		}

	m_critical.Leave();
}
//OK 2007-08-25 12:09:41 B04-315B\Tom

/** Vrati snimek pripraveny pro zobrazeni. Tento snimek je odstranen z fronty.
*
*	Muze vratit NULL pokud neni zadny snimek pripraven ke zobrazeni.
*	To se muze stat v pripade ze je fronta prazdna nebo pokud je snimek zamceny.
*
*	\warning	Volajici objekt je zodpovedny za vraceny objekt (referenci)!!!
*/
TFrameReal* TFrameQueue::GetRenderableFrame(void)
{
	TFrameReal * frame = NULL;

	m_critical.Enter();
		//pokud je fronta prazdna pak se nic nedeje
		if( m_frames.empty() ) 
		{
			frame = NULL;
		}
		else
		{
			//zjistime zda se na zacatku ve fronte nachazi renderovatelny snimek - nema zamky
			frame = m_frames.front();

			if( frame->GetLockCount() == 0 )
			{
				//snimek nema zamky - nebudem pridavat reference, pouze odstranime z fronty a vratime
				m_frames.pop();
			}
			else
			{
				//objekt ma zamky - neni mozne jej vyzvednout z fronty
				frame = NULL;
			}
		}
	m_critical.Leave();

	return frame;
}
//OK 2007-08-25 12:10:07 B04-315B\Tom

/** Vrati maximalni velikost vnitrniho bufferu.
*/
DWORD TFrameQueue::GetInputBufferLength(void) const
{
	return m_length;
}

/** Nastavi novou velikost pro vnitrni buffer.
*
*	Pokud je velikost mensi nez aktualni velikost bufferu pak jsou polozky presunuty do vystupni fronty.
*
*	\param	length		[in] nova velikost vnitrniho bufferu
*/
void TFrameQueue::SetInputBufferLength( DWORD length )
{
	m_critical.Enter();
	
	while ( length < m_buffer.size() )
	{
		//buffer obsahuje vic polozek nez je jeho nova delka - presunujeme do vystupni fronty		
		m_frames.push( m_buffer.front() );
		m_buffer.pop();
	}

	//nova delka vnitrniho bufferu
	m_length = length;

	m_critical.Leave();
}

/** Metoda odstrani z fronty vsechny snimky.
*
*	Metoda by mela byt volana na konci aplikace pred odstranenim objektu.
*/
void TFrameQueue::Free(void)
{
	m_critical.Enter();
		while( ! m_frames.empty() )
		{
			TFrameReal * f = m_frames.front();
			if ( f ) f->Release();
			m_frames.pop();
		}

		while ( ! m_buffer.empty() )
		{
			TFrameReal * f = m_buffer.front();
			if ( f ) f->Release();
			m_buffer.pop();
		}
	m_critical.Leave();
}
//OK 2007-08-25 12:10:37 B04-315B\Tom

// TFrameQueue
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
