/*
*	This file contains abstract (and others) classes from which are derived classes
*	representing image returned from camera.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/

#include <windows.h>

//FORWARD DECLARATIONS
namespace NSPipeline
{
	struct TPF_ARGB;

	class TImage;
	class TImageSet;
	class TFrame;
};

#ifndef _PIPELINE_IMAGEABSTRACT_HH_
#define _PIPELINE_IMAGEABSTRACT_HH_

#include "pipeline/TimeStamp.h"

namespace NSPipeline
{

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/** Tento enumerator definuje typ obrazku - typ pixelu uvnitr obrazku.
*/
enum EnumImageType
{
	ENUM_IMAGETYPE_GRAY		=	0,			///< obraz ve stupnich sedi - 8 bitu na pixel
	
	ENUM_IMAGETYPE_ARGB		=	1,			///< barevny obraz, 32 bitu na pixel
											///< pro little-endian
											///< poradi bajtu v pameti B,G,R,A
											///< jako DWORD 0xAARRGGBB

	ENUM_IMAGETYPE_RGB		=	2,			///< barevny obraz, 24 bitu
											///< pro little-endian
											///< poradi bajtu v pameti B,G,R											

	ENUM_IMAGETYPE_FORCED32 = 0x7fffffff,	///< zarovnani na 32-bitu
};

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImage

/** Tato abstraktni trida reprezentuje objekty pro obrazova data.
*
*	V aplikaci bude dostupna pouze tato deklarace.
*
*	Odvozene tridy musi reimplementovat vsechny virtualni metody.
*
*	Objekty tridy nelze kopirovat!
*/
class TImage
{
//PUBLIC METHODS
public:
	virtual EnumImageType	GetImageType(void) const = 0;	///< typ tohoto obrazu (typ pixelu)
	virtual	DWORD			GetPixelSize(void) const = 0;	///< velikost pixelu v bajtech

	virtual DWORD			GetWidth(void) const = 0;		///< sirka v pixelech
	virtual DWORD			GetHeight(void) const = 0;		///< vyska v pixelech
	virtual DWORD			GetPitch(void) const = 0;		///< delka radku v bajtech

	virtual DWORD			GetDataSize(void) const = 0;	///< velikost obrazovych dat v bajtech
	virtual const void*		GetData(void) const = 0;		///< ukazatel na obrazova data

//PROTECTED METHODS
protected:
	virtual					~TImage(void) = 0 {};			///< virtualni destruktor

							TImage(void) {};				///< defaultni konstruktor
					
//PROTECTED FAKE METHODS
protected:
							TImage( const TImage & orig );		///< falesny kopirovaci konstruktor
	void					operator=( const TImage & orig );	///< falesny operator prirazeni
};
//OK 2007-08-24 23:18:18 B04-315B\Tom

// TImage
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImageSet

/** Tato trida reprezentuje rozhrani pro tridu, ktera obsahuje ruzne verze jednoho 
*	obrazu ulozene jako objekty typu TImage a odvozenych trid.
*/
class TImageSet
{
//PUBLIC METHODS
public:
	virtual const TImage*	GetARGB(void) const = 0;	///< obrazek ve formatu ARGB - 32 bitu na pixel
	virtual const TImage*	GetGray(void) const = 0;	///< obrazek ve stupnich sedi - 8 bitu na pixel
	virtual const TImage*	GetRGB(void) const = 0;		///< obrazek ve formatu RGB - 24 bitu na pixel

//PROTECTED METHODS
protected:
	virtual					~TImageSet(void) = 0 {};		///< virtualni destruktor

							TImageSet(void) {};				///< defaultni konstruktor

//PROTECTED FAKE METHODS
protected:
							TImageSet( const TImageSet & orig );	///< falesny kopirovaci konstruktor
	void					operator=( const TImageSet & orig );	///< falesny operator prirazeni
};
//OK 2007-08-24 23:49:22 B04-315B\Tom

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TFrame

/** Tato trida reprezentuje snimek ziskany z kamery.
*
*	Snimek muze obsahovat vice obrazu (v ruznych formatech), ktere jsou dostupne pres metodu
*	TFrame::GetImageSet().
*
*	Na kazdy snimek je aplikovano pocitani referenci a zamku.
*	Pokud dosahne pocet referenci 0 pak je snimek odstranen a nesmi byt jiz dale pouzit.
*	
*	Zamek se pouziva ve vypocetni jednotce v pripade, ze jednotka provadi vypocet, jehoz
*	vypocet je nutny pri zobrazeni snimku. V takovem pripade ziska jednotka snimek
*	se zamkem (z dispatcheru). Snimek s nenulovym poctem snimku nesmi opustit frontu snimku
*	a tedy je zajisteno, ze se zobrazi az v okamziku, kdy bude kompletne zpracovan vsemi 
*	jednotkami, ktere nad nim udrzuji zamek.
*	
*/
class TFrame
{
//PUBLIC METHODS
public:
	virtual const TImageSet*	GetImageSet(void) const = 0;	///< objekt s obrazy

	virtual const TTimeStamp &	GetTimestamp(void) const = 0;	///< casove razitko

	virtual void				AddRefs(void) = 0;				///< zvyseni poctu referenci
	virtual void				Release(void) = 0;				///< snizeni poctu referenci
	virtual DWORD				GetRefs(void) const = 0	;		///< pocet referenci na snimek

	virtual DWORD				AddLock(void) = 0;				///< pridani zamku
	virtual DWORD				ReleaseLock(void) = 0;			///< uvolneni zamku
	virtual DWORD				GetLockCount(void) const = 0;	///< pocet zamku na snimku

//PROTECTED METHODS
protected:
	virtual						~TFrame(void) = 0 {};			///< virtualni destruktor
								TFrame(void) {};				///< defaultni konstruktor

//PROTECTED FAKE METHODS
protected:
							TFrame( const TFrame & orig );		///< falesny kopirovaci konstruktor
	void					operator=( const TFrame & orig );	///< falesny operator prirazeni
};
//OK 2007-08-24 23:49:30 B04-315B\Tom

// TFrame
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/** Rozhrani dispatcheru dostupne vypocetnim jednotkam.
*
*	Dispatcher obsahuje informace o velikosti poskytovanych snimku.
*	Tato informace je uzitecna pri inicializaci vypocetnich jednotek, ktere takovou
*	informaci mohou vyuzit, protoze se predpoklada, ze za behu aplikace se tato velikost nebude menit.
*/
class TDispatcherInterface
{
//PUBLIC METHODS
public:
	virtual DWORD			GetWidth(void) = 0;			///< informace o sirce vraceneho snimku
	virtual DWORD			GetHeight(void) = 0;		///< informace o vysce vraceneho snimku

	virtual TFrame *		GetFrame(void) = 0;			///< vrati snimek s pridanou referenci
	virtual TFrame *		GetLockedFrame(void) = 0;	///< vrati snimek s referenci a zamkem

//PROTECTED METHODS
protected:
	virtual					~TDispatcherInterface(void) = 0 {};		///< virtualni destruktor
							TDispatcherInterface(void) {};			///< defaultni konstruktor

//PROTECTED FAKE METHODS
protected:
							TDispatcherInterface( const TDispatcherInterface & orig );	///< falesny kopirovaci konstruktor
	void					operator=( const TDispatcherInterface & orig );				///< falesny operator prirazeni
};
//OK 2007-08-24 23:50:47 B04-315B\Tom

// TFrame
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

}; //END of NSPipeline

#endif