/*
*	This file contains structures for simple reading and writing from/to raw image data.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/

#include <windows.h>

//FORWARD DECLARATIONS
namespace NSPipeline
{
	struct TPF_ARGB;
	struct TPF_RGB;
};

#ifndef _PIPELINE_IMAGEPIXELFORMATS_HH_
#define _PIPELINE_IMAGEPIXELFORMATS_HH_

#include "pipeline/CriticalSection.h"
#include "pipeline/TimeStamp.h"

namespace NSPipeline
{

// Vsechny struktury jsou vnitrne zarovnany na 1 bajt
#pragma pack(1)

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TPF_ARGB

/** Barevny format - 32bitu - ARGB.
*
*	Pixel je reprezentovan jako DWORD 0xAARRGGBB.
*
*	V pameti jsou (pro little-endian) ulozeny bajtu v tomto poradi BLUE, GREEN, RED, ALPHA.
*
*	\note	Struktura je zarovnana na 1 byte, takze je zajistena shodna velikost s DWORD (32-bitu).
*/
struct TPF_ARGB
{
//PUBLIC METHODS
public:
				TPF_ARGB( BYTE alpha, BYTE red, BYTE  green, BYTE blue);

	void		SetAlpha( BYTE alpha );
	void		SetRed( BYTE red );
	void		SetGreen( BYTE green );
	void		SetBlue( BYTE blue );

	BYTE		GetAlpha(void) const;
	BYTE		GetRed(void) const;
	BYTE		GetGreen(void) const;
	BYTE		GetBlue(void) const;
//PUBLIC COMPONENTS
public:
	DWORD		m_pixel;
};
//OK 2007-08-24 22:27:16 B04-315B\Tom

/** Vytvori pixel ze zadanych barevnych slozek
*
*	\param	alpha	[in] alpha kanal - 8 bitu
*	\param	red		[in] cerveny kanal - 8 bitu
*	\param	green	[in] zeleny kanal - 8 bitu
*	\param	blue	[in] modry kanal - 8 bitu
*/
inline TPF_ARGB::TPF_ARGB( BYTE alpha, BYTE red, BYTE  green, BYTE blue)
{
	m_pixel =
		(	( (DWORD)alpha	)	<< 24	) |
		(	( (DWORD)red	)	<< 16	) |
		(	( (DWORD)green	)	<< 8	) |
		(	( (DWORD)blue	)			);
}
//OK 2007-08-24 22:27:26 B04-315B\Tom

/** Nastaveni alpha kanalu.
*
*	\param	alpha	[in] nova hodnota alpha kanalu
*/
inline void TPF_ARGB::SetAlpha( BYTE alpha )
{
	m_pixel = ( ( (DWORD)alpha ) << 24 ) | m_pixel;
}
//OK 2007-08-24 22:27:28 B04-315B\Tom

/** Nastaveni cerveneho kanalu.
*
*	\param	red	[in] nova hodnota cerveneho kanalu
*/
inline void TPF_ARGB::SetRed( BYTE red )
{
	m_pixel = ( ( (DWORD)red ) << 16 ) | m_pixel;
}
//OK 2007-08-24 22:28:16 B04-315B\Tom

/** Nastaveni zeleneho kanalu.
*
*	\param	green	[in] nova hodnota zeleneho kanalu
*/
inline void TPF_ARGB::SetGreen( BYTE green )
{
	m_pixel = ( ( (DWORD)green ) << 8 ) | m_pixel;
}
//OK 2007-08-24 22:28:19 B04-315B\Tom

/** Nastaveni modreho kanalu.
*
*	\param	blue	[in] nova hodnota modreho kanalu
*/
inline void TPF_ARGB::SetBlue( BYTE blue )
{
	m_pixel = ( (DWORD)blue ) | m_pixel;
}
//OK 2007-08-24 22:28:22 B04-315B\Tom

/** Vrati hodnotu alpha kanalu.
*/
inline BYTE TPF_ARGB::GetAlpha(void) const
{
	return (BYTE)( ( m_pixel >> 24 ) & 0xff );
}
//OK 2007-08-24 22:28:27 B04-315B\Tom

/** Vrati hodnotu cerveneho kanalu.
*/
inline BYTE TPF_ARGB::GetRed(void) const
{
	return (BYTE)( ( m_pixel >> 16 ) & 0xff );
}
//OK 2007-08-24 22:28:29 B04-315B\Tom

/** Vrati hodnotu zeleneho kanalu.
*/
inline BYTE TPF_ARGB::GetGreen(void) const
{
	return (BYTE)( ( m_pixel >> 8 ) & 0xff );
}
//OK 2007-08-24 22:28:32 B04-315B\Tom

/** Vrati hodnotu modreho kanalu.
*/
inline BYTE TPF_ARGB::GetBlue(void) const
{
	return (BYTE)( m_pixel & 0xff );
}
//OK 2007-08-24 22:28:34 B04-315B\Tom

// TPF_ARGB
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TPF_RGB

/** Barevny format - 24bitu - RGB.
*
*	V pameti jsou (pro little-endian) ulozeny bajtu v tomto poradi BLUE, GREEN, RED.
*
*	\note	Struktura je zarovnana na 1 byte, takze je zajistena shodna velikost s 24 bity.
*/
struct TPF_RGB
{
//PUBLIC METHODS
public:
				TPF_RGB( BYTE red, BYTE  green, BYTE blue);
	
	void		SetRed( BYTE red );
	void		SetGreen( BYTE green );
	void		SetBlue( BYTE blue );
	
	BYTE		GetRed(void) const;
	BYTE		GetGreen(void) const;
	BYTE		GetBlue(void) const;
//PUBLIC COMPONENTS
public:
	BYTE		m_blue;
	BYTE		m_green;
	BYTE		m_red;
};
//OK 2007-09-04 11:56:37 B04-315B\Tom

/** Vytvori pixel ze zadanych barevnych slozek
*
*	\param	red		[in] cerveny kanal - 8 bitu
*	\param	green	[in] zeleny kanal - 8 bitu
*	\param	blue	[in] modry kanal - 8 bitu
*/
inline TPF_RGB::TPF_RGB( BYTE red, BYTE  green, BYTE blue)
{
	m_blue	= blue;
	m_green	= green;
	m_red	= red;
}
//OK 2007-09-04 11:57:45 B04-315B\Tom

/** Nastaveni cerveneho kanalu.
*
*	\param	red	[in] nova hodnota cerveneho kanalu
*/
inline void TPF_RGB::SetRed( BYTE red )
{
	m_red = red;
}
//OK 2007-09-04 11:58:06 B04-315B\Tom

/** Nastaveni zeleneho kanalu.
*
*	\param	green	[in] nova hodnota zeleneho kanalu
*/
inline void TPF_RGB::SetGreen( BYTE green )
{
	m_green = green;
}
//OK 2007-09-04 11:58:20 B04-315B\Tom

/** Nastaveni modreho kanalu.
*
*	\param	blue	[in] nova hodnota modreho kanalu
*/
inline void TPF_RGB::SetBlue( BYTE blue )
{
	m_blue = blue;
}
//OK 2007-09-04 11:58:33 B04-315B\Tom

/** Vrati hodnotu cerveneho kanalu.
*/
inline BYTE TPF_RGB::GetRed(void) const
{
	return m_red;
}
//OK 2007-09-04 11:58:43 B04-315B\Tom

/** Vrati hodnotu zeleneho kanalu.
*/
inline BYTE TPF_RGB::GetGreen(void) const
{
	return m_green;
}
//OK 2007-09-04 11:58:58 B04-315B\Tom

/** Vrati hodnotu modreho kanalu.
*/
inline BYTE TPF_RGB::GetBlue(void) const
{
	return m_blue;
}
//OK 2007-09-04 11:59:21 B04-315B\Tom

// TPF_RGB
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

#pragma pack()

}; //END of NSPipeline
using namespace NSPipeline;

#endif
