/*
*	This file contains simple working unit from that may be simply derived others units.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/

#include <windows.h>

// FORWARD DECLARATIONS
namespace NSPipeline
{
	class TSimpleUnit;
	class TSimpleUnitProcessingInterface;
};

#ifndef _PIPELINE_SIMPLEUNIT_HH_
#define _PIPELINE_SIMPLEUNIT_HH_

#include "pipeline/Unit.h"
#include "pipeline/Thread.h"
#include "pipeline/CriticalSection.h"

namespace NSPipeline
{

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TSimpleUnitProcessingInterface

/** Toto je rozhrani pro objekty, ktere zajistuji vypocet v ramci jednotky TSimpleUnit.
*
*	STACI ODVODIT NOVOU TRIDU OD TOHOTO ROZHRANI A PREPSAT METODY PROCESSFRAME() A DESTRUKTOR.
*
*	METODY GETTYPE() A GETRESULT() BYT POVINNE PREPSANY NEMUSI, ALE VETSINOU JE TO NUTNE. IMPLICITNE
*	TOTIZ TYTO METODY REPREZENTUJI JEDNOTKU, KTERA NEVRACI ZADNA DATA.
*
*	Metoda ProcessFrame je pracovni metodou, ktera je automaticky volana tridou TSimpleUnit.
*	V teto metode musi byt zpracovan zadany snimek TFrame. Metoda se nemusi starat o reference 
*	ani zamky, pouze zpracuje snimek. Metoda musi probehnout naprosto bez chyb, pokud dojde
*	k nejakemu selhani pak musi byt zpracovano uvnitr metody a navenek se nesmi projevit - tedy
*	nejsou povoleny zadne vyjimky apod.
*
*	Dale je nutne prepsat metody GetType() a GetResult().
*	Pokud tyto dve nejsou prepsany, pak trida nevraci zadna data a metoda
*	GetResult vraci vzdy NULL.
*
*	Tato trida (a z ni odvozene) je vzdy odpovedna za objekt, ktery je vracen metodou
*	GetResult(). 
*/
class TSimpleUnitProcessingInterface
{
//PUBLIC ABSTRACT METHODS
public:
	/** Tato metoda obdrzi snimek, zpracuje jej a ukonci se.
	*	O jeji volani se stara logika uvnitr tridy TSimpleUnit.
	*/
	virtual void						ProcessFrame( const TFrame * frame ) = 0;

	/** Virtualni destruktor.
	*/
	virtual								~TSimpleUnitProcessingInterface(void) = 0 {};

	/** Datovy typ ukladany v teto tride.
	*
	*	Imlicitni implementace predpokladany, ze jendotka nevraci zadna data.
	*/
	virtual EnumUnitType				GetType(void) const { return ENUM_UNITTYPE_NODATA; };

	/** Ukazatel na vracena data pro snimek se zadanym ID.
	*
	*	Metoda muze vratit NULL. V bazove tride vraci NULL vzdy.
	*
	*	Tato metoda by mela byt synchronizovana pro soubezny pristup vice vlakne, nebot
	*	jedno vlakno (z vypocetni jednotky) do objektu zapisuje data a v takovem pripade
	*	musi metoda GetResult() pockat na jejich zapis.
	*
	*	Implpicitni navratova hodnota je NULL.
	*/
	virtual TUnitRetTypeInterface*		GetResult( DWORD id ) { return NULL; };
};
//OK 2007-08-25 18:31:09 B04-315B\Tom

// TSimpleUnitProcessingInterface
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TSimpleUnit

/** Tato trida reprezentuje jednotku, ktera zpracovava
*	data vzdy pouze z jednoho snimku.
*
* POUZITI:
*
*	(A)		JEDNOTKA KTERA NEVRACI ZADNA DATA
*
*		(1) Vytvorit tridu pro vypocet nad snimkem odvozenou od TSimpleUnitProcessingInterface
*			s prepsanou metodou ProcessFrame(). Takova trida nevraci zadna data.
*		(2) Pouzit objekt teto tridy primo v kontruktoru tridy TSimpleUnit.
*
*	(B)		JEDNOTKA VRACEJICI DATA
*
*		(1) Vytvorit tridu pro vypocet nad snimkem odvozenou od TSimpleUnitProcessingInterface.
*			Reimplementovat VSECHNY jeji metody. To zahrnuje vybrani (pripadne nadefinovani noveho)
*			navratoveho datoveho typu.
*		(2) Pouzit objekt teto tridy primo v kontruktoru tridy TSimpleUnit.
*/
class TSimpleUnit: public TUnitInterface
{
enum EnumUnitState;

//PUBLIC OVERRIDEN METHODS
public:
	/** Typ teto jednotky a zaroven typ navratove hodnoty
	*/
	virtual EnumUnitType			GetType(void) { return m_unit->GetType(); };

	/** Vysledek vraceny jednotkou.
	*/
	virtual TUnitRetTypeInterface*	GetResult(DWORD id) { return m_unit->GetResult( id ); }

	/** Pocet snimku, ktere nebyly zpracovany mezi poslednimi dvema zpracovanymi snimky.
	*/
	virtual DWORD					GetFrameInterval() { return m_interval; };

	virtual BOOL					Start(void);
	virtual BOOL					Stop(void);		
											
	virtual void					Release(void);

//PUBLIC METHODS
public:
	TSimpleUnit( TDispatcherInterface* dispatcher, TSimpleUnitProcessingInterface * unit, BOOL lockFrame);

//PROTECTED METHODS
protected:
	~TSimpleUnit(void);

//PRIVATE METHODS
private:
	static DWORD WINAPI		TSimpleUnitThread(void* ptr);


	EnumUnitState			GetState(void) const;
	void					SetState(EnumUnitState state);

	void					Loop(void);

//PROTECTED FAKE METHODS
protected:
							TSimpleUnit( const TSimpleUnit & orig );	///< falesny kopirovaci konstruktor
	void					operator=( const TSimpleUnit & orig );		///< falesny prirazovaci operator

//PRIVATE STRUCTURES
private:
	/** Enumerator s moznymi stavy stavoveho automatu uvnitr jednotky.
	*/
	enum EnumUnitState
	{
		ENUM_UNIT_WORK,			///< jednotka zpracovava snimek, po dokonceni prejde 
								///< do stavu ENUM_UNIT_LOADFRAME
								///< z tohoto stavu muze jednotka prejit take do stavu 
								///< ENUM_PROC_FINISHWORK aby bylo	mozne ukoncit/prerusit jeji cinnost
		ENUM_UNIT_LOADFRAME,	///< jednotka je pripravena ziskat dalsi snimek, po ziskani snimku 
								///< prejde do ENUM_UNIT_WORK
								///< z tohoto stavu muze jednotka prejit take do stavu 
								///< ENUM_UNIT_FINISHWORK aby bylo mozne ukoncit/prerusit jeji cinnost
								///< toto je zaroven pocatecni stav jednotky po spusteni
		ENUM_UNIT_FINISHWORK,	///< jednotka zpracovava snimek, po dokonceni zastavi svou cinnost
								///< prechodem do ENUM_UNIT_STOP
		ENUM_UNIT_STOP			///< signalizace zastaveni jednotky
	};

//PRIVATE COMPONENTS
private:
	BOOL							m_lockFrame;		///< urcuje zda tato jednotka zamyka snimky

	volatile DWORD					m_interval;			///< pocet snimku, ktere jednotka nestihla
														///< zpracovat pred zpracovanim posledniho snimku

	EnumUnitState					m_state;			///< aktualni stav jednotky
	TCriticalSection				m_critical;			///< kriticka sekce pro zmenu stavu jednotky

	TThread*						m_thread;			///< vlakno bezici v jednotce 
														///< pokud je NULL pak jednotka nebezi

	TDispatcherInterface*			m_dispatcher;		///< rozhrani k dispatcheru snimku

	TFrame*							m_frame;			///< prave ulozeny snimek
														///< je nad nim drzena reference
														///< a take zamek pokud jednotka zamyka snimky

	TSimpleUnitProcessingInterface*	m_unit;				///< objekt pro vlastni vypocty
};
//OK 2007-08-25 18:42:08 B04-315B\Tom

/** Vrati aktualni stav stavoveho automatu ktery ridi vypocet.
*/
inline TSimpleUnit::EnumUnitState TSimpleUnit::GetState(void) const
{
	return m_state;
}
//OK 2007-08-25 18:42:17 B04-315B\Tom

/** Nastavi novy stav stavoveho automatu, ktery ridi vypocet.
*
*	\param	state	[in] novy stav
*/
inline void TSimpleUnit::SetState( TSimpleUnit::EnumUnitState state)
{
	m_state = state;
}
//OK 2007-08-25 18:42:19 B04-315B\Tom

// TSimpleUnit
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

}; //end of NSPipeline
using namespace NSPipeline;

#endif