/**
 * OpenCV library extension of FFMPEG
 *
 * Author: Petr Chmelar (c) 2008 
 * License: GNU/GPL and Intel CV
 */

#include <opencv/cv.h>
#include "cvffmpeg.h"

int __pixel_type = PIX_FMT_BGR24;	//PIX_FMT_BGR32;

/**
 * Constructor
 */
ffMedia* newMedia() {
    ffMedia* media = (ffMedia *)malloc((1)*sizeof(ffMedia)); //alloc(ffMedia, 1);
    if (media == NULL) return NULL;
    
    media->formatCtx = NULL;
    media->videoStream = -1;
    media->codecCtx = NULL;
    media->codec = NULL;
    
    media->ffEof = 1;

    media->ffFrame = NULL;
    
#ifdef SWSCALE_SWSCALE_H
    media->sws = NULL;
#endif // SWS
    
    return media;
};


/**
 * Closes a media file
 */
void ffClose(ffMedia* media) {
    if (media != NULL) {
        media->ffEof = 1;

        #ifdef SWSCALE_SWSCALE_H
            if (media->sws != NULL) {
                ffSWSClose(media->sws);
                free(media->sws);
                media->sws = NULL;
            }
        #endif
        
        // Free the YUV frame
        if (media->ffFrame != NULL) av_free(media->ffFrame);
        media->ffFrame = NULL;

        // Close the codec
        if (media->codecCtx != NULL) {
            avcodec_close(media->codecCtx);
            media->codecCtx = NULL;
        }

        // Close the video file
        if (media->formatCtx != NULL) {
            av_close_input_file(media->formatCtx);
            media->formatCtx = NULL;
        }

        // Cancel the video stream
        media->videoStream = -1;
    }
};

void ffRelease( ffMedia** media )
{
	if( !media ) return;
	ffClose( *media );
	free( *media );
	*media = NULL;
	return;
}

/**
 * Opens a media file and returns true if succeed
 */
int ffOpenFile(ffMedia* media, const char* filename) {
    int i;      // just i
    
    // Check the media structure
    if (media == NULL) {
        return 0;
    }
    
    // Register all formats and codecs
    av_register_all();
    
    // Open video file
    if(av_open_input_file(&(media->formatCtx), filename, NULL, 0, NULL) != 0)
        return 0; // Couldn't open file
    
    // Retrieve stream information
    if(av_find_stream_info(media->formatCtx) < 0)
        return 0; // Couldn't find stream information
    
    // Dump information about file onto standard error
    dump_format(media->formatCtx, 0, filename, 0);
    
    // Find the first video stream
    for (i=0; i < (int)media->formatCtx->nb_streams; i++)
        if(media->formatCtx->streams[i]->codec->codec_type == CODEC_TYPE_VIDEO) {
            media->videoStream = i;
            break;
        }
    if (media->videoStream == -1) return 0; // Didn't find a video stream
    
    // Get a pointer to the codec context for the video stream
    media->codecCtx = media->formatCtx->streams[media->videoStream]->codec;
    
    // Find the decoder for the video stream
    media->codec = avcodec_find_decoder(media->codecCtx->codec_id);
    if(media->codec == NULL) return 0; // Codec not found
    
    // Open codec
    if(avcodec_open(media->codecCtx, media->codec) < 0) return 0; // Could not open codec
    
    // Hack to correct wrong frame rates that seem to be generated by some codecs
    if(media->codecCtx->time_base.den > 1000 && media->codecCtx->time_base.num == 1)
        media->codecCtx->time_base.num = 1000;
    
    // Allocate video frame to ffFrame
    media->ffFrame = avcodec_alloc_frame();
    if(media->ffFrame == NULL) return 0;

    ffAvFrame(media);

//#ifdef SWSCALE_SWSCALE_H

    media->sws = ffSWSInit(media->codecCtx->width, media->codecCtx->height, __pixel_type, SWS_FAST_BILINEAR);
//#endif // SWS
    
    return 1;
};


/*
Frame Grabbing (Video4Linux and IEEE1394)
Toru Tamaki sent me some sample code that demonstrates how to grab frames from a Video4Linux or IEEE1394 video source using libavformat / libavcodec. For Video4Linux, the call to av_open_input_file() should be modified as follows:

    // TODO:
    int ffOpenCapture();
    int ffOpenIEEE1349();


AVFormatParameters formatParams;
AVInputFormat *iformat;

formatParams.device = "/dev/video0";
formatParams.channel = 0;
formatParams.standard = "ntsc";
formatParams.width = 640;
formatParams.height = 480;
formatParams.frame_rate = 29;
formatParams.frame_rate_base = 1;
filename = "";
iformat = av_find_input_format("video4linux");

av_open_input_file(&ffmpegFormatContext,
                 filename, iformat, 0, &formatParams);

For IEEE1394, call av_open_input_file() like this:

AVFormatParameters formatParams;
AVInputFormat *iformat;

formatParams.device = "/dev/dv1394";
filename = "";
iformat = av_find_input_format("dv1394");

av_open_input_file(&ffmpegFormatContext,
                 filename, iformat, 0, &formatParams);

*/



/**
 * Return the next frame or NULL if finished
 */
AVFrame* ffAvFrame(ffMedia* media) {
    AVPacket    packet;
    int     frameFinished;
    
    media->ffEof = 1;
    
    // Read frames
    while(av_read_frame(media->formatCtx, &packet) >= 0) {
        // Is this a packet from the video stream?
        if(packet.stream_index == media->videoStream) {
            // Decode video frame
            avcodec_decode_video2( media->codecCtx, media->ffFrame, &frameFinished, &packet );
			//avcodec_decode_video( media->codecCtx, media->ffFrame, &frameFinished, packet.data, packet.size );
            // Free the packet that was allocated by av_read_frame
            av_free_packet(&packet);
            
            // Did we get a video frame?
            if(frameFinished) {
                if(media->ffFrame->data == NULL) return NULL;
                
                media->ffEof = 0;
                return media->ffFrame;
            }
        }
        // Process audio etc...
        else {
            // Free the packet that was allocated by av_read_frame
            av_free_packet(&packet);
        }
        
    }
    
    return NULL;
};


/**
 * Returns the actual position
 */
unsigned long ffPosition(ffMedia* media) {
    return media->codecCtx->frame_number;
    // return (unsigned long)floor(0.5 + ((((double)media->formatCtx->timestamp / (double)AV_TIME_BASE) / (double)av_q2d(media->codecCtx->time_base))));
};


/**
 * Returns the ESTIMATED number of frames in the video (NOT EXACT!)
 */
unsigned long ffLength(ffMedia* media) {

    return (unsigned long)floor(0.5 +((((double)media->formatCtx->duration / (double)AV_TIME_BASE) / (double)av_q2d(media->codecCtx->time_base))));
};

/**
 * Seek to a specified position and return the frame
 */
AVFrame* ffAvFrameSeek(ffMedia* media, unsigned long position) {
    int64_t     timestamp;
    //int64_t     MyPts;
    
    //AVPacket    packet;
    //int     frameFinished;
    
    
    // Conversion of int64_t position value into unsigned long timestamp for seeking using av_seek_frame        
    timestamp = (int64_t)floor(0.5 + (((double)position * (double)AV_TIME_BASE * (double)av_q2d(media->codecCtx->time_base))));

    // check the frame number
    printf("%-15u /%-15lld", position, timestamp);


    if (av_seek_frame(media->formatCtx, -1, timestamp , AVSEEK_FLAG_ANY) < 0) return NULL;
    else {
        // the loop here was useless and it was deleted
        avcodec_flush_buffers(media->codecCtx);
        media->codecCtx->frame_number = position;        
        return ffAvFrame(media);
    }

};

/**
 * Seek to a specified position and return the frame
 */
AVFrame* ffAvFramePreciseSeek(ffMedia* media, unsigned long position) {
    // if actual position
    if (position == ffPosition(media)) return media->ffFrame;
    // if previous
    if (position < ffPosition(media)) {
        if (av_seek_frame(media->formatCtx, -1, 1 , AVSEEK_FLAG_BACKWARD) < 0) return NULL;
        else {
            avcodec_flush_buffers(media->codecCtx);
            media->codecCtx->frame_number = 0;            
        }
    }
    
    // go to the specified position (next)
    media->codecCtx->hurry_up = 1;
    while((long)ffPosition(media) < (long)(position-1) && !media->ffEof) {    // just position-1
        ffAvFrame(media);
		//printf( "PreciseSeek: % 10ld / % 10ld  \r", ffPosition(media), position );
    };
    media->codecCtx->hurry_up = 0;
    
    return ffAvFrame(media);                                    // ^ because of this
};

#ifdef SWSCALE_SWSCALE_H

// SWS-init function
ffSWS * ffSWSInit(int w, int h, int f, int t) {
    
    ffSWS * sws = (ffSWS *)malloc(sizeof(ffSWS)); 
    if (sws == NULL) return NULL;
    
    sws->width = w;
    sws->height = h; 
    sws->format = f;
    sws->flag = t;
    
    sws->imgConvertCtx = NULL;
    
    // Allocate video frame to swsFrame
    sws->swsFrame = avcodec_alloc_frame();
    if(sws->swsFrame == NULL) return 0;
    
    // Determine required buffer size and allocate buffer
    sws->numBytes = avpicture_get_size(f, w, h);
    sws->buffer = (uint8_t *)malloc((sws->numBytes)*sizeof(uint8_t));
    
    // Assign appropriate parts of buffer to image planes in media->swsFrame
    avpicture_fill((AVPicture *)sws->swsFrame, sws->buffer, f, w, h);

#ifdef _CV_H_    
    // You can use OpenCV frame only in case of PIX_FMT_RGBA32
    if (f == __pixel_type) {
        // create cvimage header
        sws->cvFrame = cvCreateImageHeader( cvSize(w, h), 8, 3 );
    }
    else sws->cvFrame = NULL;
#endif // CV
    
    return sws;

};

// SWS-delete function
void ffSWSClose(ffSWS* sws) {

    sws->width = 0;
    sws->height = 0;
    sws->format = 0;
    sws->flag = 0;

    sws->numBytes = 0;

    #ifdef _CV_H_
    // Release cvFrame header
    if (sws->cvFrame != NULL) {
        cvReleaseImageHeader(&sws->cvFrame);
        sws->cvFrame = NULL;
    }
    #endif // CV

    // Free the SWS frame            
    if (sws->swsFrame != NULL) {
        av_free(sws->swsFrame);
        sws->swsFrame = NULL;
    }

    if (sws->buffer != NULL) {
        free(sws->buffer);
        sws->buffer = NULL;
    }

    if (sws->imgConvertCtx != NULL) {
        sws_freeContext(sws->imgConvertCtx);
        sws->imgConvertCtx = NULL;                    
    }

};


/**
 * Convert the actual frame to the IplImage or return NULL if none
 */
AVFrame* ffConvert(ffMedia* media, ffSWS* sws) {
    // check the frame
    if(media->ffFrame == NULL || media->ffFrame->data == NULL) {
        return NULL;
    }
    
    if (sws == NULL) sws = media->sws;
/*
    if (width != media->codecCtx->width || height != media->codecCtx->width) {
        if (width == 0 || height == 0) {
            width = media->codecCtx->width;
            height = media->codecCtx->height;
        } 
        else { // fuck off the old SWS
            ffSWSInit(media, width, height, format);
        }
    }
*/    
    // free the convert context
    if (sws->imgConvertCtx != NULL) sws_freeContext(sws->imgConvertCtx);

    // create it
    sws->imgConvertCtx = sws_getContext(media->codecCtx->width,
            media->codecCtx->height, media->codecCtx->pix_fmt,
            sws->width, sws->height, sws->format, sws->flag, NULL, NULL, NULL);   // SWS_BICUBIC

    // transform the image
    sws_scale(sws->imgConvertCtx, media->ffFrame->data, media->ffFrame->linesize, 0,
            media->codecCtx->height, sws->swsFrame->data, sws->swsFrame->linesize);

#ifdef _CV_H_
    if (sws->cvFrame != NULL && sws->format == __pixel_type ) {
        // set data to the cvFrame
        cvSetData(sws->cvFrame, sws->swsFrame->data[0], sws->swsFrame->linesize[0]);
    }
#endif
    
    return media->sws->swsFrame;
};




#ifdef _CV_H_
/**
 * Return the next frame or NULL if finished
 */
IplImage* ffCvFrame(ffMedia* media) {
    // get next frame in YUV
    ffAvFrame(media);
    // return the IplImage
    return ffCvConvert(media, NULL);
};

/**
 * Seek to a specified position and return the frame
 */
IplImage* ffCvFrameSeek(ffMedia* media, unsigned long position) {
    // get next frame in YUV
    ffAvFrameSeek(media, position);
    // return the IplImage
    return ffCvConvert(media, NULL);
};

/**
 * Seek to a specified position (precisely but slowly) and return the frame
 */
IplImage* ffCvFramePreciseSeek(ffMedia* media, unsigned long position) {
    // get next frame in YUV
    ffAvFramePreciseSeek(media, position);
    // return the IplImage
    return ffCvConvert(media, NULL);
};


/**
 * Convert the actual frame to the IplImage or return NULL if none
 */
IplImage* ffCvConvert(ffMedia* media, ffSWS* sws) {
    // convert the YUV frame
    ffConvert(media, sws);
    // and return the IplImage
    return media->sws->cvFrame;
};
#endif // CV
#endif // SWS





