//==============================================================================
/** \file
 * Video Processing Toolkit (VPL)						\n
 * Copyright (c) 2005-2008 by Vita Beran				\n
 *														\n
 * Author: Vita Beran, beranv@fit.vutbr.cz				\n
 * File:    vplgllfgraddist.h							\n
 * Section: lowLevelFeatures							\n
 * Date:    2008/12/03									\n
 *
 * $Id: vplpca.h 8 2008-12-11 00:54:29Z beranv $
 * 
 * \brief Global Low-level Feature extraction based on Image Gradient Distribution.
 */

#ifndef _VPL_GLLF_GRADDIST_H_
#define _VPL_GLLF_GRADDIST_H_

#include <stdio.h>
#include <opencv/cv.h>
#include "vplcore.h"

#ifdef __cplusplus
extern "C" {
#endif

/** \ingroup gr_imgftrs
	\defgroup gr_gllfGradDist Gradient Distribution
	@{

	\struct VplGllfGradDist
	\brief Class implementing Global Low-level Feature extraction based on Image Gradient Distribution. 
	The Extractor builds the image pyramid, computes gradient orientations and values and 
	calculates the weighted histogram of gradient orientations per each pyramid level.

*/
typedef struct VplGllfGradDist
{
	CvMat * dx;
	CvMat * dy;
	CvMat * mag;
	CvMat * ori;
	CvMat * frame;
	CvMat * subframe;

	CvMat * hist;
	int bins;
	int pyrs;
	int cells;
	int aperture_size;

	float gridOverlap;
	int scale_0;
} 
VplGllfGradDist;



/** \fn VplGllfGradDist * vplGllfGradDistCreate( void )
	\brief GradDist object constructor.
*/
struct VplGllfGradDist * vplGllfGradDistCreate( void );


/** \fn VplGllfGradDist * vplGllfGradDistInit( struct VplGllfGradDist * gd, int pyrs, int bins, int aperture_size )
	\brief Initialize GradDist object's structure. 
	@param gd GradDist object handler (if null, new object is constructed)
	@param pyrs amount of pyramid levels
	@param bins amount of gradient orientation bins
	@param aperture_size sobel filter aperture size
*/
struct VplGllfGradDist * vplGllfGradDistInit( struct VplGllfGradDist * gd, int pyrs, int bins, float gridOverlap, int aperture_size );



/** \fn int vplGllfGradDistGetFeature( struct VplGllfGradDist * gd, CvMat * feature )
	\brief Copy the last extracted feature vector.
	@param gd GradDist object handler.
	@param feature output vector where the descriptor is stored
*/
int	vplGllfGradDistGetFeature( struct VplGllfGradDist * gd, CvMat * feature );


double vplGllfGradDistDist( struct VplGllfGradDist * gd, CvMat * f1, CvMat * f2 );


/** \fn int vplGllfGradDistGetSize( struct VplGllfGradDist * gd )
	\brief Return the length of feature vector.
	@param gd GradDist object handle 
*/
int	vplGllfGradDistGetSize( struct VplGllfGradDist * gd );



/** \fn int vplGllfGradDistExtract( struct VplGllfGradDist * gd, CvArr * image, CvMat * feature )
	\brief Extracts the feature from input array of image planes.
	@param gd GradDist object handler.
	@param image an input grayscaled image
	@param feature output vector where the descriptor can be stored
*/
int	vplGllfGradDistExtract( struct VplGllfGradDist * gd, CvArr * image, CvMat * feature CV_DEFAULT( NULL ) );



/** \fn int vplGllfGradDistRender( struct VplGllfGradDist * gd, CvArr * canvas )
	\brief Renders the feature values to the canvas.
	@param gd a pointer to the GradDist object
	@param canvas an image to render the feature vector
*/
int	vplGllfGradDistRender( struct VplGllfGradDist * gd, CvArr * canvas );



/** \fn int vplGllfGradDistRelease( struct VplGllfGradDist ** gd )
	\brief Release the GradDist object.
	@param gd a pointer to the GradDist object 
*/
int vplGllfGradDistRelease( struct VplGllfGradDist ** gd );

/** @} */



typedef struct cImagePyr
{
	CvMat ** layers;
	int octaves, levels;
	int mattype;
} cImagePyr;
#define imagePyrGet( ip, o, l ) (ip)->layers[(ip)->levels*(o)+(l)]

struct cImagePyr * imagePyrCreate( int octaves, int levels, int mattype );
int imagePyrInit( struct cImagePyr * imagePyr, CvArr * src );
void imagePyrRelease( struct cImagePyr ** imagePyr );

struct cImagePyr * imagePyrSobel( struct cImagePyr * imagePyr, CvArr * image, int pyrCount, int aperture_size );

CvMat * gradWeighted( CvMat * magnitude, CvMat * orientation, CvRect wnd, CvMat * feature, CvMat * filter, CvMat * filterBuff );


#ifdef __cplusplus
}
#endif

#endif			//  _VPL_GLLF_GRADDIST_H_
