#ifndef _VPL_TIMIMG_H_
#define _VPL_TIMIMG_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <opencv/cv.h>


typedef struct VplTimeMeasure
{
	int64 t0, t1;

	int64 eTime;
	int64 beginTime;
	unsigned int count;
	unsigned int all;
	double aTime;
} 
VplTimeMeasure;



static void vplTimeMeasureInit( VplTimeMeasure * pt, int countAll CV_DEFAULT(0) ) 
{ 
	if( !pt ) return;
	pt->t0 = cvGetTickCount(); 
	pt->t1 = pt->t0;

	pt->count = 0;
	pt->eTime = 0;
	pt->beginTime = 0;
	pt->all = countAll;
	pt->aTime = 0.0;
}

static void vplTimeMeasureCountInc( VplTimeMeasure * pt )
{
	if( !pt ) return;
	pt->count++;
	if( pt->all > 0 )
		pt->aTime = 0.000001 * pt->all * pt->eTime / ( cvGetTickFrequency() * pt->count );
}

static void vplTimeMeasureStart( VplTimeMeasure * pt )
{
	if( !pt ) return;
	pt->beginTime = cvGetTickCount();
}

static void vplTimeMeasureEnd( VplTimeMeasure * pt )
{
	if( !pt ) return;
	if( pt->beginTime ) pt->eTime += ( cvGetTickCount() - pt->beginTime );
	vplTimeMeasureCountInc( pt );
}

static void vplTimeMeasureTick( VplTimeMeasure * pt )
{
	int64 tmp = cvGetTickCount();
	if( !pt ) return;
	pt->eTime += ( tmp - pt->beginTime );
	pt->beginTime = tmp;
}

static void vplTimeMeasureAdd( VplTimeMeasure * pt )
{
	int64 tmp = cvGetTickCount();
	if( !pt ) return;
	pt->eTime += ( tmp - pt->beginTime );
	pt->beginTime = tmp;
	vplTimeMeasureCountInc( pt );
}

static void vplTimeMeasureGetMS( VplTimeMeasure * pt, double * et, double * ft, double * at )
{
	double _et, _ft, _at;

	if( !pt ) return;

	_et = 0.000001 * pt->eTime / cvGetTickFrequency();			// ellapsed time
	_ft = ( pt->count ? _et / pt->count : _et );				// tick rate
	//_at = pt->all * _ft;										// overall time - assumption
	_at = pt->aTime;

	if( et ) *et = _et;
	if( ft ) *ft = _ft;
	if( at ) *at = _at;
}

static void vplTimeMeasurePrint( FILE * f, VplTimeMeasure * pt )		
{
	double et, ft, at;
	vplTimeMeasureGetMS( pt, &et, &ft, &at );
	if( at > 0 )
		fprintf( f, "% 6.2f%% [%f->%f]; ", 100.*et/at, et, at-et );	
	else
		fprintf( f, "%f", et );	
}

static double vplTimeMeasureET( VplTimeMeasure * pt )		
{
	double t;
	vplTimeMeasureGetMS( pt, &t, NULL, NULL );
	return t;
}

static double vplTimeMeasureFT( VplTimeMeasure * pt )		
{
	double t;
	vplTimeMeasureGetMS( pt, NULL, &t, NULL );
	return t;
}

static unsigned int vplTimeMeasureGet( VplTimeMeasure * pt )
{
	if( !pt ) return 0;
	return (unsigned int)( ( cvGetTickCount() - pt->t0 )  / cvGetTickFrequency() );
}




#define _pts_count( pts ) ((int *)(pts))
#define _pts_elems( pts ) (struct VplTimeMeasure *)(_pts_count(pts)+1)
static struct VplTimeMeasure * vplTimeMeasuresCreate( int count )
{
	struct VplTimeMeasure * pts = NULL;
	int i;

	if( count <= 0 ) return pts;
	
	pts = (struct VplTimeMeasure *)malloc( sizeof( int ) + sizeof( struct VplTimeMeasure ) * count );
	*_pts_count( pts ) = count;
	
	for( i = 0; i < count; ++i )
		vplTimeMeasureInit( _pts_elems(pts) + i, 0 );

	return pts;
}

static void vplTimeMeasuresInit( VplTimeMeasure * pts, int i, int countAll CV_DEFAULT(0) )
{
	if( !pts || ( i >= *_pts_count( pts ) ) ) return;
	vplTimeMeasureInit( _pts_elems( pts ) + i, countAll );
}

static void vplTimeMeasuresStart( struct VplTimeMeasure * pts, int i )
{
	if( !pts || ( i >= *_pts_count( pts ) ) ) return;
	vplTimeMeasureStart( _pts_elems( pts ) + i );
}

static void vplTimeMeasuresEnd( struct VplTimeMeasure * pts, int i )
{
	if( !pts || ( i >= *_pts_count( pts ) ) ) return;
	vplTimeMeasureEnd( _pts_elems( pts ) + i );
}

static void vplTimeMeasuresTick( VplTimeMeasure * pts, int i )
{
	if( !pts || ( i >= *_pts_count( pts ) ) ) return;
	vplTimeMeasureTick( _pts_elems( pts ) + i );
}

static void vplTimeMeasuresCountInc( VplTimeMeasure * pts, int i )
{
	if( !pts || ( i >= *_pts_count( pts ) ) ) return;
	vplTimeMeasureCountInc( _pts_elems( pts ) + i );
}

static void vplTimeMeasuresAdd( VplTimeMeasure * pts, int i )
{
	if( !pts || ( i >= *_pts_count( pts ) ) ) return;
	vplTimeMeasureAdd( _pts_elems( pts ) + i );
}

static void vplTimeMeasuresGetMS( VplTimeMeasure * pts, int i, double * et, double * ft, double * at )
{
	if( !pts || ( i >= *_pts_count( pts ) ) ) return;
	vplTimeMeasureGetMS( _pts_elems( pts ) + i, et, ft, at );
}

static void vplTimeMeasuresPrint( FILE * f, VplTimeMeasure * pts, int i )
{
	int k;
	if( !pts || ( i >= *_pts_count( pts ) ) ) return;
	if( i >= 0 )
		vplTimeMeasurePrint( f, _pts_elems( pts ) + i );
	else
		for( k = 0; k < *_pts_count( pts ); ++k )
		{
			vplTimeMeasurePrint( stdout, _pts_elems( pts ) + k ); 
			printf( "\n" );
		}
}



static void vplTimeMeasuresRelease( struct VplTimeMeasure ** pts )
{
	if( !pts || !*pts ) return;
	free( *pts );
	*pts = NULL;
	return;
}

static void vplTimeMeasureDrawProgress( CvMat * canvas, VplTimeMeasure * pts, int i, CvScalar color, CvFont * font, char * name )
{
	char buff[256];
	double et, ft, at, data;
	int pos;
	CvScalar bCol;
	if( !canvas || !pts ) return;

	cvRectangle( canvas, cvPoint( 0, 0 ), cvPoint( canvas->cols-1, canvas->rows-1 ), CV_RGB(0,0,0), CV_FILLED, 8, 0 );
	vplTimeMeasuresGetMS( pts, i, &et, &ft, &at );
	at = MAX( 1., at );
	data = et/at;
	pos = cvRound(data*canvas->cols);

	bCol = cvScalarAll( 0.5+0.5*(pow( 2.0, data )-1.0) );
	cvRectangle( canvas, cvPoint( 0, 0 ), cvPoint( pos, canvas->rows-1 ), 
		cvScalar( bCol.val[0]*color.val[0], bCol.val[1]*color.val[1], bCol.val[2]*color.val[2], bCol.val[3]*color.val[3] ),
		CV_FILLED, 8, 0 );
	cvLine( canvas, cvPoint( pos, 0 ), cvPoint( pos, canvas->rows-1 ), color, 1, 8, 0 );
	if( font )
	{
		CvSize ts;
		if( name ) sprintf( buff, "%s: %6.2f%%", name, 100.0*data );
		else	   sprintf( buff, "%6.2f%%", 100.0*data );
		cvGetTextSize( buff, font, &ts, NULL );
		cvPutText( canvas, buff, cvPoint( (canvas->cols-ts.width)/2, (canvas->rows+ts.height)/2 ), font, CV_RGB(255, 255, 255 ) );
	}
	cvRectangle( canvas, cvPoint( 0, 0 ), cvPoint( canvas->cols-1, canvas->rows-1 ), CV_RGB(255, 255, 255 ), 1, 8, 0 );

	return;
}

//
//static void drawBarN( CvMat * canvas, int count, double * data, CvScalar color, CvFont * font, char * title )
//{
//	char buff[32];
//	int i, h;
//	if( !canvas ) return;
//	cvRectangle( canvas, cvPoint( 0, 0 ), cvPoint( canvas->cols-1, canvas->rows-1 ), CV_RGB(0,0,0), CV_FILLED, 8, 0 );
//	h = (int)( 1.0*canvas->rows/ count );
//	for( i = 0; i < count; ++i )
//	{
//		CvScalar bCol = cvScalarAll( 2.0*(pow( 2.0, 1.0*(i+1)/count )-1.0) );
//		cvRectangle( canvas, cvPoint( 0, i*h ), cvPoint( (int)(data[i]*canvas->cols), (i+1)*h-1 ), 
//			cvScalar( bCol.val[0]*color.val[0], bCol.val[1]*color.val[1], bCol.val[2]*color.val[2], bCol.val[3]*color.val[3] ),
//			CV_FILLED, 8, 0 );
//		if( font )
//		{
//			CvSize ts;
//			int bl;
//			sprintf( buff, "%6.2f%%", 100.0*data[i] );
//			cvGetTextSize( buff, font, &ts, &bl );
//			cvPutText( canvas, buff, cvPoint( 100-ts.width, i*h+(h+ts.height)/2 ), font, CV_RGB(255, 255, 255 ) );
//		}
//	}
//	if( font && title )
//	{
//		CvSize ts;
//		cvGetTextSize( title, font, &ts, NULL );
//		cvPutText( canvas, title, cvPoint( 100+20, (canvas->rows + ts.height)/2 ), font, CV_RGB(255, 255, 255 ) );
//	}
//	cvRectangle( canvas, cvPoint( 0, 0 ), cvPoint( canvas->cols-1, canvas->rows-1 ), CV_RGB(255, 255, 255 ), 1, 8, 0 );
//
//	return;
//}
//
//static void showTimesLog( CvMat * canvas, VplTimeMeasure * pts, CvFont * font, char ** titles )
//{
//	int border = 5;
//	int i, h;
//	CvMat submat;
//	double sum, tim;
//
//	if( !canvas || !pts  ) return;
//
//	sum = 0.0; 
//	for( i = 0; i < *_pts_count( pts ); ++i )
//	{
//		vplTimeMeasureGetMS( _pts_elems( pts ) + i, 0, &tim, 0 );
//		sum += tim;
//	}
//
//	h = (int)(1.0*canvas->rows/ *_pts_count( pts ));
//
//	for( i = 0; i < *_pts_count( pts ); ++i )
//	{
//		vplTimeMeasureGetMS( _pts_elems( pts ) + i, 0, 0, &tim, 0 );
//		tim /= sum;
//		drawBarN( cvGetSubRect( canvas, &submat, cvRect( border, h*i+border, canvas->cols-2*border, h-2*border ) ), 1, &tim, CV_RGB( 40, 40, 200 ), font, ( titles ? titles[i] : NULL ) );
//	}
//	return;
//}



#ifdef __cplusplus
}
#endif

#endif			//  _VPL_TIMIMG_H_



