//==============================================================================
/** \file vplcore.h
 * Video Processing Toolkit (VPL)						\n
 * Copyright (c) 2005-2008 by Vita Beran				\n
 *														\n
 * Author: Vita Beran, beranv@fit.vutbr.cz				\n
 * File:    vplcore.h									\n
 * Section: core										\n
 * Date:    2008/12/03									\n
 *
 * $Id: vplcore.h 66M 2009-03-18 11:41:18Z (local) $
 * 
 * \brief Useful VPL functions and structure.
 */

/** \mainpage

	VPL is the Video Processing Library used for image and video processing, result storage, features extraction and analysis, etc.

	The VPL uses OpenCV and PostgreSQL.


	Go to <a href="https://merlin.fit.vutbr.cz/wiki-graph/index.php/Video_Processing_Toolkit" target="blank">wiki pages</a> for comments, tutorials, examples, etc.

	\section sec_downloads Downloads
	- <a href="../ffmpeg_win32.zip">build FFMPEG for win32</a>
	- <a href="../vpl1-3.zip">VPL source</a>

*/


#ifndef _VPL_CORE_H_
#define _VPL_CORE_H_

#include <stdio.h>
#include <opencv/cxcore.h>

#ifdef __cplusplus
extern "C" {
#endif


/** @defgroup gr_core Core functions
    \brief Fundamental functions.
    @{
 */

/** \typedef typedef float vplDataType
	\brief Type used as default data type in VPL.
*/
typedef float vplDataType;


/** \def VPL_DATA_MAT_TYPE 
	\brief Type of CvMat used as default in VPL.
*/
#define VPL_DATA_MAT_TYPE CV_32FC1



/**	\struct CvMatPrintFormat
	\brief Structure for printing configuration. 
*/
typedef struct CvMatPrintFormat
{
	char _format[32], * format;			///< format of how to print one matrix element
	char separator[10];					///< matrix element separator
	FILE * output;						///< output stream (file, standard output, etc.)
} CvMatPrintFormat;



/**	\fn struct CvMatPrintFormat cvMatPrintFormat()
	\brief Default initialization of printing configuration structure.
*/
static struct CvMatPrintFormat cvMatPrintFormat( void ) 
{ 
	CvMatPrintFormat mpf; 
	mpf.format = NULL; 
	mpf.output = stdout; 
	sprintf( mpf.separator, "; " ); 
	return mpf; 
}


CvSeq * vplDbSeqFloatToInt( CvSeq * seq, double scale, double shift );



/**	Print a matrix using predefined format.
*/
void cvMatPrint( CvMat * mat, struct CvMatPrintFormat * matPrintFormat );


/** Check matrix validity in VPL scope.
*/
#define vplMatCheck( mat ) ( (mat) && CV_IS_MAT(mat) && CV_MAT_TYPE( (mat)->type ) == VPL_DATA_MAT_TYPE )


/** Check matrix validity and size.
*/
#define vplMatCheckSize( mat, r, c ) ( vplMatCheck( mat ) && (mat)->cols == (c) && (mat)->rows == (r) ) 


/** Create new matrix or and initialize the existing one.
*/
#define vplMatInit( mat, r, c ) { if( !mat ) (mat) = cvCreateMat( r, c, VPL_DATA_MAT_TYPE ); cvZero( mat ); }


#define vplMatVecElem( vec, i ) (vec)->data.fl[i]


/** \fn void vplColorsRand( CvScalar * colors, int count )
	\brief Generates array of random colors.
	\param colors	Array of colors (CvScalar values).
	\param count	Size of the array.
*/
void vplColorsRand( CvScalar * colors, int count );


CV_INLINE CvPoint2D32f cvPoint2D64fTo32f( CvPoint2D64f p64 ) 
{ 
	CvPoint2D32f p32;
	p32.x = (float)p64.x;
	p32.y = (float)p64.y;
	return p32;
}

CV_INLINE CvPoint2D64f cvPoint2D32fTo64f( CvPoint2D32f p32 ) 
{ 
	CvPoint2D64f p64;
	p64.x = (double)p32.x;
	p64.y = (double)p32.y;
	return p64;
}


CV_INLINE CvSize2D32f cvPoint2D64fToSize32f( CvPoint2D64f p64 ) 
{ 
	CvSize2D32f p32;
	p32.width  = (float)( floor( p64.x+0.5 ) );
	p32.height = (float)( floor( p64.y+0.5 ) );
	return p32;
}

CV_INLINE CvPoint2D64f cvSize2D32fToPoint64f( CvSize2D32f p32 ) 
{ 
	CvPoint2D64f p64;
	p64.x = (double)p32.width;
	p64.y = (double)p32.height;
	return p64;
}



CV_INLINE char * strtolow( char * str )
{
	int i = 0;
	if( !str ) return str;
	while( str[i] != '\0' )
	{
		if( str[i] >= 'A' && str[i] <= 'Z' ) str[i] += 32;
		i++;
	}
	return str;
}

/** @} */


#ifdef __cplusplus
}
#endif

#endif		// _VPL_CORE_H_

