//==============================================================================
/** \file
 * Video Processing Toolkit (VPL)						\n
 * Copyright (c) 2005-2008 by Vita Beran				\n
 *														\n
 * Author: Vita Beran, beranv@fit.vutbr.cz				\n
 * File:    vplgllf2dgridhist.h							\n
 * Section: lowLevelFeatures							\n
 * Date:    2008/12/03									\n
 *
 * $Id: vplpca.h 8 2008-12-11 00:54:29Z beranv $
 * 
 * \brief Global Low-level Feature extraction based on Image Histogram.
 */

#ifndef _VPL_GLLF_COLHIST_H_
#define _VPL_GLLF_COLHIST_H_

#include <stdio.h>
#include <opencv/cv.h>
#include "vplcore.h"

#ifdef __cplusplus
extern "C" {
#endif

/** \ingroup gr_imgftrs
	\defgroup gr_gllf2DGridHist Grid Histogram
	@{

	\struct VplGllf2DGridHist
	\brief Class implementing Global Low-level Feature extraction based on Image Histogram. 
	The Extractor takes several image planes, divides the image space into subwindows 
	(grid) and calculates histograms for each of them. The subwindows can overlap.

*/
typedef struct VplGllf2DGridHist
{
	CvHistogram ** hist;				///< array of histograms (one histogram for each subwindow)
	int dims;							///< dimensionality - amount of image planes used for histogram calculation
	CvSize grid;						///< amount of vertical and horizontal subwindows
	float  gridOverlap;					///< overlapping area (e.g. 15% of subwindow size [1.15])
	int * sizes;						///< amount of histogram bins for each dimension

	int featureSize;					///< feature vector length for whole image (sum of all subwindows)
	int histSize;						///< feature vector length for one subwindow

	CvMat ** planes;					// structures for parallel optimization
	int nt;								// number of threads
} 
VplGllf2DGridHist;



/** \fn VplGllf2DGridHist * vplGllf2DGridHistCreate( void )
	\brief 2DGridHist object constructor.
*/
struct VplGllf2DGridHist * vplGllf2DGridHistCreate( void );



/** \fn VplGllf2DGridHist * vplGllf2DGridHistInit( struct VplGllf2DGridHist * gh, CvSize grid, float gridOverlap, int dims, int * sizes, float ** ranges )
	\brief Initialize 2DGridHist object's structure. The initialization parameters follows the rules for
	initialization of OpenCV histograms.
	@param gh 2DGridHist object handler (if null, new object is constructed).
	@param grid amount of subwindows in vertical and horizontal direction (e.g. cvSize(4,3))
	@param gridOverlap overlapping area (e.g. 15% of subwindow size [1.15])
	@param dims dimensionality - amount of image planes used for histogram calculation
	@param sizes amount of histogram bins for each dimension (e.g. [18,10])
	@param ranges the array of ranges (couples) for each dimension (e.g. [[0,180],[0,255]])
*/
struct VplGllf2DGridHist * vplGllf2DGridHistInit( struct VplGllf2DGridHist * gh, CvSize grid, float gridOverlap, int dims, int * sizes, float ** ranges );



/** \fn vplGllf2DGridHistGetFeature( struct VplGllf2DGridHist * gh, CvMat * feature )
	\brief Copy the last extracted feature vector.
	@param gh 2DGridHist object handler.
	@param feature output vector where the descriptor is stored
*/
int	vplGllf2DGridHistGetFeature( struct VplGllf2DGridHist * gh, CvMat * feature );



/** \fn vplGllf2DGridHistGetSize( struct VplGllf2DGridHist * gh )
	\brief Return the length of feature vector.
	@param gh 2DGridHist object handler.
*/
int	vplGllf2DGridHistGetSize( struct VplGllf2DGridHist * gh );



/** \fn vplGllf2DGridHistExtract( struct VplGllf2DGridHist * gh, CvMat ** images, CvMat * feature )
	\brief Extracts the feature from input array of image planes.
	@param gh 2DGridHist object handler.
	@param images an array of input image planes
	@param feature output vector where the descriptor can be stored
*/
int	vplGllf2DGridHistExtract( struct VplGllf2DGridHist * gh, CvMat ** images, CvMat * feature CV_DEFAULT( NULL ) );



/** \fn vplGllf2DGridHistRender( struct VplGllf2DGridHist * gh, CvArr * canvas )
	\brief Renders the feature values to the canvas.
	@param gh 2DGridHist object handler.
	@param canvas an image to render the feature vector
*/
int	vplGllf2DGridHistRender( struct VplGllf2DGridHist * gh, CvArr * canvas );



/** \fn int vplGllf2DGridHistRelease( struct VplGllf2DGridHist ** gh )
	\brief Release the 2DGridHist object.
	@param gh a pointer to the 2DGridHist object 
*/
int vplGllf2DGridHistRelease( struct VplGllf2DGridHist ** gh );


/** @} */


#ifdef __cplusplus
}
#endif

#endif			//  _VPL_GLLF_COLHIST_H_
