
#include <stdio.h>

#ifdef _OPENMP
#include <omp.h>
#endif

#include <opencv/cv.h>
#include "vplcore.h"
#include "vplgllfgraddist.h"
#include "vpltiming.h"


int vplGllfGradDistClear( struct VplGllfGradDist * gd );


struct VplGllfGradDist * vplGllfGradDistCreate( void )
{
	struct VplGllfGradDist * gd = NULL;

	CV_FUNCNAME( "vplGllfGradDistCreate" );
	__BEGIN__

	gd = ( VplGllfGradDist * )malloc( sizeof( VplGllfGradDist ) );
	if( !gd ) return NULL;

	gd->hist = NULL;
	gd->frame	= NULL;
	gd->subframe = NULL;
	gd->dx  = NULL;
	gd->dy  = NULL;
	gd->mag = NULL;
	gd->ori = NULL;

	gd->pyrs = 0;
	gd->bins = 0;
	gd->featureSize = 0;
	gd->aperture_size = 3;

	//vtm = vplTimeMeasuresCreate( 20 );

	__END__
	return gd;
}


struct VplGllfGradDist * vplGllfGradDistInit( struct VplGllfGradDist * gd, int pyrs, int bins, int aperture_size )
{
	CV_FUNCNAME( "vplGllfGradDistInit" );
	__BEGIN__

	if( !gd ) gd = vplGllfGradDistCreate();
	else           vplGllfGradDistClear( gd );

	gd->hist = cvCreateMat( pyrs, bins, VPL_DATA_MAT_TYPE );
	cvZero( gd->hist );

	gd->pyrs = pyrs;
	gd->bins = bins;
	gd->featureSize = bins * pyrs;
	gd->aperture_size = aperture_size;

	__END__
	return gd;
}


int	vplGllfGradDistGetFeature( struct VplGllfGradDist * gd, CvMat * feature )
{
	CV_FUNCNAME( "vplGllfGradDistGetFeature" );
	__BEGIN__
	CvMat m;

	if( !gd ) EXIT;
	if( !feature || feature->cols != gd->featureSize ) EXIT;

	m = cvMat( 1, gd->pyrs * gd->bins, gd->hist->type, gd->hist->data.ptr );
	cvConvertScale( &m, feature, 1.0, 0.0 );
	
	return 1;

	__END__
	return 0;
}


int	vplGllfGradDistGetSize( struct VplGllfGradDist * gd )
{
	CV_FUNCNAME( "vplGllfGradDistGetSize" );
	if( !gd ) return 0;
	return gd->featureSize;
}



#define RESET_SUB_MAT( dst, orig ) { *(dst) = cvMat((orig)->rows, (orig)->cols, (dst)->type, (dst)->data.ptr ); (dst)->hdr_refcount = 1; *((dst)->refcount) = 1;}

#define SUB_MAT( subwnd ) \
			subframe = cvGetSubRect( subframe, &_subframe, subwnd );	\
			dx  = cvGetSubRect( dx,  &_dx,  subwnd );					\
			dy  = cvGetSubRect( dy,  &_dy,  subwnd );					\
			mag = cvGetSubRect( mag, &_mag, subwnd );					\
			ori = cvGetSubRect( ori, &_ori, subwnd );					


int	vplGllfGradDistExtract( struct VplGllfGradDist * gd, CvArr * image, CvMat * feature, CvMat * filter )
{

	CV_FUNCNAME( "vplGllfGradDistExtract" );
	__BEGIN__

	int i;
	CvMat swap;
	double map;
    CvMat srcstub, * src = (CvMat*)image;
	CvMat * dx, * dy, * mag, * ori, * subframe, * frame;
	CvMat  _dx,  _dy,  _mag,  _ori,  _subframe,  _frame;
	CvRect subwnd;

	if( !gd || !image ) return -1;

	src = cvGetMat( image, &srcstub, NULL, 0 );
	if( CV_MAT_TYPE( src->type ) != CV_8UC1 ) return -2;

	if( gd->frame && !CV_ARE_SIZES_EQ( gd->frame, src ) )
	{
		if( gd->frame )		cvReleaseMat( &gd->frame );
		if( gd->subframe )	cvReleaseMat( &gd->subframe );
		if( gd->dx )		cvReleaseMat( &gd->dx );
		if( gd->dy )		cvReleaseMat( &gd->dy );
		if( gd->mag )		cvReleaseMat( &gd->mag );
		if( gd->ori )		cvReleaseMat( &gd->ori );
	}

	if( !gd->frame )
	{
		gd->frame    = cvCreateMat( src->rows, src->cols, VPL_DATA_MAT_TYPE );
		gd->subframe = cvCreateMat( src->rows, src->cols, VPL_DATA_MAT_TYPE );
		gd->dx		= cvCreateMat( src->rows, src->cols, VPL_DATA_MAT_TYPE );
		gd->dy		= cvCreateMat( src->rows, src->cols, VPL_DATA_MAT_TYPE );
		gd->mag		= cvCreateMat( src->rows, src->cols, VPL_DATA_MAT_TYPE );
		gd->ori		= cvCreateMat( src->rows, src->cols, VPL_DATA_MAT_TYPE );
	}

	subwnd = cvRect( 0, 0, src->cols, src->rows );
	subframe = gd->subframe;
	dx  = gd->dx;
	dy  = gd->dy;
	mag = gd->mag;
	ori = gd->ori;

	SUB_MAT( subwnd );
	frame = cvGetSubRect( gd->frame, &_frame, subwnd );

	cvConvert( src, frame );

	cvZero( gd->hist );
	map = 1.0 * gd->bins / 361;
	for( i = 0; i < gd->pyrs; ++i )
	{
		CvMat m;
		int x, y;
		float * pori, * pmag;

		//vplTimeMeasuresStart( vtm, 10+i );
		//vplTimeMeasuresStart( vtm, 0 );
		#ifdef _OPENMP
			#pragma omp parallel default(shared)
			{
				int tn = omp_get_thread_num();
				if( tn == 0 )
					cvSobel( frame, dx, 1, 0, gd->aperture_size );
				else if( tn == 1 )
					cvSobel( frame, dy, 0, 1, gd->aperture_size );
			}
		#else
			cvSobel( frame, dx, 1, 0, gd->aperture_size );
			cvSobel( frame, dy, 0, 1, gd->aperture_size );
		#endif
		//vplTimeMeasuresAdd( vtm, 0 );
		//vplTimeMeasuresEnd( vtm, 10+i );
		//printf( "Level: %d; Sobel: %f\n", i, 0.001*vplTimeMeasuresAvgGet( vtm, 10+i ) );


		//vplTimeMeasuresStart( vtm, 1 );
		#ifdef _OPENMP
			#pragma omp parallel default(shared)
			{
				int iwidth, icol;
				int tn = omp_get_thread_num();
				int nt = omp_get_num_threads();

				iwidth = frame->cols / nt;			/* size of partition */
				icol = tn * iwidth;					/* starting array index */
				if( tn == nt-1 )					/* last thread may do more */
					iwidth = frame->cols - icol;

				{
					CvMat dx2, dy2, mag2, ori2;
					CvRect r = cvRect( icol, 0, iwidth, frame->rows );
					cvGetSubRect( gd->dx,   &dx2, r );
					cvGetSubRect( gd->dy,   &dy2, r );
					cvGetSubRect( gd->mag, &mag2, r );
					cvGetSubRect( gd->ori, &ori2, r );

					cvCartToPolar( &dx2, &dy2, &mag2, &ori2, 1 );
				}
			}
		#else
			cvCartToPolar( dx, dy, mag, ori, 1 );
		#endif
		//vplTimeMeasuresAdd( vtm, 1 );

		if( filter && CV_ARE_SIZES_EQ(filter,mag) && CV_ARE_TYPES_EQ(filter,mag) )
			cvMul( filter, mag, mag, 1.0 );

		//vplTimeMeasuresStart( vtm, 2 );
		cvGetRow( gd->hist, &m, i );

		pori = (float *)( ori->data.ptr );
		pmag = (float *)( mag->data.ptr );
		for( y = 0; y < ori->rows; ++y )
		{
			for( x = 0; x < ori->cols; ++x )	
				m.data.fl[(int)(pori[x] * map)] += pmag[x];

			pori = (float *)( (uchar *)pori + ori->step );
			pmag = (float *)( (uchar *)pmag + mag->step );
		}

		cvConvertScale( &m, &m, 1.0 / ( frame->rows * frame->cols ), 0 );

		subwnd = cvRect( 0, 0, frame->cols/2, frame->rows/2 );
		SUB_MAT( subwnd );
		//vplTimeMeasuresAdd( vtm, 2 );

		//vplTimeMeasuresStart( vtm, 3 );
		cvPyrDown( frame, subframe, CV_GAUSSIAN_5x5 );
		CV_SWAP( _frame, _subframe, swap );
		//vplTimeMeasuresAdd( vtm, 3 );

	}
	//vplTimeMeasuresEnd( vtm, 0 );	vplTimeMeasuresEnd( vtm, 1 );	vplTimeMeasuresEnd( vtm, 2 );	vplTimeMeasuresEnd( vtm, 3 );
	//printf( "Sobel: %f; Polar: %f; Hist: %f; PyrDown: %f; Sum: %f; ", 		0.001*vplTimeMeasuresAvgGet( vtm, 0 ), 0.001*vplTimeMeasuresAvgGet( vtm, 1 ), 0.001*vplTimeMeasuresAvgGet( vtm, 2 ), 0.001*vplTimeMeasuresAvgGet( vtm, 3 ),		0.001*( vplTimeMeasuresAvgGet( vtm, 0 ) + vplTimeMeasuresAvgGet( vtm, 1 ) + vplTimeMeasuresAvgGet( vtm, 2 ) + vplTimeMeasuresAvgGet( vtm, 3 ) ) );

	if( feature && feature->cols == gd->featureSize )
	{
		CvMat m = cvMat( 1, gd->pyrs * gd->bins, gd->hist->type, gd->hist->data.ptr );
		cvConvertScale( &m, feature, 1.0, 0.0 );
	}

	__END__
	return 1;
}




int	vplGllfGradDistRender( struct VplGllfGradDist * gd, CvArr * canvas )
{
	CV_FUNCNAME( "vplGllfGradDistRender" );
	__BEGIN__

    CvMat srcstub, *src = (CvMat*)canvas;
	int i, j;
	double xs, ys, mav;

	if( !gd || !canvas ) return -1;

	src = cvGetMat( canvas, &srcstub, &i, 0 );
	cvZero( src );

	xs = 1.0 * src->cols / gd->bins;
	ys = 1.0 * src->rows / gd->pyrs;
	for( j = 0; j < gd->pyrs; ++j )
	{
		CvMat m;
		cvGetRow( gd->hist, &m, j );
		cvMinMaxLoc( &m, 0, &mav, 0, 0, 0 );

		for( i = 0; i < gd->bins; ++i )
		{
			int val = cvRound( m.data.db[i] * 255 / mav );
			cvRectangle( src, cvPoint( (int)(i*xs), (int)(j*ys) ), cvPoint( (int)((i+1)*xs), (int)((j+1)*ys) ), CV_RGB(val,val,val), CV_FILLED, 8, 0 );
		}
	}
	cvRectangle( src, cvPoint( 0, 0 ), cvPoint( src->cols-1, src->rows-1 ), CV_RGB(100,100,200), 3, 8, 0 );

	__END__
	return 1;
}



int vplGllfGradDistClear( struct VplGllfGradDist * gd )
{
	CV_FUNCNAME( "vplGllfGradDistClear" );

	if( !gd ) return -1;

	if( gd->hist ) cvReleaseMat( &gd->hist );

	if( gd->frame )	  cvReleaseMat( &gd->frame );
	if( gd->subframe ) cvReleaseMat( &gd->subframe );
	if( gd->dx )		  cvReleaseMat( &gd->dx );
	if( gd->dy )		  cvReleaseMat( &gd->dy );
	if( gd->mag )	  cvReleaseMat( &gd->mag );
	if( gd->ori )	  cvReleaseMat( &gd->ori );

	gd->pyrs = 0;
	gd->bins = 0;
	gd->featureSize = 0;

	return 1;
}


int vplGllfGradDistRelease( struct VplGllfGradDist ** gd )
{
	CV_FUNCNAME( "vplGllfGradDistRelease" );

	struct VplGllfGradDist * g;

	if( !gd || !*gd ) return 1;

	g = *gd;
	vplGllfGradDistClear( g );

	free( g );
	*gd = NULL;

	return 1;
}
