///////////////////////////////////////////////////////////////////////////////
// $Id$
//
// EasyDent
// Navigation and implants planning software for dentistry.
//
// Copyright (c) 2008 by 3Dim Laboratory s.r.o.
//
///////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// Header file for CMarchingCubes class

#ifndef MC_MARCHCUBE_H
#define MC_MARCHCUBE_H

////////////////////////////////////////////////////////////
// Includes

#include <VectorEntity/mctris.h>

// MDSTk
#include <MDSTk/Image/mdsDensityVolume.h>
#include <MDSTk/Module/mdsSerializable.h>
#include <MDSTk/Module/mdsProgress.h>

// STL
#include <vector>

////////////////////////////////////////////////////////////
//! Functor for thresholding volume on a specified range.

template <typename V, typename T>
class CThresholdFunctor 
{
public:
    //! Default constructor.
    CThresholdFunctor(const T& low_threshold, const T& high_threshold) : m_low_threshold(low_threshold), m_high_threshold(high_threshold) {}

    //! Limits value of a given parameter.
    unsigned char operator ()(V & Volume, int x, int y, int z)
    {
        // take voxel value for given coordinates
        T   voxel_value = Volume.get(x, y, z);

        // test volume size to voxel position
        if ( (Volume.getXSize() > x) && (Volume.getYSize() > y) && (Volume.getZSize() > z) )
        {
            // test volume value to threshold
            if ((voxel_value >= m_low_threshold) && (voxel_value <= m_high_threshold))
                return 1;
            else
                return 0;
        }
        else
            return 0;
    }

protected:

    T       m_low_threshold, m_high_threshold;             //! Low and high threshold values.
};

////////////////////////////////////////////////////////////
//! CMarchingCubes algorithm implementation class.

class CMarchingCubes : public mds::mod::CProgress
	{
		private:
            unsigned char               * cube_code_matrix;                                 //! Pointer on up codes of cubes.
            unsigned char               * down_cube_code_matrix;                            //! Pointer on down codes of cubes.
            unsigned char               * state_matrix_up, * state_matrix_down;             //! Pointer on voxels state matrix, up and down.
            vctl::MCPoint3D             cube_vertices[8];                                   //! Actual cube vertices coordinates.
            vctl::MCVertex		        ** node_matrix_up_h, ** node_matrix_up_v;           //! Pointer on up work matrices of edges vertices.
            vctl::MCVertex		        ** node_matrix_down_h, ** node_matrix_down_v;       //! Pointer on down work matrices of edges vertices.
            vctl::MCVertex		        ** node_matrix_middle;                              //! Pointer on middle work matrices of edges vertices.
            vctl::MCTriS                * m_tris;                                           //! Pointer on created tri mesh.
            int                         work_matrices_size_x, work_matrices_size_y;         //! Size of help work matrices.

		public:
            //! Marching cubes class constructor.
            CMarchingCubes();

            //! Marching cubes class destructor.
            ~CMarchingCubes();

		private:

            //! Allocation of work matrices.
            void AllocWorktMem(int size_x, int size_y);
            //! Deallocation of work matrices.
            void DeallocWorktMem();

            //! Make tris for actual cube, given coordinates of processed planes and cube code.
            void MakeTri(int x, int y, unsigned char cube_code);

            //! Patching holes of tri mesh in case of complement cube code
            void HoleFilling(int x, int y, unsigned char code);

            //! Get pointer on vertex of cube node by given coordinates and cube edge index.
            vctl::MCVertex * GetCubeEdgeNode(int edge_index, int x, int y);
            //! Set pointer on vertex of cube node by given coordinates and cube edge index.
            void SetCubeEdgeNode(int edge_index, int x, int y, vctl::MCVertex * new_vertex);

            //! Set cube node Z coordinates.
            void SetCodeCoordinateZ(double z, double dz)              
            { 
                cube_vertices[0].SetZ(z); cube_vertices[1].SetZ(cube_vertices[0].GetZ());
                cube_vertices[2].SetZ(cube_vertices[0].GetZ()); cube_vertices[3].SetZ(cube_vertices[0].GetZ());
                cube_vertices[4].SetZ(cube_vertices[0].GetZ() + dz); cube_vertices[5].SetZ(cube_vertices[4].GetZ());
                cube_vertices[6].SetZ(cube_vertices[4].GetZ()); cube_vertices[7].SetZ(cube_vertices[4].GetZ()); 
            };
            //! Set cube node Y coordinates.
            void SetCodeCoordinateY(double y, double dy)              
            { 
                cube_vertices[0].SetY(y); cube_vertices[1].SetY(cube_vertices[0].GetY());
                cube_vertices[2].SetY(cube_vertices[0].GetY() + dy); cube_vertices[3].SetY(cube_vertices[2].GetY());
                cube_vertices[4].SetY(cube_vertices[0].GetY()); cube_vertices[5].SetY(cube_vertices[0].GetY());
                cube_vertices[6].SetY(cube_vertices[2].GetY()); cube_vertices[7].SetY(cube_vertices[2].GetY()); 
            };
            //! Set cube node X coordinates.
            void SetCodeCoordinateX(double x, double dx)              
            { 
                cube_vertices[0].SetX(x); cube_vertices[1].SetX(cube_vertices[0].GetX() + dx);
                cube_vertices[2].SetX(cube_vertices[1].GetX()); cube_vertices[3].SetX(cube_vertices[0].GetX());
                cube_vertices[4].SetX(cube_vertices[0].GetX()); cube_vertices[5].SetX(cube_vertices[1].GetX());
                cube_vertices[6].SetX(cube_vertices[1].GetX()); cube_vertices[7].SetX(cube_vertices[0].GetX()); 
            };

            //! Get cube code of down cube.
            unsigned char GetCubeCodeDown(int x, int y)                      { return ( *(down_cube_code_matrix + (y+1)*work_matrices_size_x + x + 1) ); };
            //! Get cube code of front cube.
            unsigned char GetCubeCodeFront(int x, int y)                     { return ( *(cube_code_matrix + y*work_matrices_size_x + x + 1) ); };
            //! Get cube code of left cube.
            unsigned char GetCubeCodeLeft(int x, int y)                      { return ( *(cube_code_matrix + (y+1)*work_matrices_size_x + x) ); };

            //! Create actual cube code and save it into code matrix
            unsigned char MakeCubeCode(int x, int y)          
            { 
                // create cube code
                unsigned char code = *(state_matrix_down + y*work_matrices_size_x + x);
                code += ( *(state_matrix_down + y*work_matrices_size_x + x + 1) ) << 1;
                code += ( *(state_matrix_down + ((y+1)*work_matrices_size_x) + x + 1) ) << 2;
                code += ( *(state_matrix_down + ((y+1)*work_matrices_size_x) + x) ) << 3;
                code += ( *(state_matrix_up + y*work_matrices_size_x + x) ) << 4;
                code += ( *(state_matrix_up + y*work_matrices_size_x + x + 1) ) << 5;
                code += ( *(state_matrix_up + ((y+1)*work_matrices_size_x) + x + 1) ) << 6;
                code += ( *(state_matrix_up + ((y+1)*work_matrices_size_x) + x) ) << 7;
                // save cube code into code matrix
                *(cube_code_matrix + (y+1)*work_matrices_size_x + x + 1) = code;
                return code;
            };

            //! Calculate number of nodes for cube code.
            int CubeCodeNodeNumber(unsigned char cube_code)
            {
                int voxels = cube_code & 1; 
                voxels += (cube_code >> 1) & 1; voxels += (cube_code >> 2) & 1; voxels += (cube_code >> 3) & 1;
                voxels += (cube_code >> 4) & 1; voxels += (cube_code >> 5) & 1; voxels += (cube_code >> 6) & 1; voxels += (cube_code >> 7) & 1;
                return voxels;
            };

            //! Test orientation of created mesh tris for edge given by two vertices.
            bool TestEdgeDirection(vctl::MCVertex * u0, vctl::MCVertex * u1);

            //! Test of created mesh tris orientation, if all tris have same orientation.
            bool TestTriOrientation();

            //! Markup all vertices of input tri mesh. 
            //! Divide them on three types: flat, edge and corner.
            bool MarkupVertices();

            //! Eliminate input flat vertex, the vertex lay in flat area
            //! @return Return false, if input vertex is not eliminated, otherwise return true.
            bool EliminateFlatVertex(vctl::MCVertex * eliminate_vertex);

            //! Eliminate input flat vertex, the vertex lay on edge
            //! @return Return false, if input vertex is not eliminated, otherwise return true.
            bool EliminateEdgeVertex(vctl::MCVertex * eliminate_vertex);

            //! Make collapse of eliminate vertex into result vertex.
            bool CollapseEdge(vctl::MCVertex * eliminate_vertex, vctl::MCVertex * result_vertex, std::vector<vctl::MCTri *> & tri_list);

            //! Swap all edges of input tri mesh to improve their quality (shape). 
            bool SwapTriEdges();

            //! Swap maximal edge of input tri to improve his quality.
            //! @return return true, if the swap was realized.
            bool SwapTriImproveQuality(vctl::MCTri * swap_tri);

        public:

            //! Reduce flat areas of tri mesh generated by Marching Cubes algorithm
            bool ReduceFlatAreas(vctl::MCTriS & tris);

            //! Marching cubes triangle mesh generation.
            template <typename Function>
            bool GenerateMesh(mds::img::CDensityVolume & volume, vctl::MCTriS & tris, Function volume_functor, bool FlowReduction)
            {
                int                   volume_size_x, volume_size_y, volume_size_z;                      // actual volume grid size
                double                dx, dy, dz;                                                       // actual volume voxel size
                unsigned char         cube_code;                                                        // code of actual cube
                unsigned char         * state_matrix_ptr, * cube_code_matrix_ptr;                       // help pointers on state and code matrix
                vctl::MCVertex        ** node_matrix_ptr;                                               // help pointer on node matrix


                // save pointers on actual tri mesh
                m_tris = &tris;

                // take volume size
                volume_size_x = volume.getXSize();
                volume_size_y = volume.getYSize();
                volume_size_z = volume.getZSize();

                // take voxel size
                dx = volume.getDX();
                dy = volume.getDY();
                dz = volume.getDZ();

                // allocation of work matrices
                AllocWorktMem(volume_size_x, volume_size_y);

                // set maximal progress value
                setProgressMax(volume_size_z);
                // start function progress
                beginProgress();

                // cycle throw volume in Z axis
                for (int z = 0; z <= volume_size_z; ++z)
                {
                    // set cube node Z coordinates
                    SetCodeCoordinateZ((z-1)*dz, dz);

                    // cycle throw volume in Y axis
                    for (int y = 0; y <= volume_size_y; ++y)
                    {
                        // Set cube node Y coordinates
                        SetCodeCoordinateY((y-1)*dy, dy);

                        // cycle throw volume in X axis
                        for (int x = 0; x <= volume_size_x; ++x)
                        {
                            // Set cube node X coordinates
                            SetCodeCoordinateX((x-1)*dx, dx);

                            // get volume voxel state and save it into up voxel state matrix
                            *(state_matrix_up + ((y+1)*work_matrices_size_x) + x + 1) = volume_functor(volume, x, y, z);

                            // create actual cube code and save it into code matrix
                            cube_code = MakeCubeCode(x, y);

                            // test of empty or full cube, with no cross surface
                            if ((cube_code != 0) && (cube_code != 255))
                            {
                                // make tris for actual cube
                                MakeTri(x, y, cube_code);
                                // fill possible holes for actual cube
                                HoleFilling(x, y, cube_code);
                            }
                        }
                    }

                    // switch up and down state matrices
                    state_matrix_ptr = state_matrix_down;
                    state_matrix_down = state_matrix_up;
                    state_matrix_up = state_matrix_ptr;
                    // switch up and down node matrices
                    node_matrix_ptr = node_matrix_down_h;
                    node_matrix_down_h = node_matrix_up_h;
                    node_matrix_up_h = node_matrix_ptr;
                    node_matrix_ptr = node_matrix_down_v;
                    node_matrix_down_v = node_matrix_up_v;
                    node_matrix_up_v = node_matrix_ptr;
                    // switch cube code matrix with down cube code matrix
                    cube_code_matrix_ptr = cube_code_matrix;
                    cube_code_matrix = down_cube_code_matrix;
                    down_cube_code_matrix = cube_code_matrix_ptr;

                    // clearing state, node and code matrices
                    memset(state_matrix_up, 0, work_matrices_size_x * work_matrices_size_y * sizeof(unsigned char));
                    memset(node_matrix_middle, 0, work_matrices_size_x * work_matrices_size_y * sizeof(vctl::MCVertex *));
                    memset(node_matrix_up_h, 0, work_matrices_size_x * work_matrices_size_y * sizeof(vctl::MCVertex *));
                    memset(node_matrix_up_v, 0, work_matrices_size_x * work_matrices_size_y * sizeof(vctl::MCVertex *));
                    memset(cube_code_matrix, 0, work_matrices_size_x * work_matrices_size_y * sizeof(unsigned char));

                    // set progress value and test function termination
                    if ( !progress() )
                    {
                        // deallocation of work matrices
                        DeallocWorktMem();
                        // end function
                        return true;
                    }
                }

                // finish function progress
                endProgress();

                // deallocation of work matrices
                DeallocWorktMem();

                // test of created mesh tris orientation
                //if (TestTriOrientation())
                //    return false;

                return true;
            };

	};

#endif

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
