///////////////////////////////////////////////////////////////////////////////
// $Id: CDecimator.cpp 636 2009-11-13 10:05:16Z krsek $
//
// Copyright (c) 2008 by 3Dim Laboratory s.r.o.
//
///////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// Includes

//#include "alg\CDecimator.h"
#include "CDecimator.h"

////////////////////////////////////////////////////////////
//

bool CDecimator::Reduce(vctl::MCTriS & tris, int final_tri_number)
{
    int                                     start_tri_number = tris.GetNumber();        // input tri mesh beginning number
    int                                     progress_step;                              // reduction process progress step value 
    int                                     index = 0;                                  // collapsing index
    vctl::MCEdge                            * actual_edge;                              // pointer to actual minimal edge
    vctl::MCEdge                            * second_edge;                              // pointer to second minimal edge
    vctl::MCEdgeS::MT_QUEUE                 * edge_queue;                               // pointer on edge evaluation queue
    vctl::MCVertex                          * collapse_vertex;                          // result vertex after edge collapsing
    double                                  actual_edge_volume;                         // collapse volume of actual edge
    double                                  actual_edge_new_volume;                     // new collapse volume of actual edge


    // test final tri number
    if (final_tri_number >= start_tri_number)
        return true;

    // take pointer on edge evaluation queue
    edge_queue = tris.GetEdgeS()->GetQueue();

    // make edges for input tri mesh
    tris.GetEdgeS()->ClearAll();
    tris.MakeAllTrisEdges();

    // evaluate created edges
    if ( ! EvaluateAllEdgesVolume(tris))
        return false;

    // calculate decimation progress step
    progress_step = (start_tri_number - final_tri_number) / 2000 + 1;
    // set maximal progress value
    setProgressMax(1000);
    // start function progress
    beginProgress();

    // take actual two minimal edges
    edge_queue->GetMinFirstTwo(&actual_edge, &second_edge);

    // cycle of edge reduction until final tri number is reached
    while(tris.GetNumber() >= final_tri_number)
    {
        // save collapse volume of actual edge
        actual_edge_volume = actual_edge->GetValue();
        // reevaluate collapse of actual edge
        EvaluateEdgeVolume(tris, actual_edge);
        // test changes of actual edge collapse volume and collapse variant
        if ( (actual_edge->GetValue() > second_edge->GetValue()) || actual_edge->GetObject() == 100 )
        //if ( (fabs(actual_edge->GetValue() - actual_edge_volume) > 0.001) || actual_edge->GetObject() == 100 )
        {
            // save new collapse volume of actual edge
            actual_edge_new_volume = actual_edge->GetValue();
            // set back old collapse volume of actual edge
            actual_edge->SetValue(actual_edge_volume);
            // unsort actual edge
            tris.GetEdgeS()->UnSort(actual_edge);
            // set back new collapse volume of actual edge
            actual_edge->SetValue(actual_edge_new_volume);
            // unsort actual edge with new collapse volume
            tris.GetEdgeS()->Sort(actual_edge);
        }
        else
        {
            // set back old collapse volume of actual edge
            actual_edge->SetValue(actual_edge_volume);
            // make actual edge collapse 
            if ( (collapse_vertex = CollapseEdge(tris, actual_edge)) != NULL)
                ++index;

            // test for progress incrementation step
            if ( (index % progress_step) == 0)
            {
                // set progress value and test function termination
                if ( ! progress() )
                    return true;
            }
        }

        // take actual two minimal edges
        edge_queue->GetMinFirstTwo(&actual_edge, &second_edge);
    }

    // finish function progress
    endProgress();

    // delete input tri mesh edges
    tris.GetEdgeS()->ClearAll();

    // OK
    return true;
}

////////////////////////////////////////////////////////////
//

bool CDecimator::EvaluateAllEdgesVolume(vctl::MCTriS & tris)
{
    vctl::MCEdge            * actual_edge = tris.GetEdgeS()->GetFirst();		// pointer on actual edge of input tri mesh
    int                     progress_step;                                      // reduction process progress step value 
    int                     index = 0;                                          // actual edge index


    // calculate decimation progress step
    progress_step = tris.GetEdgeS()->GetNumber() / 1000 + 1;
    // set maximal progress value
    setProgressMax(1000);
    // start function progress
    beginProgress();
    
    // cycle of input mesh edges
    while (actual_edge != NULL)
    {
        // evaluation of actual edge
        EvaluateEdgeVolume(tris, actual_edge);

        // test for progress incrementation step
        if ( (index % progress_step) == 0)
        {
            // set progress value and test function termination
            if ( ! progress() )
                return true;
        }

        // take next edge of input tri mesh
        actual_edge = actual_edge->GetNext();
        ++index;
    }

    // sort edges by their value
    tris.GetEdgeS()->SortAll();

    // OK
    return true;
}

////////////////////////////////////////////////////////////
//

void CDecimator::EvaluateEdgeVolume(vctl::MCTriS & tris, vctl::MCEdge * evaluated_edge)
{
    vctl::MCVertex                  * u0, *u1;                      // pointers on input edge vertices
    double                          collapse_volume_u0;             // collapse volume for new vertex position in first edge vertex
    double                          collapse_volume_u1;             // collapse volume for new vertex position in second edge vertex
    double                          collapse_volume;                // best collapse variant volume value
    int								variant_index;			        // best collapse variant index


    // set pointers on input edge vertices
    assert(evaluated_edge != NULL);
    u0 = evaluated_edge->GetVertex(0);
    u1 = evaluated_edge->GetVertex(1);
    assert(u0 != NULL);
    assert(u1 != NULL);

    // take tris around input edge vertices, exclude tris shared by input edge
    u0->GetRegisteredTriListExclude(m_tri_u0, u1);
    u1->GetRegisteredTriListExclude(m_tri_u1, u0);
    assert(m_tri_u0.size() > 0);
    assert(m_tri_u1.size() > 0);

    // take tris around input edge
    tris.GetTriEdge(u0, u1, m_tri_edge);
    assert(m_tri_edge.size() == 2);

    // test for tri duplicities in case of input edge collapse
    if ( ! TestCollapseDuplicities(m_tri_u0, m_tri_u1, m_tri_edge))
    {
        // collapses volume calculations, two variants 
        collapse_volume_u0 = EvaluateCollapse(u0, u1, m_tri_u0, m_tri_u1, u0);
        collapse_volume_u1 = EvaluateCollapse(u0, u1, m_tri_u0, m_tri_u1, u1);

        // volumes compare, variant choice
        if (collapse_volume_u0 < collapse_volume_u1)
        {
            collapse_volume = collapse_volume_u0 + u1->GetValue();
            variant_index = 0;
        }
        else
        {
            collapse_volume = collapse_volume_u1 + u0->GetValue();
            variant_index = 1;
        }
    }
    else
    {   // collapse is not possible because of duplicities
        variant_index = 100;        
        collapse_volume = 1e9 + ( (double)rand() / (RAND_MAX + 1) * 100 );
    }

    // save volume value and index of best collapse variant
    evaluated_edge->SetObject(variant_index);
    evaluated_edge->SetValue(collapse_volume);
}

////////////////////////////////////////////////////////////
//

bool CDecimator::TestCollapseDuplicities(std::vector<vctl::MCTri *> & tri_u0, std::vector<vctl::MCTri *> & tri_u1, std::vector<vctl::MCTri *> & tri_edge)
{
    bool						            result = false;				// test for collapse duplicities result, default value is false = no duplicities
    std::vector<vctl::MCTri *>::iterator    tri_iterator;               // vector tris pointer iterator
    std::vector<vctl::MCTri *>::iterator    tri_iterator_end;           // vector tris pointer end iterator


    // markup of tris around first edge vertex u0
    tri_iterator_end = tri_u0.end();
    for (tri_iterator = tri_u0.begin(); tri_iterator != tri_iterator_end; ++tri_iterator)
    {
        assert(*tri_iterator != NULL);
        (*tri_iterator)->GetVertex(0)->SetFlag(vctl::ENTITY_MARK_FLAG);
        (*tri_iterator)->GetVertex(1)->SetFlag(vctl::ENTITY_MARK_FLAG);
        (*tri_iterator)->GetVertex(2)->SetFlag(vctl::ENTITY_MARK_FLAG);
    }

    // remove mark from tris around input edge
    tri_iterator_end = tri_edge.end();
    for (tri_iterator = tri_edge.begin(); tri_iterator != tri_iterator_end; ++tri_iterator)
    {
        assert(*tri_iterator != NULL);
        (*tri_iterator)->GetVertex(0)->MaskFlag(vctl::ENTITY_MARK_FLAG);
        (*tri_iterator)->GetVertex(1)->MaskFlag(vctl::ENTITY_MARK_FLAG);
        (*tri_iterator)->GetVertex(2)->MaskFlag(vctl::ENTITY_MARK_FLAG);
    }

    // search markup vertices of tris around second edge vertex u1
    tri_iterator_end = tri_u1.end();
    for (tri_iterator = tri_u1.begin(); tri_iterator != tri_iterator_end; ++tri_iterator)
    {
        assert(*tri_iterator != NULL);
        // test markup vertices of actual tri
        if ( ((*tri_iterator)->GetVertex(0)->TestFlag(vctl::ENTITY_MARK_FLAG)) || ((*tri_iterator)->GetVertex(1)->TestFlag(vctl::ENTITY_MARK_FLAG)) || ((*tri_iterator)->GetVertex(2)->TestFlag(vctl::ENTITY_MARK_FLAG)) )
        {
            result = true;
            break;
        }
    }

    // remove mark from tris around first edge vertex u0
    tri_iterator_end = tri_u0.end();
    for (tri_iterator = tri_u0.begin(); tri_iterator != tri_iterator_end; ++tri_iterator)
    {
        (*tri_iterator)->GetVertex(0)->MaskFlag(vctl::ENTITY_MARK_FLAG);
        (*tri_iterator)->GetVertex(1)->MaskFlag(vctl::ENTITY_MARK_FLAG);
        (*tri_iterator)->GetVertex(2)->MaskFlag(vctl::ENTITY_MARK_FLAG);
    }

    return result;
}

////////////////////////////////////////////////////////////
//

double CDecimator::EvaluateCollapse(vctl::MCVertex * u0, vctl::MCVertex * u1, std::vector<vctl::MCTri *> & tri_u0, std::vector<vctl::MCTri *> & tri_u1, vctl::MCVertex * new_vertex)
{
    std::vector<vctl::MCTri *>::iterator    tri_iterator;               // vector tris pointer iterator
    std::vector<vctl::MCTri *>::iterator    tri_iterator_end;           // vector tris pointer end iterator
    vctl::MCVector3D			            tri_normal;                 // normal vector of actual tri
    vctl::MCVertex                          * ur0, * ur1;               // rest vertices of actual tri against actual edge vertex
    double					                collapse_volume = 0.0;      // collapse error volume


    // test of vertices pointer existence
    assert(u0 != NULL);
    assert(u1 != NULL);
    assert(new_vertex != NULL);

    // test of new vertex, if it is first input edge vertex
    if (new_vertex != u0)
    {
        // take vector of edge vertex move, tetrahedron creation
        m_move_vector.Make(u0, new_vertex);

        // cycle of tris around first input edge vertex
        tri_iterator_end = tri_u0.end();
        for (tri_iterator = tri_u0.begin(); tri_iterator != tri_iterator_end; ++tri_iterator)
        {
            assert(*tri_iterator != NULL);
            // take rest vertices of actual tri against actual edge vertex
            (*tri_iterator)->GetRestEdge(u0, &ur0, &ur1);
            assert( (ur0 != NULL) && (ur1 != NULL) );
            assert( (ur0 != ur1) && (ur0 != u0) && (ur1 != u0) );

            // take actual tri and collapsed tri normals
            (*tri_iterator)->GetNormal(tri_normal);
            vctl::MCVector3D::MakeNormal(ur0, ur1, new_vertex, m_collapsed_normal);

            // collapsed error volume calculation
            collapse_volume += fabs( vctl::MCVector3D::DotProduct(m_collapsed_normal, m_move_vector) * 0.1666666667 );

            // test of angle between actual tri and collapsed tri
            if ( (vctl::MCVector3D::DotProduct(tri_normal, m_collapsed_normal)) < 0.7 )
            {
                // it is not possible to realize that collapse
                return (collapse_volume + 1e6  + ( (double)rand() / (RAND_MAX + 1) * 100) );
            }
        }
    }

    // test of new vertex, if it is second input edge vertex
    if (new_vertex != u1)
    {
        // take vector of edge vertex move, tetrahedron creation
        m_move_vector.Make(u1, new_vertex);

        // cycle of tris around first input edge vertex
        tri_iterator_end = tri_u1.end();
        for (tri_iterator = tri_u1.begin(); tri_iterator != tri_iterator_end; ++tri_iterator)
        {
            assert(*tri_iterator != NULL);
            // take rest vertices of actual tri against actual edge vertex
            (*tri_iterator)->GetRestEdge(u1, &ur0, &ur1);
            assert( (ur0 != NULL) && (ur1 != NULL) );
            assert( (ur0 != ur1) && (ur0 != u1) && (ur1 != u1) );

            // take actual tri and collapsed tri normals
            (*tri_iterator)->GetNormal(tri_normal);
            vctl::MCVector3D::MakeNormal(ur0, ur1, new_vertex, m_collapsed_normal);

            // collapsed error volume calculation
            collapse_volume += fabs( vctl::MCVector3D::DotProduct(m_collapsed_normal, m_move_vector) * 0.1666666667 );

            // test of angle between actual tri and collapsed tri
            if ( (vctl::MCVector3D::DotProduct(tri_normal, m_collapsed_normal)) < 0.7 )
            {
                // it is not possible to realize that collapse
                return (collapse_volume + 1e6 + ( (double)rand() / (RAND_MAX + 1) * 100) );
            }
        }
    }

    // return calculated collapse volume
    return collapse_volume;
}

////////////////////////////////////////////////////////////
//

vctl::MCVertex * CDecimator::CollapseEdge(vctl::MCTriS & tris, vctl::MCEdge * collapsed_edge)
{
    int                                     collapse_variant;				// collapse variant index 
    vctl::MCVertex                          * u0, *u1;                      // pointers on input edge vertices
    std::vector<vctl::MCTri *>::iterator    tri_iterator;                   // vector tris pointer iterator
    std::vector<vctl::MCTri *>::iterator    tri_iterator_end;               // vector tris pointer end iterator


    // set pointers on input edge vertices
    assert(collapsed_edge != NULL);
    u0 = collapsed_edge->GetVertex(0);
    u1 = collapsed_edge->GetVertex(1);
    assert(u0 != NULL);
    assert(u1 != NULL);

    // take tris around input edge vertices, exclude tris shared by input edge
    u0->GetRegisteredTriListExclude(m_tri_u0, u1);
    u1->GetRegisteredTriListExclude(m_tri_u1, u0);
    assert(m_tri_u0.size() > 0);
    assert(m_tri_u1.size() > 0);

    // take tris around input edge
    tris.GetTriEdge(u0, u1, m_tri_edge);
    assert(m_tri_edge.size() == 2);

    // test of tetrahedron edge collapse
    if ((m_tri_u0.size() == 1) && (m_tri_u1.size() == 1))
    {
        // delete collapsed tetrahedron data
        DeleteTetra(tris, m_tri_edge[0]);
        // return rest vertex
        return NULL;
    }

    // take collapse variant index
    collapse_variant = collapsed_edge->GetObject();
    assert(collapse_variant >= 0);

    // set new vertex coordinates by collapse variant
    if (collapse_variant == 0)
    {
        // set coordinates of second vertex u1 by vertex u0
        u1->SetPoint3D(u0);

        // create list of influenced tris
        //m_tri_list_influenced.assign(m_tri_u1.begin(), m_tri_u1.end());
    }
    else if (collapse_variant == 1)
    {
        // set coordinates of second vertex u1 by vertex u0
        u0->SetPoint3D(u1);

        // create list of influenced tris
        //m_tri_list_influenced.assign(m_tri_u0.begin(), m_tri_u0.end());
    }
    else if (collapse_variant == 100)
    {
        // unsort collpased edge
        tris.GetEdgeS()->UnSort(collapsed_edge);
        // modify value of collpased edge
        collapsed_edge->SetValue(collapsed_edge->GetValue() + 1000.0);
        // sort collpased edge
        tris.GetEdgeS()->Sort(collapsed_edge);
        // clear list of influenced tris
        //m_tri_list_influenced.clear();
        // return no collapse
        return NULL;
    }

    // set rest vertex value by collapsed edge value
    u1->SetValue(collapsed_edge->GetValue());

    // realization of edge collapse
    MakeCollapse(tris, u0, u1, m_tri_u0, m_tri_edge);

    // return rest vertex
    return u1;
}

////////////////////////////////////////////////////////////
//

void CDecimator::MakeCollapse(vctl::MCTriS & tris, vctl::MCVertex * u0, vctl::MCVertex * u1, std::vector<vctl::MCTri *> & tri_u0, std::vector<vctl::MCTri *> & tri_edge)
{
    std::vector<vctl::MCTri *>::iterator    tri_iterator;               // vector tris pointer iterator
    std::vector<vctl::MCTri *>::iterator    tri_iterator_end;           // vector tris pointer end iterator
    std::vector<vctl::MCEdge *>::iterator   edge_iterator;              // vector edge pointers around vertex iterator
    std::vector<vctl::MCEdge *>::iterator   edge_iterator_end;          // vector edge pointers around vertex end iterator
    vctl::MCVertex                          * u2, * u3;                 // pointers on rest vertices of tris around collapsed edge


    // test tris around collapsed edge
    assert(tri_edge.size() == 2);
    assert(tri_edge[0] != NULL);
    assert(tri_edge[1] != NULL);

    // take rest vertices of tris around collapsed edge
    u2 = tri_edge[0]->GetRestVertex(u0, u1);
    u3 = tri_edge[1]->GetRestVertex(u0, u1);

    // cycle of tris around collapsed edge first vertex u0, change from u0 to u1
    tri_iterator_end = tri_u0.end();
    for (tri_iterator = tri_u0.begin(); tri_iterator != tri_iterator_end; ++tri_iterator)
    {
        assert(*tri_iterator != NULL);
        // deregistration
        (*tri_iterator)->DeRegistration();
        // change vertex
        (*tri_iterator)->ChangeVertex(u0, u1);
        // registration
        (*tri_iterator)->Registration();
    }

    // take edges around first vertex of collapsed edge
    u0->GetRegisteredEdgeList(m_edge_list);
    edge_iterator_end = m_edge_list.end();

    // cycle of edges around first vertex of collapsed edge
    for (edge_iterator = m_edge_list.begin(); edge_iterator != edge_iterator_end; ++edge_iterator)
    {
        assert( *edge_iterator != NULL );
        // test vertices of tris around collapsed edge for actual edge
        if ( (*edge_iterator)->IsVertexBool(u1) || (*edge_iterator)->IsVertexBool(u2) || (*edge_iterator)->IsVertexBool(u3))
        {
            // unsort redundant edge
            tris.GetEdgeS()->UnSort( *edge_iterator );
            // delete redundant edge
            tris.GetEdgeS()->Erase( *edge_iterator );
        }
        else
        {
            // deregistration
            (*edge_iterator)->DeRegistration();
            // change vertex from u1 to u0
            (*edge_iterator)->ChangeVertex(u0, u1);
            // registration
            (*edge_iterator)->Registration();
        }
    }

    // delete tris around collapsed edge
    tris.Erase(tri_edge[0]);
    tris.Erase(tri_edge[1]);

    // delete collapsed edge first vertex u0
    tris.GetVerticeS()->Erase(u0);
}

////////////////////////////////////////////////////////////
//

void CDecimator::DeleteTetra(vctl::MCTriS & tris, vctl::MCTri * base_tri)
{
    vctl::MCVertex                  * tri_vertex[4];                    // array of pointer on tetrahedron vertices
    vctl::MCTri                     * neighbor_tri[3];                  // array of pointers on neighbor tri for base tri of tetrahedron


    // take neighbor tri of base tri
    assert(base_tri != NULL);
    base_tri->GetNeighbours(neighbor_tri);

    // take vertices of tetrahedron
    tri_vertex[0] = base_tri->GetVertex(0);
    tri_vertex[1] = base_tri->GetVertex(1);
    tri_vertex[2] = base_tri->GetVertex(2);
    tri_vertex[3] = neighbor_tri[0]->GetRestVertex(tri_vertex[0], tri_vertex[1]);
    assert(neighbor_tri[1]->IsVertexBool(tri_vertex[3]) == true);
    assert(neighbor_tri[2]->IsVertexBool(tri_vertex[3]) == true);

    // take and delete edges registered for tetrahedron vertices
    tri_vertex[0]->GetRegisteredEdgeList(m_edge_list);
    assert(m_edge_list.size() == 3);
    assert( (m_edge_list[0] != NULL) && (m_edge_list[1] != NULL) && (m_edge_list[2] != NULL) );
    tris.GetEdgeS()->UnSort(m_edge_list[0]);
    tris.GetEdgeS()->UnSort(m_edge_list[1]);
    tris.GetEdgeS()->UnSort(m_edge_list[2]);
    tris.GetEdgeS()->Erase(m_edge_list[0]);
    tris.GetEdgeS()->Erase(m_edge_list[1]);
    tris.GetEdgeS()->Erase(m_edge_list[2]);
    tri_vertex[1]->GetRegisteredEdgeList(m_edge_list);
    assert(m_edge_list.size() == 2);
    assert( (m_edge_list[0] != NULL) && (m_edge_list[1] != NULL) );
    tris.GetEdgeS()->UnSort(m_edge_list[0]);
    tris.GetEdgeS()->UnSort(m_edge_list[1]);
    tris.GetEdgeS()->Erase(m_edge_list[0]);
    tris.GetEdgeS()->Erase(m_edge_list[1]);
    tri_vertex[2]->GetRegisteredEdgeList(m_edge_list);
    assert(m_edge_list.size() == 1);
    assert(m_edge_list[0] != NULL);
    tris.GetEdgeS()->UnSort(m_edge_list[0]);
    tris.GetEdgeS()->Erase(m_edge_list[0]);

    // delete tetrahedron tris
    tris.Erase(base_tri);
    tris.Erase(neighbor_tri[0]);
    tris.Erase(neighbor_tri[1]);
    tris.Erase(neighbor_tri[2]);
    assert(tri_vertex[0]->GetRegisteredTri() == NULL);

    // delete tetrahedron vertices
    tris.GetVerticeS()->Erase(tri_vertex[0]);
    tris.GetVerticeS()->Erase(tri_vertex[1]);
    tris.GetVerticeS()->Erase(tri_vertex[2]);
    tris.GetVerticeS()->Erase(tri_vertex[3]);
}

////////////////////////////////////////////////////////////
//

void CDecimator::ReevaluateColapseChanges(vctl::MCTriS & tris, vctl::MCVertex * new_vertex)
{
    std::vector<vctl::MCTri *>::iterator    tri_iterator;               // vector iterator of collapse influenced tri pointers 
    std::vector<vctl::MCTri *>::iterator    tri_iterator_end;           // vector end iterator of collapse influenced tri pointers 
    std::vector<vctl::MCEdge *>::iterator   edge_iterator;              // vector of edge pointers around new vertex iterator
    std::vector<vctl::MCEdge *>::iterator   edge_iterator_end;          // vector of edge pointers around new vertex end iterator
    std::vector<vctl::MCEdge *>::iterator   edge_iterator_rest;         // vector of edge pointers around rest vertex iterator
    std::vector<vctl::MCEdge *>::iterator   edge_iterator_rest_end;     // vector of edge pointers around rest vertex end iterator
    vctl::MCVertex                          * rest_vertex;              // pointer on rest vertex from actual edge, against starting new vertex


    // cycle of influenced tris
    tri_iterator_end = m_tri_list_influenced.end();
    for (tri_iterator = m_tri_list_influenced.begin(); tri_iterator != tri_iterator_end; ++tri_iterator)
    {
        // set flag of actual tri all vertices 
        (*tri_iterator)->GetVertex(0)->SetFlag(vctl::ENTITY_MARK_FLAG);
        (*tri_iterator)->GetVertex(1)->SetFlag(vctl::ENTITY_MARK_FLAG);
        (*tri_iterator)->GetVertex(2)->SetFlag(vctl::ENTITY_MARK_FLAG);
    }

    // clear vector
    m_edge_list_influenced.clear();

    // take edges around first vertex of collapsed edge
    new_vertex->GetRegisteredEdgeList(m_edge_list);
    edge_iterator_end = m_edge_list.end();

    // remove frag from new vertex
    new_vertex->MaskFlag(vctl::ENTITY_MARK_FLAG);

    // cycle of edges around first vertex of collapsed edge
    for (edge_iterator = m_edge_list.begin(); edge_iterator != edge_iterator_end; ++edge_iterator)
    {
        // take rest vertex from actual edge, against starting new vertex
        assert(*edge_iterator != NULL);
        rest_vertex = (*edge_iterator)->GetRestVertex(new_vertex);

        // test flag of edge rest vertex
        if (rest_vertex->TestFlag(vctl::ENTITY_MARK_FLAG))
        {
            // remove frag from rest vertex
            rest_vertex->MaskFlag(vctl::ENTITY_MARK_FLAG);

            // take edges around rest vertex
            rest_vertex->GetRegisteredEdgeList(m_edge_list_rest);
            edge_iterator_rest_end = m_edge_list_rest.end();

            // cycle of edges around rest vertex
            for (edge_iterator_rest = m_edge_list_rest.begin(); edge_iterator_rest != edge_iterator_rest_end; ++edge_iterator_rest)
            {
                // test of the rest rest vertex to be markup
                assert(*edge_iterator_rest != NULL);
                if ( ! (*edge_iterator_rest)->TestFlag(vctl::ENTITY_MARK_FLAG) )
                {
                    // add actual rest edge into vector of influenced edges
                    m_edge_list_influenced.push_back(*edge_iterator_rest);
                    // markup added edge
                    (*edge_iterator_rest)->SetFlag(vctl::ENTITY_MARK_FLAG);
                }
            }
        }
        else
        {
            // add actual edge into vector of influenced edges
            m_edge_list_influenced.push_back(*edge_iterator);
            // markup added edge
            (*edge_iterator)->SetFlag(vctl::ENTITY_MARK_FLAG);
        }
    }

    // cycle of influenced edges to reevaluate them
    edge_iterator_end = m_edge_list_influenced.end();
    for (edge_iterator = m_edge_list_influenced.begin(); edge_iterator != edge_iterator_end; ++edge_iterator)
    {
        assert(*edge_iterator != NULL);
        // unsort actual edge
        tris.GetEdgeS()->UnSort(*edge_iterator);
        // evaluation of actual edge
        EvaluateEdgeVolume(tris, *edge_iterator);
        // sort actual edge
        tris.GetEdgeS()->Sort(*edge_iterator);
        // mask flag from actual edge
        (*edge_iterator)->MaskFlag(vctl::ENTITY_MARK_FLAG);
    }
}

////////////////////////////////////////////////////////////
//

bool CDecimator::TestVertices(vctl::MCTriS & tris)
{
	// set pointer on first vertex of smoothed mesh
	vctl::MCVertex	* actual_vertex = (tris.GetVerticeS())->GetFirst();

	// cycle of soothed mesh vertices to set them new coordinates
	while (actual_vertex != NULL)
	{
        // test voxel registered entity
        if (actual_vertex->GetRegisteredEdge() != NULL)
            actual_vertex->SetRegisteredEdge(NULL);

		// take next vertex of smoothed mesh and shift point iterator
		actual_vertex = actual_vertex->GetNext();
	}



    // ok
    return true;
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////

