///////////////////////////////////////////////////////////////////////////////
// $Id$
//
// EasyDent
// Navigation and implants planning software for dentistry.
//
// Copyright (c) 2008 by 3Dim Laboratory s.r.o.
//
///////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// Includes

#include "CSmoothing.h"

////////////////////////////////////////////////////////////
//

bool CSmoothing::Smooth(vctl::MCTriS & tris, int repeating, double & lambda, double & k_pb)
{
    vctl::MCVertex                           * actual_vertex;                   // pointer on actual vertex of smoothed mesh
    std::vector<vctl::MCPoint3D>             point_vector;                      // 3D points vector, for vertices new coordinates
    std::vector<vctl::MCPoint3D>::iterator   point_vector_iter;                 // 3D points vector iterator 
    double                                   smooth_factor[2];                  // G. Taubin smoothing scale factors


    // 3D points vector reservation
    point_vector.resize(tris.GetVerticeS()->GetNumber());

    // smoothing factors initialization
    smooth_factor[0] = lambda;
    smooth_factor[1] = 1.0 / (k_pb - (1.0 / lambda));

    // set maximal progress value
    setProgressMax(repeating*2);
    // start function progress
    beginProgress();

    // smoothing cycle
    for (int smooth_iter = 0; smooth_iter < repeating; ++smooth_iter)
    {
        // cycle of two steps G. Taubin smoothing
        for (int step = 0; step < 2; ++step)
        {
            // points vector iterator initialization
            point_vector_iter = point_vector.begin();
            // set pointer on first vertex of smoothed mesh
            actual_vertex = (tris.GetVerticeS())->GetFirst();

            // cycle of smoothed mesh actual vertex
            while (actual_vertex != NULL)
            {
                // calculation of actual vertex new coordinates
                if ( ! SmoothVertex(actual_vertex, *point_vector_iter, smooth_factor[step]))
                    return false;

                // take next vertex of smoothed mesh and shift point iterator
                actual_vertex = actual_vertex->GetNext();
                ++point_vector_iter;
            }

            // points vector iterator initialization
            point_vector_iter = point_vector.begin();
            // set pointer on first vertex of smoothed mesh
            actual_vertex = (tris.GetVerticeS())->GetFirst();

            // cycle of soothed mesh vertices to set them new coordinates
            while (actual_vertex != NULL)
            {
                // set new coordinates of actual vertex
                actual_vertex->SetPoint3D(*point_vector_iter);

                // take next vertex of smoothed mesh and shift point iterator
                actual_vertex = actual_vertex->GetNext();
                ++point_vector_iter;
            }

            // set progress value and test function termination
            if ( ! progress() )
                return true;
        }
    }

    // finish function progress
    endProgress();

    // OK
    return true;
}

///////////////////////////////////////////////////////////
// 

bool CSmoothing::SmoothVertex(vctl::MCVertex * actual_vertex, vctl::MCPoint3D & smoothed_point, double & smooth_factor)
{
    std::vector<vctl::MCVertex *>             neighbor_vertex;               // vector of neighbor vertices for input vertex
    std::vector<vctl::MCVertex *>::iterator   neighbor_vertex_iter;          // vector iterator of neighbor vertices for input vertex
    double                                    dx = 0, dy = 0, dz = 0;         // point coordinates accumulation values


    // initialization of smoothed point
    smoothed_point.SetXYZ(0.0, 0.0, 0.0);

    // take neighbor vertices for input vertex
    if ( ! GetNeighborVertices(actual_vertex, neighbor_vertex, 1))
        return false;

    // cycle of neighbor vertices
    for (neighbor_vertex_iter = neighbor_vertex.begin(); neighbor_vertex_iter != neighbor_vertex.end(); ++neighbor_vertex_iter)
    {
        // point coordinate accumulation
        dx += ((*neighbor_vertex_iter)->GetX() - actual_vertex->GetX());
        dy += ((*neighbor_vertex_iter)->GetY() - actual_vertex->GetY());
        dz += ((*neighbor_vertex_iter)->GetZ() - actual_vertex->GetZ());
    }

    // smoothed coordinates calculation
    smoothed_point.SetX( actual_vertex->GetX() + (smooth_factor * (dx / neighbor_vertex.size())) );
    smoothed_point.SetY( actual_vertex->GetY() + (smooth_factor * (dy / neighbor_vertex.size())) );
    smoothed_point.SetZ( actual_vertex->GetZ() + (smooth_factor * (dz / neighbor_vertex.size())) );

    return true;
}

///////////////////////////////////////////////////////////
// 

bool CSmoothing::GetNeighborVertices(vctl::MCVertex * actual_vertex, std::vector<vctl::MCVertex *> & neighbor_vertex, int neighbor_layers)
{
    std::vector<vctl::MCTri *>                    neighbor_tris;                  // vector of tris registered for actual vertex
    std::vector<vctl::MCTri *>::iterator          neighbor_tris_iter;             // vector iterator of tris registered for actual vertex
    std::vector<vctl::MCTri *>::iterator          neighbor_tris_iter_end;         // vector end iterator of tris registered for actual vertex
    std::vector<vctl::MCVertex *>::size_type      neighbor_vertex_find_size;      // number of neighbor vertices to search
    std::vector<vctl::MCVertex *>::size_type      neighbor_vertex_index;          // index of actual neighbor vertex
    std::vector<vctl::MCVertex *>::size_type      neighbor_vertex_low_limit;      // number of neighbor vertices found before actual neighbor layer cycle
    vctl::MCVertex                                * work_vertex;                  // work vertex pointer 
    vctl::MCVertex                                * ur0, * ur1;                   // pointers on rest vertices of actual tri against actual vertex


    // take tris registered for actual vertex
    actual_vertex->GetRegisteredTriList(neighbor_tris);

    // actual vertex markup
    actual_vertex->SetFlag(vctl::ENTITY_MARK_FLAG);

    // output neighbor vertices vector initialization
    neighbor_vertex.clear();

    // cycle of tris registered for actual vertex
    neighbor_tris_iter_end = neighbor_tris.end();
    for (neighbor_tris_iter = neighbor_tris.begin(); neighbor_tris_iter != neighbor_tris_iter_end; ++neighbor_tris_iter)
    {
        assert(*neighbor_tris_iter != NULL);
        // take rest vertices of actual tri against actual vertex
        (*neighbor_tris_iter)->GetRestEdge(actual_vertex, &ur0, &ur1);
        assert( (ur0 != NULL) && (ur1 != NULL) && (ur0 != ur1));
        // markup of inserted vertex
        ur0->SetFlag(vctl::ENTITY_MARK_FLAG);
        // push first of rest vertex into neighbor vertices vector
        neighbor_vertex.push_back(ur0);
    }

    // initialize searching neighbor vertex low limit
    neighbor_vertex_find_size = 1;

    // cycle of next neighbor layers
    for (int neighbour_layers_index = 1; neighbour_layers_index < neighbor_layers; ++neighbour_layers_index)
    {
        // set searching neighbor vertex low limit
        neighbor_vertex_low_limit = neighbor_vertex_find_size;
        // set number of neighbor vertices
        neighbor_vertex_find_size = neighbor_vertex.size();

        // cycle of neighbor vertices to search
        for (neighbor_vertex_index = neighbor_vertex_low_limit; neighbor_vertex_index < neighbor_vertex_find_size; ++neighbor_vertex_index)
        {
            // take actual tri vertex pointer 
            work_vertex = neighbor_vertex[neighbor_vertex_index];
            assert(work_vertex != NULL);
            // take vertices of actual work tri
            work_vertex->GetRegisteredTriList(neighbor_tris);

            // cycle of tris registered for actual work vertex
            neighbor_tris_iter_end = neighbor_tris.end();
            for (neighbor_tris_iter = neighbor_tris.begin(); neighbor_tris_iter != neighbor_tris_iter_end; ++neighbor_tris_iter)
            {
                assert(*neighbor_tris_iter != NULL);
                // take rest vertices of actual tri against actual vertex
                (*neighbor_tris_iter)->GetRestEdge(actual_vertex, &ur0, &ur1);
                assert( (ur0 != NULL) && (ur1 != NULL) && (ur0 != ur1));
                // test inserted vertex to be markup
                if ( ! ur0->TestFlag(vctl::ENTITY_MARK_FLAG))
                {
                    // markup of inserted vertex
                    ur0->SetFlag(vctl::ENTITY_MARK_FLAG);
                    // push first of rest vertex into neighbor vertices vector
                    neighbor_vertex.push_back(ur0);
                }
            }
        }
    }

    // cycle of found neighbor vertices to remove theirs markup
    for ( std::vector<vctl::MCVertex *>::iterator neighbor_vertex_iter = neighbor_vertex.begin(); neighbor_vertex_iter != neighbor_vertex.end(); ++neighbor_vertex_iter)
        // remove vertex markup
        (*neighbor_vertex_iter)->MaskFlag(vctl::ENTITY_MARK_FLAG);

    // remove markup of input vertex
    actual_vertex->MaskFlag(vctl::ENTITY_MARK_FLAG);

    return true;
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////

