//==============================================================================
/*! \file
 *
 * Author:  Premysl Krsek, krsek@fit.vutbr.cz   \n
 * File:    DecimatorModule.cpp                 \n
 * Date:    2008/10                             \n
 *
 * $Id:$
 */

#include "DecimatorModule.h"
#include "CDecimator.h"

#include <time.h>

// MDSTk
#include <MDSTk/Math/mdsBase.h>

// STL
//#include <sstream>


//==============================================================================
/*
* Global constants.
*/

//! Module description
const std::string MODULE_DESCRIPTION    = "MDSTk based tri mesh Decimating module, implements Volume error metrics polygonal surface simplification.";

//! Additional command line arguments
const std::string MODULE_ARGUMENTS      = "t:p:m";

//! Additional arguments
const std::string MODULE_ARGUMENT_TRIANGLES   = "t";
const std::string MODULE_ARGUMENT_PERCENTS    = "p";
const std::string MODULE_ARGUMENT_MANIFOLD    = "m";

//! Default parameters values
const int DEFAULT_PERCENTS      = 10;
const int DEFAULT_TRIANGLES     = 1000;
const int PERCENTS_MIN          = 1;
const int PERCENTS_MAX          = 99;

//==============================================================================
/*
 * Implementation of the class CDecimatorModule.
 */
CDecimatorModule::CDecimatorModule(const std::string& sDescription)
    : mds::mod::CModule(sDescription)
{
    allowArguments(MODULE_ARGUMENTS);
}

//==============================================================================

CDecimatorModule::~CDecimatorModule()
{
}

//==============================================================================

bool CDecimatorModule::startup()
{
    // Note
    MDS_LOG_NOTE("Module startup");

    // Test existence of input and output channel
    if( getNumOfInputs() != 1 || getNumOfOutputs() != 1 )
    {
        MDS_CERR('<' << m_sFilename << "> Wrong number of input and output channels" << std::endl);
        return false;
    }

    // Parameters 'p'
    m_iPercents = DEFAULT_PERCENTS;
    m_Arguments.value(MODULE_ARGUMENT_PERCENTS, m_iPercents);
    if ( (m_iPercents < PERCENTS_MIN) || (m_iPercents > PERCENTS_MAX) )
    {
        MDS_CERR('<' << m_sFilename << "> Bad 'p' parameter value: type -h for help" << std::endl);
        printUsage();
        return false;
    }

    // Parameter 't'
    if ( m_Arguments.exists(MODULE_ARGUMENT_TRIANGLES) )
    {
        m_bUseTrianglesNumber = true;
        m_iTriangles = DEFAULT_TRIANGLES;
        m_Arguments.value(MODULE_ARGUMENT_TRIANGLES, m_iTriangles);
        if( m_iTriangles < 0 )
        {
            MDS_CERR('<' << m_sFilename << "> Bad 't' parameter value: type -h for help" << std::endl);
            printUsage();
            return false;
        }
    }
    else
        m_bUseTrianglesNumber = false;

    // Parameter 'm'
    if ( m_Arguments.exists(MODULE_ARGUMENT_MANIFOLD) )
        m_bTestManifold = true;
    else
        m_bTestManifold = false;

    // O.K.
    return true;
}

//==============================================================================

bool CDecimatorModule::main()
{
    // Note
    MDS_LOG_NOTE("Module main function");

    // I/O channels
    mds::mod::CChannel *pIChannel = getInput(0);
    mds::mod::CChannel *pOChannel = getOutput(0);

    // Is any input?
    if( !pIChannel->isConnected() )
    {
        return false;
    }

    // Create a new triangle mesh
    vctl::MCTriS                tri_mesh;

    // Wait for data
    if( pIChannel->wait(1000) )
    {
        // Read input triangle mesh from the input channel
        if( readInput(pIChannel, &tri_mesh) )
        {
            // create smoothing algorithm class
            CDecimator              decimator;   

            // smoothing of input tri mesh
            decimator.registerProgressFunc(mds::mod::CProgress::tProgressFunc(this, &CDecimatorModule::ProgressFunction));

            // calculate output tri mesh size
            int output_mesh_size;
            if (m_bUseTrianglesNumber)
                output_mesh_size = m_iTriangles;
            else
                output_mesh_size = (tri_mesh.GetNumber() * m_iPercents) / 100;

            // measure processing time
            time_t timer_1, timer_2;
            time(&timer_1);

            // start smoothing of triangle meshes
            if ( decimator.Reduce(tri_mesh, output_mesh_size) )
            {
                // test output mesh to be manifold
                if (m_bTestManifold)
                {
                    if ( ! tri_mesh.TestManifold() )
                    {
                        MDS_LOG_NOTE("Output mesh IS NOT manifold!");
                        MDS_CERR("Output mesh IS NOT manifold!");
                    }
                    else
                    {
                        MDS_LOG_NOTE("Output mesh IS manifold!");
                        MDS_CERR("Output mesh IS manifold!");
                    }
                }

                // calculate processing time
                time(&timer_2);
                MDS_CERR("\nProcessing time is: " << (timer_2 - timer_1) << "\n");

                // Write smoothed triangle mesh to the output channel
                if( !writeOutput(pOChannel, &tri_mesh) )
                {
                    MDS_CERR('<' << m_sFilename << "> Failed to write the output smoothed MDSTk::VectorEntity triangle mesh" << std::endl);
                }
            }
            else
            {
                MDS_CERR(" Decimation process failed! " << std::endl);
            }

        }
        else
        {
            MDS_CERR('<' << m_sFilename << "> Failed to read input MDSTk::VectorEntity Triangle mesh" << std::endl);
            return false;
        }
    }
    else
    {
        MDS_LOG_NOTE("Wait timeout");
    }

    // Returning true means to continue processing the input channel
    return true;
}

//==============================================================================

void CDecimatorModule::shutdown()
{
    // Note
    MDS_LOG_NOTE("Module shutdown");
}

//==============================================================================

void CDecimatorModule::writeExtendedUsage(std::ostream& Stream)
{
    MDS_CERR(std::endl);
    MDS_CERR("Extended usage: [-t iValue] [-p iValue] [-m]" << std::endl);
    MDS_CERR("Options:" << std::endl);
    MDS_CERR("  -t    Number of triangles of output tri mesh." << std::endl);
    MDS_CERR(std::endl);
    MDS_CERR("  -p    Size of output tri mesh in percents to input mesh." << std::endl);
    MDS_CERR("        Default value is " << DEFAULT_PERCENTS << std::endl);
    MDS_CERR("        Percents value range is from " << PERCENTS_MIN << " to " << PERCENTS_MAX << std::endl);
    MDS_CERR("  -m    Test output tri mesh to be manifold." << std::endl);
}

//==============================================================================
/*
 * Function main() which creates and executes the console application.
 */
int main(int argc, char *argv[])
{
    // Creation of the module using smart pointer
    CDecimatorModulePtr spModule(new CDecimatorModule(MODULE_DESCRIPTION));

    // Initialize and execute the module
    if( spModule->init(argc, argv) )
    {
        spModule->run();
    }

    // Console application finished
    return 0;
}

