/*------------------------------------------------------------------*/
/* EvoCCs_boradcast						    */
/*   Tool for evolutionary design of broadcast-based                */
/*   collective communications on wormhole networks                 */
/*                                                                  */
/* Author: Jiri Jaros						    */
/*	   jarosjir@fit.vutbr.cz                                    */
/* 	   Brno University of Technology			    */
/* 	   GPL Licence, 2010					    */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
/* MMB.cpp - Implementation of Many-to-Many broadcast communications*/
/*         - implementation file                                    */
/*------------------------------------------------------------------*/

#pragma hdrstop

#include "MNB.h"
#include <iostream>
#include <fstream>
#include <algorithm>
#include <stdlib.h>
#include <sstream>
#include <values.h>
#include <iterator>

#include "GeneticCore.h"
#include "Topology.h"

using namespace std;


//--  parameter initialization --//
const int FileNameParNum  = 1;
const int CommStepsParNum = 2;
const int FlushStatisticsInterval = 500;
const int FlushSolutionInterval = FlushStatisticsInterval * 10;

extern Random RandomGenerator;
extern Topology topology;
extern unsigned int MaxExtraSteps;
extern unsigned int SurroundSize;
int CommSteps;
int BestGeneration = -1;



//--  Container for all source destination pahts --//
vector< vector< Topology::Paths> > AllProcPaths;
vector< vector< Topology::Paths> > AllProcPathsFromFile;


struct TPathStruct{
  int src;
  int dst;
  int index;
};



/*------------------------------------------------------------------*/
/*------------------------- Auxiliary function ---------------------*/
/*------------------------------------------------------------------*/

/**
 *  SavePathsToFile - Save all shortest paths to file 
*/
void SavePathsToFile(string FileName);


/**
 *  GetAllPaths - Find all shortest paths
*/
void GetAllPaths(string Filename, int NumOfSteps);

/**
 *  NormalizeMNB - Normalization of chromozome for AAB communication
*/
int NormalizeMNB(Chromozome &chr);

/**
 *  LoadPathsFromFile - Load shortest paths from File
*/
void LoadPathsFromFile(string FileName);

/**
 *  GetSourceNode - Get initiator of OAB communciation
*/
int GetSourceNode(int Index);
  // Get initiator of OAB communciation

/**
 *  GetDestinationNode - Get termination node of Point-to-Point message
*/
int GetDestinationNode(int Index);
  
/**
 *  GetReceiverIndex - Where is the receiver
*/
unsigned int GetReceiverIndex(int Receiver, int WhereIthOABStart);

/**
 *  GetInvolvedNode - Get node which can become a distributor for OAB
*/
int GetInvolvedNode(int Index);
   

/**
 *  PureShortestPaths - Pure shortest paths - leave only paths from surround
*/
void PureShortestPaths(int SurroundSize);

/**
 *  GetPossibleSendersForRecievers - find possible source
*/
void GetPossibleSendersForRecievers(vector <int> &PossibleSenders,
        vector <int> WhoHaveMessage, int Receiver);
  
/**
 *  GetAllPossibleSendersForRecievers - intersection of Involved and Reachable nodes
*/
void GetAllPossibleSendersForRecievers(vector <int> &PossibleSenders, int Receiver);

/**
 *  MMBPrintChromozome - print MNBSchedule to stderr
*/
void MMBPrintChromozome(Chromozome& chr);

/**
 *  PrintConflictPosition - print positions of conflicts
*/
string PrintConflictPosition(Chromozome& chr);




/*____________________________________________________________________________*/
/*____________________________Implementation__________________________________*/
/*____________________________________________________________________________*/


// end of GetAllPaths
//--------------------------------------------------------------------------------

/**
 *  GetAllPaths - Find all paths between any source-destination pair
*/
void GetAllPaths(string Filename, int NumOfSteps)
{


int ProcCount = topology.getProcCount();

  AllProcPaths.resize(ProcCount);
  
  for (int i=0; i< ProcCount; i++) { // init
     AllProcPaths[i].resize(ProcCount);

     vector<int> ZeroPath;
     ZeroPath.push_back(i);
     AllProcPaths[i][i].push_back(ZeroPath);
  }


    for (int src = 0; src < ProcCount; src++) {
      // From each source search only for paths to destinations with a higher
      // index.
      for (int dst = 0; dst < ProcCount; dst++) {
        // find all shortest paths from src dst
        if (!topology.findShortestPaths(src, dst, AllProcPaths[src][dst])) {
          // No path with length <= configData.commSteps was found.
          cerr << "Error: No path from node " << src <<
              " to node " << dst << " with length less or equal to " <<
              CommSteps << " was found." << endl;
          cerr << "You should increase number of commuication steps." << endl;
          exit(1);
        }
      }
      // For destinations with lower index than source, just reverse already
      // found paths from dst to src.
    }

    PureShortestPaths(SurroundSize);
} // end of GetAllPaths
//--------------------------------------------------------------------------------

/**
 *  LoadPathsFromFile -load pahts from a file - useful for debuging
*/
void LoadPathsFromFile(string FileName){

  cout << "Start to read file " << FileName << endl;
  ifstream dataFile(FileName.c_str()); // data will be read from this file
  if (!dataFile) { // file open failed
    throw string("Unable to open file '") + FileName + "'";
  }


  int ProcessorCount;
  // read from input file one line at a time
  string line;
  // state of an automaton reading input file

  getline(dataFile, line);
  istringstream lineStream(line.c_str());
  lineStream >> ProcessorCount;
    // Nacteni poctu procesoru
  topology.setProcCount(ProcessorCount);

  AllProcPaths.resize(ProcessorCount);
  for (int i=0; i< ProcessorCount; i++)
    AllProcPaths[i].resize(ProcessorCount);

  cout << "I've read the number of processors - "  << ProcessorCount << endl;

  while (getline(dataFile, line)) {
    string::size_type pos;
    // find and remove comments

    pos = line.find_first_not_of(" \t"); // find first no space character
    if (pos > 0 && pos != string::npos) { // spaces found
      line.erase(0, pos); // remove spaces at the beginning
    }
    pos = line.find_last_not_of(" \t"); // find last no space character
    if (pos < (line.size() - 1) && pos != string::npos) { // spaces found
      line.erase(pos+1); // remove spaces at the end
    }

    int SourceProcessor;
    int DestProcessor;
    switch (line[0]){
      case 'S' : {
                  line.erase(0, 1);
                  istringstream lineStream(line.c_str());
                  lineStream >> SourceProcessor;
                  break;
                   }
      case 'D' : {
                  line.erase(0, 1);
                  istringstream lineStream(line.c_str());
                  lineStream >> DestProcessor;
                  break;
                   }
      default : {
        vector <int> Path;
        istringstream lineStream(line.c_str());
        int PathValue;
        while (!lineStream.eof()){
          lineStream >> PathValue;
          Path.push_back(PathValue);
        }

        AllProcPaths[SourceProcessor][DestProcessor].push_back(Path);
        break;
      }
   } /* case */
  }
}// end of LoadPathsFromFile
//--------------------------------------------------------------------------------



/**
 *  OABOnInitAlgorithm -Initialize algorithm, start logs, for Broadcast problems
*/
void OABOnInitAlgorithm(GeneticCore& GA, int argc, char** argv){

 string FileName = argv[FileNameParNum];
 CommSteps = atoi(argv[CommStepsParNum]);


 try {
    topology.readFile(FileName);
    topology.printNeighbors(cout);
    cerr << "Communication steps =  " << CommSteps << endl;
    cerr << "Surrouding size = " << SurroundSize << endl;


    cout << "--- Transmiters ---" << endl;
    cerr << "--- Transmiters ---" << endl;
    cout << "Transmiter count = " << topology.Transmiters.size() << endl;
    cerr << "Transmiter count = " << topology.Transmiters.size() << endl;

    for (unsigned int i=0; i< topology.Transmiters.size(); i++){
     cout << topology.Transmiters[i] <<" , ";
     cerr << topology.Transmiters[i] <<" , ";
    }

    if (topology.Transmiters.size() == 0) throw string("Number of OAB initiator is 0");


    cout << endl <<"--- Receivers ---" << endl;
    cerr << endl <<"--- Receivers ---" << endl;
    cout << "Receivers count = " << topology.Receivers.size() << endl;
    cerr << "Receivers count = " << topology.Receivers.size() << endl;
    for (unsigned int i=0; i< topology.Receivers.size(); i++){
     cout << topology.Receivers[i] <<" , ";
     cerr << topology.Receivers[i] <<" , ";
    }

    if (topology.Receivers.size() == 0) throw string("Number of OAB receivers is 0");


    cout << endl <<"--- Involved nodes ---" << endl;
    cerr << endl <<"--- Involved nodes ---" << endl;
    cout << "Involved nodes count = " << topology.InvolvedNodes.size() << endl;
    cerr << "Involved nodes count = " << topology.InvolvedNodes.size() << endl;
    for (unsigned int i=0; i< topology.InvolvedNodes.size(); i++){
     cout << topology.InvolvedNodes[i] <<" , ";
     cerr << topology.InvolvedNodes[i] <<" , ";
    }

    cout << endl;
    cerr << endl;
  } catch (string e) {
    cerr << "Error while loading topology data: " << e << endl;
    exit(1);
  }

  GA.SetChromozomeLength(topology.GetCommunicatingNodeCount() * 3);

  cout << "Chromozome size = " << GA.GetChromozomeLength() << endl;
  cout << "FileName = " << FileName << endl;
  cout << "Communication steps =  " << CommSteps << endl;


  GetAllPaths(FileName, CommSteps);
  cout << "Processor count     =  " << topology.getProcCount() << endl;


}// end of OABOnInitAlgorithm
//--------------------------------------------------------------------------------


/**
 *  MNBInitChromozome -Broadcast chromosome initialization
*/
void MNBInitChromozome(Chromozome& chr){

  int InvolvedCount   = topology.InvolvedNodes.size();

   /* - initialization of MNB chromozome */
  for (int Gene = 0; Gene < chr.Length(); Gene += 3){


    int OABDestination = GetDestinationNode(Gene);

    vector <int> AllPossibleSenders;
    GetAllPossibleSendersForRecievers(AllPossibleSenders,OABDestination);

    int OABSource = AllPossibleSenders[RandomGenerator.IntRandom(AllPossibleSenders.size())];
      // - from whom I can accept message --//

    chr[Gene] = OABSource;
      // who is source
    chr[Gene+1] = RandomGenerator.IntRandom(AllProcPaths[OABSource][OABDestination].size());
      // which way from source
    chr[Gene+2] = 0; //RandomGenerator.IntRandom(CommSteps);
      // step number
      
 } // - end of Sender -//


}// end of MNBInitChromozome
//--------------------------------------------------------------------------------


/**
 *  NormalizeMNB -Broadcast restoration heuristic
*/
int NormalizeMNB(Chromozome &chr){


  int Conflicts = 0;
  vector <int> WhoHaveMessage;  // index of node who already have a message
  vector <int> WhoReceive; 	// index of node that recieve message in some step

  int OABSourceCount    = topology.Transmiters.size();
  int OABReceiversCount = topology.Receivers.size();


  for (int OABSourceIndex = 0; OABSourceIndex < OABSourceCount; OABSourceIndex ++) {

    WhoHaveMessage.clear();
    WhoReceive.clear();

      //- Who is OAB source -//
    int OABSourceNode    = topology.Transmiters[OABSourceIndex];

     //- index of OABs -//
    int IndexOfOABSource = ((OABSourceIndex * OABReceiversCount) + OABSourceIndex) * 3;
    int IthOABStart      = (OABSourceIndex * OABReceiversCount)*3;
    int IthOABFinish     = IthOABStart + (OABReceiversCount *3);




    //- InitOABSource -//
    if (topology.IsReceiver(OABSourceNode)) {
      chr[IndexOfOABSource] = OABSourceNode;    // Seneder
      chr[IndexOfOABSource + 1] = 0;
      chr[IndexOfOABSource + 2] = 0;
    }


    // OAB normalization
    WhoHaveMessage.push_back(OABSourceNode); //Initializator of OAB

    int MaxCommSteps = CommSteps;
    int StepNumber = 0;


    while (StepNumber < MaxCommSteps){
      WhoReceive.clear();


      for (int Gene = IthOABStart; Gene <IthOABFinish; Gene +=3){ // Who recieve in step
        if (chr[Gene+2] == StepNumber) WhoReceive.push_back(GetDestinationNode(Gene));

           // -- Generated Step > MaxCommSteps --//
        if (chr[Gene+2] >= MaxCommSteps) MaxCommSteps = chr[Gene+2] + 1;
      }



      for (unsigned int WhoReceiveIndex = 0; WhoReceiveIndex < WhoReceive.size(); WhoReceiveIndex++){

        int ReceiverIndex = GetReceiverIndex(WhoReceive[WhoReceiveIndex], IthOABStart);
        int ActualReceiver = WhoReceive[WhoReceiveIndex];
          // - Where is Reciever in Genome - //

        vector<int> PossibleSenders;
        GetPossibleSendersForRecievers(PossibleSenders,WhoHaveMessage,ActualReceiver);



        if (PossibleSenders.size() == 0){
          //-- Somebody in surrouding has not a message --//
          chr[ReceiverIndex+2] = StepNumber +1;
          WhoReceive[WhoReceiveIndex] = -1;

          if ((StepNumber + 1) >= MaxCommSteps) MaxCommSteps++;

        }
        else
        { //-- Somebody has a message --//

            if (!topology.IsNodeInSurroundOf(ActualReceiver, chr[ReceiverIndex])){
                 //-- if node is not in surrouding, then change it--//
               int NewSource = PossibleSenders[RandomGenerator.IntRandom(PossibleSenders.size())];
               chr[ReceiverIndex] = NewSource;
            }else
               //-- Node is in surrouding --//

               if (find(PossibleSenders.begin(), PossibleSenders.end(), chr[ReceiverIndex])
                  == PossibleSenders.end()) { // Seneder doesn't have a message

                 int NewSource = PossibleSenders[RandomGenerator.IntRandom(PossibleSenders.size())];

                 chr[ReceiverIndex] = NewSource;
               } // - if
            } // - else in sur
        }

      if (StepNumber == 0) WhoHaveMessage.clear(); //eliminating twice insertion of OAB source
      for (unsigned int i=0; i<WhoReceive.size(); i++) // Add new nodes
          if (WhoReceive[i] != -1) WhoHaveMessage.push_back(WhoReceive[i]);
      StepNumber ++;
    }

   //- correction of shortest paths -//

    for (int Gene = IthOABStart; Gene < IthOABFinish; Gene += 3){
     if (AllProcPaths[chr[Gene]][GetDestinationNode(Gene)].size() == 0) {
      cout << " Fatal error in function: Normalize MNB " << endl;
      cout << "OAB. = " << Gene / 24 + 1  << ", Gene = " << Gene <<endl;
      cout <<" Source = " << chr[Gene] << ", Dest = " << GetDestinationNode(Gene) << endl;
      cout <<" Step Number = " << chr[Gene + 2] << endl;
      cout << endl;
      cout << chr.ToString() << endl;
      exit (0);

     }
     chr[Gene + 1] = chr[Gene + 1] % AllProcPaths[chr[Gene]][GetDestinationNode(Gene)].size();
    }
  }


  //-- max CommStep violation --//

  int StepGene = 2;
  Conflicts = 0;
  while (StepGene < chr.Length()){
    int ViolationSize =  chr[StepGene] - (CommSteps -1);

    if (ViolationSize >0)
    Conflicts = Conflicts + ViolationSize;

    StepGene +=3;
  }


  return Conflicts;

}// end of NormalizeMNB
//--------------------------------------------------------------------------------




//----------------------------------------------------------------------------//
//--           New normalization with proportional Source assignment        --//
//----------------------------------------------------------------------------//


/**
 *  InitUssageOfSource -Init source usage - K-port restriction heuristic
*/
void InitUssageOfSource(vector <int>& Ussage){

  Ussage.clear();
  for (unsigned int i = 0; i < topology.InvolvedNodes.size(); i++){
    Ussage.push_back(0);
  }
} // end of InitUssageOfSource
//--------------------------------------------------------------------------------


/**
 *  GenerateRandomizedOABNormalizationSequence -randomize OAB sequence
*/
void GenerateRandomizedOABNormalizationSequence(vector <int>& Seq, int SequenceSize){

  vector <int> PossibleNumbers;

  for (int i=0; i< SequenceSize; i++)
    PossibleNumbers.push_back(i);

  Seq.clear();


  for (int i = 0; i < SequenceSize; i ++) {
    int rnd   = RandomGenerator.IntRandom(PossibleNumbers.size());
    Seq.push_back(PossibleNumbers[rnd]);
    PossibleNumbers.erase(PossibleNumbers.begin() + rnd);
  }

}// end of GenerateRandomizedOABNormalizationSequence
//--------------------------------------------------------------------------------


/**
 *  GetIndexOfInvolvedNode - index transformation function
*/
unsigned int GetIndexOfInvolvedNode(int Node){

  int Index = MAXINT;

  for (unsigned int i=0; i<topology.InvolvedNodes.size(); i++)
    if (topology.InvolvedNodes[i] == Node){
       Index = i;
       break;
    }

  if (Index == MAXINT) return MAXINT;
  else return Index;
}// end of GetIndexOfInvolvedNode
//--------------------------------------------------------------------------------



/**
 *  SelectGoodSourceAndSetUsage - Proportionally select good source for the message
*/
int SelectGoodSourceAndSetUssage(vector <int>& Ussage, vector<int>& WhoHaveMessage){


  int WhoHaveMessageSize = WhoHaveMessage.size();
  int FirstSource  = WhoHaveMessage[RandomGenerator.IntRandom(WhoHaveMessageSize)];
  int SecondSource = WhoHaveMessage[RandomGenerator.IntRandom(WhoHaveMessageSize)];

  unsigned int IndexOfFirstSource  = GetIndexOfInvolvedNode(FirstSource);
  unsigned int IndexOfSecondSource = GetIndexOfInvolvedNode(SecondSource);

  int SelectedSource = FirstSource;

  if (Ussage[IndexOfFirstSource] > Ussage[IndexOfSecondSource]) {
      SelectedSource = SecondSource;
      Ussage[IndexOfSecondSource]++;
  } else {
      Ussage[IndexOfFirstSource]++ ;
  }

 return SelectedSource;
} // end of SelectGoodSourceAndSetUssage
//--------------------------------------------------------------------------------



/**
 *  NormalizeProportionalMNB - K-port restriction restoration
*/
void NormalizeProportionalMNB(Chromozome &chr){

  vector <int> WhoHaveMessage;  // index of node who already have a message
  vector <int> WhoReceive; // index of node that recieve message in some step


  vector<int> UssageOfSource;   // Vector of involved nodes ussage - how many message they resend --//
  vector<int> OABNormalizationSequence;

  InitUssageOfSource(UssageOfSource);


  int OABSourceCount    = topology.Transmiters.size();
  int OABReceiversCount = topology.Receivers.size();

  GenerateRandomizedOABNormalizationSequence(OABNormalizationSequence,OABSourceCount);



  for (int OABSourceIndexSeq = 0; OABSourceIndexSeq < OABSourceCount; OABSourceIndexSeq ++) {


    int OABSourceIndex = OABNormalizationSequence[OABSourceIndexSeq];

    WhoHaveMessage.clear();
    WhoReceive.clear();

      //- Who is OAB source -//
    int OABSourceNode    = topology.Transmiters[OABSourceIndex];

     //- index of OABs -//
    int IndexOfOABSource = ((OABSourceIndex * OABReceiversCount) + OABSourceIndex) * 3;
    int IthOABStart      = (OABSourceIndex * OABReceiversCount)*3;
    int IthOABFinish     = IthOABStart + (OABReceiversCount *3);


    //- InitOABSource -//
    chr[IndexOfOABSource] = OABSourceNode;    // Seneder
    chr[IndexOfOABSource + 1] = 0;
    chr[IndexOfOABSource + 2] = 0;


    // OAB normalization
    WhoHaveMessage.push_back(OABSourceNode); //Initializator of OAB

    for (int StepNumber = 0; StepNumber < CommSteps; StepNumber++) {
      WhoReceive.clear();

      for (int Gene = IthOABStart; Gene <IthOABFinish; Gene +=3){ // Who recieve in step
        if (chr[Gene+2] == StepNumber) WhoReceive.push_back(GetDestinationNode(Gene));
      }

      for (unsigned int WhoReceiveIndex = 0; WhoReceiveIndex < WhoReceive.size(); WhoReceiveIndex++){

          // - Where is Reciever in Genome - //
        int ReceiverIndex = GetReceiverIndex(WhoReceive[WhoReceiveIndex], IthOABStart);



        if (find(WhoHaveMessage.begin(), WhoHaveMessage.end(), chr[ReceiverIndex])
            == WhoHaveMessage.end()) { // Seneder doesn't have a message

           int NewSource = SelectGoodSourceAndSetUssage(UssageOfSource,WhoHaveMessage);
           chr[ReceiverIndex] = NewSource;


        } // - if
      } // - for

      for (unsigned int i=0; i<WhoReceive.size(); i++) // Add new nodes
        WhoHaveMessage.push_back(WhoReceive[i]);
    }

   //- correction of shortest paths -//

    for (int Gene = IthOABStart; Gene < IthOABFinish; Gene += 3)
     chr[Gene + 1] = chr[Gene + 1] % AllProcPaths[chr[Gene]][GetDestinationNode(Gene)].size();
  }


} // end of NormalizeProportionalMNB
//--------------------------------------------------------------------------------


//----------------------------------------------------------------------------//
//--                    end of NEW normalization                            --//
//----------------------------------------------------------------------------//



/**
 *  ComparePaths - Returns number of conflicts in this two ways
*/
int ComparePaths(vector<int>& Path1, vector<int>& Path2){
  int result = 0;

  for (int i = 0 ; i< ((signed) Path1.size()) -1; i++)
    for (int j = 0 ; j< ((signed) Path2.size()) -1; j++)
      if ((Path1[i] == Path2[j]) && (Path1[i+1] == Path2[j+1])) result++;

 return result;
}// end of ComparePaths
//-------------------------------------------------------------------------------- 


/**
 *  MNBFitnessFunction- Fitness function implementation
*/
double MNBFitnessFunction(Chromozome& chr){

  int conflicts= NormalizeMNB(chr); // Normalization of chromozome;
  vector<struct TPathStruct> PathsInSameStep; // Name Paths in the same step



  for (int Step=0; Step<CommSteps; Step++){
    PathsInSameStep.clear();

    for (int Gene = 0; Gene< chr.Length(); Gene += 3) // Find which paths are in the same step
      if (chr[Gene+2] == Step) {
         struct TPathStruct PathStruct;
         PathStruct.src = chr[Gene];
         PathStruct.dst = GetDestinationNode(Gene);
         PathStruct.index = chr[Gene+1];

         PathsInSameStep.push_back(PathStruct);
      }

   
   
    for (unsigned int i=0; i< PathsInSameStep.size(); i++){ //First Paths
      for (unsigned int j = i+1; j< PathsInSameStep.size(); j++){ // Second Paths

          conflicts += ComparePaths(AllProcPaths[PathsInSameStep[i].src][PathsInSameStep[i].dst][PathsInSameStep[i].index],
                                    AllProcPaths[PathsInSameStep[j].src][PathsInSameStep[j].dst][PathsInSameStep[j].index]);
      }
    }
  }

 return (-1) * conflicts;

}// end of MNBFitnessFunction
//-------------------------------------------------------------------------------- 


/**
 *  MNBMutation- Mutation and local search
*/
void MNBMutation(Chromozome& chr){

  int OABReceiversCount = topology.Receivers.size();
  int OABSourceIndex    = RandomGenerator.IntRandom(topology.Transmiters.size());
  int OABSourceNode     = topology.Transmiters[OABSourceIndex];
    //- Which OAB will be modified --//

  int IthOABStart    = (OABSourceIndex * OABReceiversCount) * 3;
  int IthOABFinish   = IthOABStart + (OABReceiversCount *3);



  int Pos = IthOABStart + RandomGenerator.IntRandom(topology.Receivers.size() * 3);
  
  vector <int> WhoHaveMessage;

  switch (Pos % 3 ){
    case 0 : {

      WhoHaveMessage.push_back(OABSourceNode);
      for (int i=IthOABStart; i<IthOABFinish; i+=3)
      if (chr[i+2] < chr[Pos+2]) WhoHaveMessage.push_back(GetDestinationNode(i));

        vector<int> PossibleSenders;
        GetPossibleSendersForRecievers(PossibleSenders,WhoHaveMessage,GetDestinationNode(Pos));
        if (PossibleSenders.size() == 0){
          chr[Pos] = WhoHaveMessage[RandomGenerator.IntRandom(WhoHaveMessage.size())];
        }else {
          chr[Pos] = PossibleSenders[RandomGenerator.IntRandom(PossibleSenders.size())];
        }

    break;
    }
  case 1 : chr[Pos] = RandomGenerator.IntRandom(AllProcPaths[chr[Pos-1]] [GetDestinationNode(Pos -1)].size());break;
  case 2 : chr[Pos] = RandomGenerator.IntRandom(CommSteps); break;
/*  case 2 : {
             double Variance = (CommSteps/3 > 1) ? CommSteps/3 :1;
             int GaussShift = RandomGenerator.IntGaussRandom(CommSteps, Variance,CommSteps/2);
             if (GaussShift >0) cout <<" OK= "<< GaussShift;
              chr[Pos] = abs(chr[Pos] + GaussShift) % CommSteps;            
	   break;
  	}*/

 }

} // end of MNBMutation
//-------------------------------------------------------------------------------- 


/**
 *  OABOnFinishGeneration- Logs on generation completion
*/
void   OABOnFinishGeneration(Statistics Stat, int ActGeneration){
  if ((ActGeneration % FlushStatisticsInterval) == 0) {
    cout << (ActGeneration) <<"th generation "<< endl;
    cout << "MaxFit = " <<(  Stat.Max() )<< "  MinFit = "<< (Stat.Min()) << "  AvgFit = "<<
              ( Stat.Avg()) << "  Diver = " << (Stat.Diver()) <<endl;


         }
    if (Stat.Max() == 0)  {
      BestGeneration = ActGeneration;
      OABOnFinish(Stat, ActGeneration);
      exit(EXIT_SUCCESS);
  }
}// end of OABOnFinishGeneration
//-------------------------------------------------------------------------------- 



/**
 *   MMBPrintChromozome- Print MMB chromozome in a form of schedule
*/
void MMBPrintChromozome(Chromozome& chr){


   //-- Find Max Comm step --//
  int MaxCommStep = CommSteps;

  int StepGene = 2;
  while (StepGene < chr.Length()){
    if (chr[StepGene] >= MaxCommStep) MaxCommStep = chr[StepGene]+1;
    StepGene +=3;
  }


 vector<struct TPathStruct> PathsInSameStep; // Name Paths in the same step
 int OABReceiversCount = topology.Receivers.size();

 for (int Step=0; Step<MaxCommStep; Step++){
   cerr << endl <<"_____________________________________________________" << endl;
   cerr <<"--------" <<Step << ". step of comunication ---------" << endl;
   for (unsigned int OABSourceIndex = 0; OABSourceIndex < topology.Transmiters.size(); OABSourceIndex++){

     int OABSourceNode = topology.Transmiters[OABSourceIndex];
     cerr << "  ----------------------------" << endl;
     cerr << "  -- OAB source = " << OABSourceNode << " --" << endl;

     PathsInSameStep.clear();


    int IthOABStart    = (OABSourceIndex * OABReceiversCount) * 3;
    int IthOABFinish   = IthOABStart + (OABReceiversCount *3);


     for (int Gene = IthOABStart; Gene < IthOABFinish; Gene += 3) // Find which paths are in the same step
      if (chr[Gene+2]== Step) {
         struct TPathStruct PathStruct;
         PathStruct.src = chr[Gene];
         PathStruct.dst = GetDestinationNode(Gene);
         PathStruct.index = chr[Gene+1];

         PathsInSameStep.push_back(PathStruct);
      } //-- for (int gene... //

      for (unsigned int i=0; i< PathsInSameStep.size(); i++){ //Path
        int src = PathsInSameStep[i].src;
        int dst = PathsInSameStep[i].dst;
        int PathIndex = PathsInSameStep[i].index;

        cerr << "  Message from "<< src << " -> " << dst << ": " << endl;

        for (unsigned int j =0; j< (AllProcPaths[src][dst][PathIndex].size()); j++)
          cerr << AllProcPaths[src][dst][PathIndex][j] << ", ";

        cerr << endl;
     }
  }
  }
}// end of MMBPrintChromozome
//-------------------------------------------------------------------------------- 


/**
 *   OABOnFinish- Logs on finish algorithm
*/
void   OABOnFinish(Statistics Stat, int ActGeneration){

  if (Stat.Max() == 0)  cout << "End of algorithm - Global optima was found" << endl;
  else cout << "End of algorithm - Stuck in local optima" << endl;
  
  cerr << "Global optima was reached in generation no.  = " << BestGeneration<< endl;
  cerr << "The best solution: " << (Stat.BestChromozome()).ToString()<< endl;
  cerr << "Fitness function = " << Stat.BestChromozome().Fitness() << endl;
  MMBPrintChromozome(Stat.BestChromozome());
}// end of OABOnFinish
//-------------------------------------------------------------------------------- 



/**
 *   GetSourceNode- Get initiator of OAB communciation
*/
 int GetSourceNode(int Index){
 return (topology.Transmiters[(Index / (topology.Receivers.size() * 3))]);
}// end of GetSourceNode
//-------------------------------------------------------------------------------- 

/**
 *   GetDestinationNode - Get termination node of Point-to-Point message
*/
int GetDestinationNode(int Index) {

 return (topology.Receivers[(Index / 3) % topology.Receivers.size()]);
}// end of GetDestinationNode
//-------------------------------------------------------------------------------- 


/**
 *   GetInvolvedNode - Get node which can be middle node for OAB
*/
int GetInvolvedNode(int Index){
 return (topology.InvolvedNodes[Index]);
} // end of GetInvolvedNode
//-------------------------------------------------------------------------------- 


/**
 *    GetReceiverIndex  
*/
unsigned int GetReceiverIndex(int Receiver, int WhereIthOABStart){

 unsigned int IndexOfReceiver = MAXINT;
 for (unsigned int i = 0; i < topology.Receivers.size(); i++){
   if (topology.Receivers[i] == Receiver) {
     IndexOfReceiver = i;
     break;
   }
 }

 if (IndexOfReceiver == MAXINT) return MAXINT;
 else return (WhereIthOABStart + IndexOfReceiver*3);
}// end of GetReceiverIndex
//-------------------------------------------------------------------------------- 



/**
 *    SavePathsToFile  - Save all shortest paths to file 
*/
void SavePathsToFile(string FileName){

 int ProcCount = topology.getProcCount();
 ofstream Fout(FileName.c_str());

  for (int Source = 0; Source < topology.getProcCount(); Source ++){
     Fout << endl << "S = " << Source << endl;
    for (int Destination = 0; Destination < topology.getProcCount(); Destination ++){
       Fout << "D = " << Destination << endl;

       for (int Path = 0; Path < AllProcPaths[Source][Destination].size(); Path ++){
          Fout << "  " ;
          for (int PathEl = 0; PathEl < AllProcPaths[Source][Destination][Path].size(); PathEl++){
            Fout << AllProcPaths[Source][Destination][Path][PathEl] << " ";
          }
          Fout << endl;
       }
    }
  }
}// end of SavePathsToFile
//-------------------------------------------------------------------------------- 



/**
 *    PureShortestPaths  - Pure shortest paths - leave only paths from surround
*/
void PureShortestPaths(int SurroundSize){

  int ProcCount = topology.getProcCount();

  for (int Source = 0; Source < ProcCount; Source ++){
    for (int Destination = 0; Destination < ProcCount; Destination ++){

     unsigned int Path = 0;
     while (Path < AllProcPaths[Source][Destination].size()) {
       if ( AllProcPaths[Source][Destination][Path].size() > SurroundSize+1) {
         AllProcPaths[Source][Destination].erase(AllProcPaths[Source][Destination].begin()+Path); //
       }
       else {
         Path++;
       }
     } //-- Path --//

    } //-- Destination --//
  } //-- Source --//

    //-- Set Surrouding --//
  topology.SetReachableNodes(AllProcPaths);
}// end of SavePathsToFile
//-------------------------------------------------------------------------------- 



/**
 *    GetPossibleSendersForRecievers  - Collect possible distributors for a message
*/
void GetPossibleSendersForRecievers(vector <int> &PossibleSenders,
        vector <int> WhoHaveMessage, int Receiver){

  PossibleSenders.clear();

  for (int Source = 0; Source < WhoHaveMessage.size(); Source++){

    if (find(topology.ReachableNodes[Receiver].begin(),topology.ReachableNodes[Receiver].end(),
             WhoHaveMessage[Source]) != topology.ReachableNodes[Receiver].end()) { // Sender is in the surrouding
    PossibleSenders.push_back(WhoHaveMessage[Source]);
    } //-if
  }//- for
}// end of GetPossibleSendersForRecievers
//-------------------------------------------------------------------------------- 

/**
 *    GetAllPossibleSendersForRecievers  - intersection of Involved and Reachable nodes
*/
void GetAllPossibleSendersForRecievers(vector <int> &PossibleSenders, int Receiver){

  PossibleSenders.clear();

  for (int Source = 0; Source < topology.InvolvedNodes.size(); Source++){

    if (find(topology.ReachableNodes[Receiver].begin(),topology.ReachableNodes[Receiver].end(),
             topology.InvolvedNodes[Source]) != topology.ReachableNodes[Receiver].end()) { // Sender is in the surrouding
    PossibleSenders.push_back(topology.InvolvedNodes[Source]);
    } //-if
  }//- for
}// end of GetAllPossibleSendersForRecievers
//-------------------------------------------------------------------------------- 



/**
 *    PrintConflictPosition  - Print conflicts positions
*/
string PrintConflictPosition(Chromozome& chr){


  //-- Validation conflicts --//
  int StepGene = 2;
  int Conflicts = 0;

  while (StepGene < chr.Length()){
    int ViolationSize =  chr[StepGene] - (CommSteps -1);

    if (ViolationSize >0)
    Conflicts = Conflicts + ViolationSize;

    StepGene +=3;
  }

  ostringstream LogStream;
  LogStream << "Violation conflicts = " << Conflicts << endl;
  
  
  //-- paths conflicts --//


  vector<struct TPathStruct> PathsInSameStep; // Name Paths in the same step



  for (int Step=0; Step<CommSteps; Step++){
    PathsInSameStep.clear();
    LogStream  << "---Step = " << Step << "---"<<endl;

    for (int Gene = 0; Gene< chr.Length(); Gene += 3) // Find which paths are in the same step
      if (chr[Gene+2] == Step) {
         struct TPathStruct PathStruct;
         PathStruct.src = chr[Gene];
         PathStruct.dst = GetDestinationNode(Gene);
         PathStruct.index = chr[Gene+1];

         PathsInSameStep.push_back(PathStruct);
      }

   
    Conflicts =0;    
    for (unsigned int i=0; i< PathsInSameStep.size(); i++){ //First Paths
      for (unsigned int j = i+1; j< PathsInSameStep.size(); j++){ // Second Paths
          int PomConflicts =  ComparePaths(AllProcPaths[PathsInSameStep[i].src][PathsInSameStep[i].dst][PathsInSameStep[i].index],
                                   AllProcPaths[PathsInSameStep[j].src][PathsInSameStep[j].dst][PathsInSameStep[j].index]);
          if (PomConflicts != 0) {
             Conflicts += PomConflicts;
             LogStream  << "[" << PathsInSameStep[i].src <<"-> " << PathsInSameStep[i].dst <<"] x ";
             LogStream  << "[" << PathsInSameStep[j].src <<"-> " << PathsInSameStep[j].dst <<"]" << endl;
             LogStream  << "Count = " << PomConflicts << endl;
             
          } 
                       
                                    
      }
    }
  }
        
  return LogStream.str();
}// end of PrintConflictPosition
//-------------------------------------------------------------------------------- 


//---------------------------------------------------------------------------
#pragma package(smart_init)
