/*------------------------------------------------------------------*/
/* EvoCCs_boradcast						    */
/*   Tool for evolutionary design of broadcast-based                */
/*   collective communications on wormhole networks                 */
/*                                                                  */
/* Author: Jiri Jaros						    */
/*	   jarosjir@fit.vutbr.cz                                    */
/* 	   Brno University of Technology			    */
/* 	   GPL Licence, 2010					    */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
/* Topology.h - Topology container implementation 		    */
/*            - header file	                          	    */
/*------------------------------------------------------------------*/

#ifndef TOPOLOGY_H
#define TOPOLOGY_H

#include <string>
#include <vector>
#include <limits>
#include <set>
using namespace std;

/* This class contains iformation about topology of a particlar interconnection
 * network.
 */
class Topology {
  private:
    // disallow copy construction until it is fully implemented
    Topology(Topology &t) {}
    // disallow assignment construction until it is fully implemented
    Topology& operator=(Topology &t) {return *this;}

    int  CommunicatingNodeCount;

    void SetInvolvedNodes();

  public:
    // type of index of a node
    typedef int NodeIndexType;
    /* 2-dimensional vector of shortest paths
     * 1st dimension: index of a path
     * 2nd dimension: index of a node on a particular path
     */
    typedef vector<vector<NodeIndexType> > Paths;
    /* representation of a node in tree used for searching paths between
     * processing nodes
     */
    vector<NodeIndexType> Transmiters;
    vector<NodeIndexType> Receivers;
    vector<NodeIndexType> InvolvedNodes; // Transmiters and Receiver involved in MNB can be used as midle-station in OAB

    vector<vector<NodeIndexType> > ReachableNodes; // Which nodes can exchange informations (only in Surrounding)


    class Node {
      public:
        // index of this node
        NodeIndexType index;
        // vector of all children
        vector<Node> children;
        // path from root to this node (contains all indices on the path:
        // root, ..., index)
        vector<NodeIndexType> path;

        // constructor for a root
        Node(NodeIndexType ind);
        // constructor which initializes path according to its parent
        Node(NodeIndexType ind, Node &parent);
    };

    // constructor
    Topology();
    // destructor
    virtual ~Topology();
    /* Read neighborhood information from a file.
     * Input file format:
     * 1st line: number_of_processors max_number_of_links_from_each_processor
     * all other lines: processor_number neighbor1 neighbor2 ...
     *
     * Comments are supported anywhere (start with #, end at end of line).
     * Empty lines and white space characters can be inserted anywhere
     * in the file.
     */
    void readFile(string fileName) throw (string);
    // print neighbor data to output stream 'outFile'
    void printNeighbors(ostream &outFile);
    /* Find all shortest paths between processors 'src' and 'dst'
     * and store them in array paths.
     * If 'lengthIncrement' is greater than zero, then also paths whose length
     * is greaater up to 'lengthIncrement' than the shortest path are searched.
     * Total length of every path is limited to 'maxLength'. If no such path
     * is found, 'false' is returned, otherwise 'true' is returned.
     */
    bool findShortestPaths(NodeIndexType src,
        NodeIndexType dst, Paths &paths);
    // Copies paths from srcPaths to dstPaths, but with inverted order of nodes
    // (source and destination nodes are exchanged). dstPaths should be empty.
    void reversePaths(Paths &srcPaths, Paths &dstPaths);
    // returns number of processors in the network
    NodeIndexType getProcCount() {return procCount;}
    // returns maximal number of links per a processor
    NodeIndexType getMaxLinksPerProc() {return maxLinksPerProc;}

    void setProcCount(int ProcessorCount) {procCount = ProcessorCount;}

    bool IsTransmiter   (NodeIndexType Node);
    bool IsReceiver     (NodeIndexType Node);
    bool IsInvolvedNode (NodeIndexType Node);

    int  GetCommunicatingNodeCount() {return CommunicatingNodeCount; }
    void SetReachableNodes(vector< vector< Topology::Paths> > & AllProcPaths); //-- Set reachable nodes --//
    bool IsNodeInSurroundOf(int BaseNode, int Node); //-- is node in surrouding?

  protected:
    // number of processors in the network
    NodeIndexType procCount;
    // maximal number of links per a processor
    NodeIndexType maxLinksPerProc;
    // actual number of links from each processor
    NodeIndexType *linksPerProc;
    /* information about neighbors of processors
     * Each positition i in the array contains an array of indices of all
     * neighbors of processor i.
     * The first element of the array of indices contains actual number
     * of neighbors of processor i.
     */
    NodeIndexType **neighbors;
    /* Expands 'node' of a tree to a specified 'depth'.
     * If a node with index 'dst' is created, then its path is added to paths
     * and'true' is returned. Otherwise 'false' is returned.
     */
    bool expandNode(Node &node, unsigned short depth, NodeIndexType dst,
        Paths &paths);

    void readNodeType(istringstream& lineStream, int procID, string fileName, int lineNum);
};// end of Topology
//--------------------------------------------------------------------------------






#endif // TOPOLOGY_H

