/*------------------------------------------------------------------*/
/* EvoCCs_scatter						    */
/*   Tool for evolutionary design of broadcast-based                */
/*   collective communications on wormhole networks                 */
/*                                                                  */
/* Author: Jiri Jaros						    */
/*	   jarosjir@fit.vutbr.cz                                    */
/* 	   Brno University of Technology			    */
/* 	   GPL Licence, 2010					    */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
/* MMB.cpp - Implementation of Many-to-Many scatter communications  */
/*         - implementation file                                    */
/*------------------------------------------------------------------*/

#pragma hdrstop

#include "MMS.h"
#include <iostream>
#include <fstream>
#include <algorithm>
#include <stdlib.h>
#include <sstream>

#include "GeneticCore.h"
#include "Topology.h"
#include "Port_check.h"


using namespace std;

//--  parameter initialization --//
const int FileNameParNum  = 1;
const int CommStepsParNum = 2;
const int FlushStatisticsInterval = 500;
const int FlushSolutionInterval = FlushStatisticsInterval*10;

extern Random RandomGenerator;
extern Topology topology;
extern int MaxExtraSteps;


int CommSteps;
int BestGeneration = -1;

//--  Container for all source destination pahts --//
vector<Topology::Paths> AllPaths;
vector< vector< Topology::Paths> > AllProcPaths;

struct TPathStruct{
   int src;
   int dst;
   int index;
  };
 

/**
 *  GetAllPaths - Find all shortest paths
*/
void GetAllPaths(string Filename, int NumOfSteps);

/**
 *  ComputeMaxProcPaths - Max possible paths from a source to a destination
*/
int ComputeMaxProcPaths();


/**
 *  ComparePaths - Compare two paths and finds conflit count
*/
int ComparePaths(vector<int>& Path1, vector<int>& Path2);


/**
 *  GetSourceNode - vrati index zdrojoveho uzlu (prepocet)
*/
int GetSourceNode(int Index);

/**
 *  GetDestinationNode - vrati index ciloveho uzlu
*/
int GetDestinationNode(int Index);

/**
 *  SetUniformSteps - set nearly the same number of receivers in all step
*/  
void SetUniformSteps(Chromozome& chr);

/**
 *  GetStepIndex - returns Step index in the chr
*/    
int GetStepIndex(int src, int dst);

/**
 *  GetPathIndex - reurtns path index in the chr
*/    
int GetPathIndex(int src, int dst);
 
/**
 *  LocalSearch - perform a localSearch Technique
*/    
double LocalSearch(Chromozome &chr, TUsageOfSlots &SlotUsage);

/**
 *  PrintConflictPosition - print positions of conflicts
*/      
string PrintConflictPosition(Chromozome& chr);



/*___________________________________________________________________________*/
/*_____________________________ Implementation ______________________________*/
/*___________________________________________________________________________*/

/**
 *  GetAllPaths - Find all paths between any source-destination pair
*/
void GetAllPaths(string Filename, int NumOfSteps){

  /* read of input file */
  int ProcCount = topology.getProcCount();

  AllProcPaths.resize(ProcCount);
  for (int i=0; i< ProcCount; i++) { // init
    AllProcPaths[i].resize(ProcCount);

    vector<int> ZeroPath;
    ZeroPath.push_back(i);
    AllProcPaths[i][i].push_back(ZeroPath);
  }

  // all shortest paths among all nodes, must be a dynamic array

    for (int src = 0; src < ProcCount; src++) {
      // From each source search only for paths to destinations with a higher
      // index.
      for (int dst = 0; dst < ProcCount; dst++) {

        // find all shortest paths from src dst
        if (!topology.findShortestPaths(src, dst, AllProcPaths[src][dst])) {
          // No path with length <= configData.commSteps was found.
          cerr << "Error: No path from node " << src <<
              " to node " << dst << " with length less or equal to " <<
              CommSteps << " was found." << endl;
          cerr << "You should increase number of commuication steps." << endl;
          exit(1);
        }
      }
      // For destinations with lower index than source, just reverse already
      // found paths from dst to src.
    }

} // end of GetAllPaths
//--------------------------------------------------------------------------------

/**
 *  MMSOnInitAlgorithm -Scatter problem initialization
*/
void MMSOnInitAlgorithm(GeneticCore& GA, int argc, char** argv){
  /* Algorithm initialization */

  string FileName = argv[FileNameParNum];
  CommSteps = atoi(argv[CommStepsParNum]);

// read of input file
  try {
    topology.readFile(FileName);
    topology.printNeighbors(cout);
    
    cerr << "Communication steps =  " << CommSteps << endl;
    cout << "--- Transmiters ---" << endl;
    cerr << "--- Transmiters ---" << endl;
    cout << "Transmiter count = " << topology.Transmiters.size() << endl;
    cerr << "Transmiter count = " << topology.Transmiters.size() << endl;

    
    
    for (int i=0; i< topology.Transmiters.size(); i++){
     cout << topology.Transmiters[i] <<" , ";
     cerr << topology.Transmiters[i] <<" , ";
    }
    
    cout << endl <<"--- Receivers ---" << endl;
    cerr << endl <<"--- Receivers ---" << endl;
    cout << "Receivers count = " << topology.Recievers.size() << endl;
    cerr << "Receivers count = " << topology.Recievers.size() << endl;

    
    for (int i=0; i< topology.Recievers.size(); i++){
     cout << topology.Recievers[i] <<" , ";
     cerr << topology.Recievers[i] <<" , ";
    }
    cout << endl;
    cerr << endl;
  } catch (string e) {
    cerr << "Error while loading topology data: " << e << endl;
    exit(1);
  }

  GA.SetChromozomeLength(topology.GetCommunicatingNodeCount() * 2);

  cout << "Chromozome size = " << GA.GetChromozomeLength() << endl;
  cout << "FileName = " << FileName << endl;
  cout << "Communication steps =  " << CommSteps << endl;

  GetAllPaths(FileName, CommSteps);
  cout << "Processor count     =  " << topology.getProcCount() << endl;


}// end of MMSOnInitAlgorithm
//--------------------------------------------------------------------------------



/**
 *  MMSInitChromozome - Scatter chromosome initialization
*/
void MMSInitChromozome(Chromozome& chr){
  /* Chromozome initialization */

  for (int Gene = 0; Gene <chr.Length(); Gene = Gene + 2){
    int OASSource      = GetSourceNode(Gene);
    int OASDestination = GetDestinationNode(Gene);

    chr[Gene] =  RandomGenerator.IntRandom(AllProcPaths[OASSource][OASDestination].size());
     
    if (chr[Gene] < 0 ) chr[Gene] = 0;

    chr[Gene+1] = RandomGenerator.IntRandom(CommSteps);
  }

} // end of MMSInitChromozome
//--------------------------------------------------------------------------------


/**
 *  ComparePaths - Returns number of conflicts in this two ways
*/
int ComparePaths(vector<int>& Path1, vector<int>& Path2){
 /* compare two paths... */
  int result = 0;

  for (int i=0; i < (signed) Path1.size()-1; i++)
    for ( int j=0; j < (signed) Path2.size()-1; j++)
      if ((Path1[i] == Path2[j]) && (Path1[i+1] == Path2[j+1])) result++;


  return result;
} // end of ComparePaths
//-------------------------------------------------------------------------------- 


/**
 *  ComputeFitnessValue - Returns number of conflicts in this two ways
*/
double ComputeFitnessValue(Chromozome& chr)
{
  
  int conflicts = 0;
  int TransmitersCount = topology.Transmiters.size();
  int RecieversCount  = topology.Recievers.size();
  vector<struct TPathStruct> PathsInSameStep; /* Name Paths in the same step */
  
  
  for (int Step=0; Step<CommSteps; Step++){
    PathsInSameStep.clear();

    for (int src=0; src < TransmitersCount; src++) // Find which paths are used in the same step
      for (int dst=0; dst < RecieversCount; dst++){
        if (chr[GetStepIndex(src,dst)] == Step) {
          struct TPathStruct PathStruct;
          PathStruct.src = topology.Transmiters[src];
          PathStruct.dst = topology.Recievers[dst];
          PathStruct.index = chr[GetPathIndex(src,dst)];

          PathsInSameStep.push_back(PathStruct); // Add Number of dest proc
       }
     } /* finding paths */

      /* conflicts' test */
    for (unsigned int i=0; i< PathsInSameStep.size(); i++){ //First Paths
      for (unsigned int j = i+1; j< PathsInSameStep.size(); j++){ // Second Paths
          conflicts += ComparePaths(AllProcPaths[PathsInSameStep[i].src][PathsInSameStep[i].dst][PathsInSameStep[i].index],
                                    AllProcPaths[PathsInSameStep[j].src][PathsInSameStep[j].dst][PathsInSameStep[j].index]);
       }
    }  /* conflict test */
   } /* for (int step=0 ... */
  return (-1)*conflicts;
}// end of ComputeFitnessValue
//-------------------------------------------------------------------------------- 



/**
 *  CheckPortUsage - Port utilization heuristic
*/
void CheckPortUsage(Chromozome &chr, TUsageOfSlots &SlotUsage){

   int TransmitersCount = topology.Transmiters.size();
  int RecieversCount   = topology.Recievers.size();
  
  for (int src = 0; src < TransmitersCount; src++){
    for (int dst = 0; dst < RecieversCount; dst++){

      int step = chr[GetStepIndex(src,dst)];
      if ((src!= dst) && (SlotUsage.IsColumnCollision(dst,step) || SlotUsage.IsRowCollision(src, step)))
      {
        int NewStep = SlotUsage.GetAndSetCorrectSlot(src, dst);
        if (NewStep != -1) chr[GetStepIndex(src,dst)]  = NewStep;
      }
    }
  }
}// end of CheckPortUsage
//-------------------------------------------------------------------------------- 


/**
 *  MMSComputeFitness - Compute MMS fitness function
*/
double MMSComputeFitness(Chromozome& chr){


  int TransmitersCount = topology.Transmiters.size();
  int RecieversCount  = topology.Recievers.size(); 
     
      //-- Port's correction --//
//  TUsageOfSlots SlotUsage(&chr, TransmitersCount, RecieversCount, CommSteps);

//  CheckPortUsage(chr, SlotUsage);  
    
  return ComputeFitnessValue(chr);
/*  LocalSearch(chr, SlotUsage); */

} // end of MMSComputeFitness
//-------------------------------------------------------------------------------- 


/**
 *  MMSComputeFitness - Print MMS chromozome in a form of schedul
*/
void MMSPrintChromozome(Chromozome& chr){


  int TransmitersCount = topology.Transmiters.size();
  int RecieversCount   = topology.Recievers.size();
  vector<struct TPathStruct> PathsInSameStep; // Name Paths in the same step


  for (int Step=0; Step<CommSteps; Step++){
    PathsInSameStep.clear();
    cerr <<"--------" <<Step << ". step of comunication ---------" << endl;

    for (int src=0; src < TransmitersCount; src++) // Who sends in the same time
      for (int dst=0; dst < RecieversCount; dst++){
        if (chr[GetStepIndex(src,dst)] == Step) {
         struct TPathStruct PathStruct;
         PathStruct.src = topology.Transmiters[src];
         PathStruct.dst = topology.Recievers[dst];
         PathStruct.index = chr[GetPathIndex(src,dst)];
         PathsInSameStep.push_back(PathStruct); // Add Number of dest proc
       } /* if (chr)... */
     }


       /* print paths in the same communication step */
    for (unsigned int i=0; i< PathsInSameStep.size(); i++){
      int src = PathsInSameStep[i].src;
      int dst = PathsInSameStep[i].dst;
      int PathIndex = PathsInSameStep[i].index;

      cerr << "  Message from "<< src << " -> " << dst << ": " << endl;

      for (unsigned int j =0; j< (AllProcPaths[src][dst][PathIndex].size()); j++)
        cerr << AllProcPaths[src][dst][PathIndex][j] << ", ";

      cerr << endl;
    } /* print paths...*/
  } /* for all steps... */
}  // end of MMSPrintChromozome
//-------------------------------------------------------------------------------- 



/**
 *  LocalSearch - Local search in MMS chromozome
*/
double LocalSearch(Chromozome &chr, TUsageOfSlots &SlotUsage)
{


  Chromozome BestChr = chr;
  double BestFitness = ComputeFitnessValue(chr);
  double NewFitness = 0;
  
  
  int RandomSearchStepNumber = RandomGenerator.IntRandom(chr.Length());
  
  
  for (int SearchStep = 0; SearchStep < RandomSearchStepNumber; SearchStep++){
    chr = BestChr;
    int Pos = RandomGenerator.IntRandom(chr.Length()>>1);  
  
    if ((SearchStep % 10) != 0) { //-- path generating   --//
      int OASSource      = GetSourceNode(Pos<<1);
      int OASDestination = GetDestinationNode(Pos<<1); //(ProcCount * GeneWidth);

      chr[Pos<<1] = RandomGenerator.IntRandom(AllProcPaths[OASSource][OASDestination].size());
    } else {                  //-- step generating --//
        int NewStep      = RandomGenerator.IntRandom(CommSteps);
        int PreviousStep = chr[(Pos<<1) + 1];
        chr[(Pos<<1) + 1] = NewStep;
//        SlotUsage.SetPortUsage(GetSourceNode(Pos),GetDestinationNode(Pos),NewStep, PreviousStep);

//        CheckPortUsage(chr, SlotUsage);
    }
  
    NewFitness = ComputeFitnessValue(chr);
    if (NewFitness > BestFitness) {
       BestChr = chr;
       BestFitness = NewFitness;

    }
  
  } // for
  
  return BestFitness;
}// end of LocalSearch
//-------------------------------------------------------------------------------- 



/**
 *  MMSMutation - Mutation of MNS chromosome
*/
void MMSMutation(Chromozome& chr){
     
  double What = RandomGenerator.DoubleRandom(); 

  if (What < 0.5) {
    // - Path Generating

    int Pos = RandomGenerator.IntRandom(chr.Length() >> 1) << 1;   

    int OASSource      = GetSourceNode(Pos);
    int OASDestination = GetDestinationNode(Pos); //(ProcCount * GeneWidth);
       
  if (RandomGenerator.DoubleRandom() >0.8) 
      chr[Pos] = RandomGenerator.IntRandom(AllProcPaths[OASSource][OASDestination].size());  
  else {      
    int MaxValue =  AllProcPaths[OASSource][OASDestination].size();
    chr[Pos] = RandomGenerator.IntGaussRandom(chr[Pos], MaxValue /9, 0, MaxValue);
  }
  

  } // Path Generating

  else {                  //- step generating --//
    double Which = RandomGenerator.DoubleRandom(); // random step
    if (Which < 0.5) {    
//0.6
      int Pos = (RandomGenerator.IntRandom(chr.Length() >> 1) << 1) + 1;        
      chr[Pos] = RandomGenerator.IntRandom(CommSteps);
//      chr[Pos] = RandomGenerator.IntGaussRandom(chr[Pos],CommSteps/9,0,CommSteps);      
      
    }else {                                        // swap two comms
      int TransmitersCount = topology.Transmiters.size();
      int RecieversCount   = topology.Recievers.size();
     
      int src =  RandomGenerator.IntRandom(TransmitersCount);
      int dst1   =  RandomGenerator.IntRandom(RecieversCount);
      int dst2   =  RandomGenerator.IntRandom(RecieversCount);
     
      int Pom = chr[2*(RecieversCount *src +dst1) +1];
      chr[2*(RecieversCount *src +dst1) +1] = chr[2*(RecieversCount *src +dst2) +1];
      chr[2*(RecieversCount *src +dst2) +1] = Pom;
    
    }
  }
}// end of MMSMutation
//-------------------------------------------------------------------------------- 


/**
 *  MMSOnFinishGeneration - logs on finish generation
*/
void   MMSOnFinishGeneration(Statistics Stat, int ActGeneration){

  if ((ActGeneration %FlushStatisticsInterval) == 0) {
    cout << (ActGeneration) <<"th generation "<< endl;
    cout << "MaxFit = " <<(  Stat.Max() )<< "  MinFit = "<< (Stat.Min()) << "  AvgFit = "<<
              ( Stat.Avg()) << "  Diver = " << (Stat.Diver()) <<endl;
              	     
         }
    if (Stat.Max() == 0)  {
      BestGeneration = ActGeneration;
      MMSOnFinish(Stat, ActGeneration);
      exit(EXIT_SUCCESS);
  }
} // end of MMSOnFinishGeneration
//-----------------------------------------------------------------------------

/**
 *  MMSOnFinish - logs on finish algorithm
*/
void   MMSOnFinish(Statistics Stat, int ActGeneration){

 cerr << "Global optima was reached in generation no.  = " << BestGeneration<< endl;
 cerr << "The best solution: " << (Stat.BestChromozome()).ToString()<< endl;
 cerr << "Fitness function = " << Stat.BestChromozome().Fitness() << endl;

 MMSPrintChromozome(Stat.BestChromozome());
 if ( Stat.BestChromozome().Fitness() == 0)  cout << "End of algorithm - Global optima was found" << endl;
 else cout << "End of algorithm - Stuck in local optima " << endl;

} // end of MMSOnFinish 
//-----------------------------------------------------------------------------

/**
 *  GetSourceNode
*/
int GetSourceNode(int Index)
{

  return (topology.Transmiters[(Index / topology.Recievers.size())>> 1]);
} // end of GetSourceNode 
//-----------------------------------------------------------------------------

/**
 *  GetDestinationNode
*/
int GetDestinationNode(int Index){

  return (topology.Recievers[(Index >> 1) % topology.Recievers.size()]);
} // end of GetDestinationNode
//-----------------------------------------------------------------------------


/**
 *  GetStepIndex
*/
int GetStepIndex(int src, int dst)
{

 return  (((topology.Recievers.size()*src +dst) << 1) + 1);
} // end of GetStepIndex 
//-----------------------------------------------------------------------------

/**
 *  GetPathIndex
*/
int GetPathIndex(int src, int dst)
{
 
  return  ((topology.Recievers.size()*src +dst) << 1);
} // end of GetPathIndex
//----------------------------------------------------------------------------- 

/**
 *  PrintConflictPosition
*/
string PrintConflictPosition(Chromozome& chr){

  int conflicts = 0;
  int TransmitersCount = topology.Transmiters.size();
  int RecieversCount  = topology.Recievers.size();
  vector<struct TPathStruct> PathsInSameStep; /* Name Paths in the same step */
  

  ostringstream LogStream;
  
  for (int Step=0; Step<CommSteps; Step++){
    PathsInSameStep.clear();
    LogStream  << "---Step = " << Step << "---"<<endl;

    for (int src=0; src < TransmitersCount; src++) // Find which paths are used in the same step
      for (int dst=0; dst < RecieversCount; dst++){
        if (chr[GetStepIndex(src,dst)] == Step) {
          struct TPathStruct PathStruct;
          PathStruct.src = topology.Transmiters[src];
          PathStruct.dst = topology.Recievers[dst];
          PathStruct.index = chr[GetPathIndex(src,dst)];

          PathsInSameStep.push_back(PathStruct); // Add Number of dest proc
       }
     } /* finding paths */

      /* conflicts' test */
    for (unsigned int i=0; i< PathsInSameStep.size(); i++){ //First Paths
      for (unsigned int j = i+1; j< PathsInSameStep.size(); j++){ // Second Paths
        int PomConflicts =  ComparePaths(AllProcPaths[PathsInSameStep[i].src][PathsInSameStep[i].dst][PathsInSameStep[i].index],
                                   AllProcPaths[PathsInSameStep[j].src][PathsInSameStep[j].dst][PathsInSameStep[j].index]);
          if (PomConflicts != 0) {
             conflicts += PomConflicts;
             LogStream  << "[" << PathsInSameStep[i].src <<"-> " << PathsInSameStep[i].dst <<"] x ";
             LogStream  << "[" << PathsInSameStep[j].src <<"-> " << PathsInSameStep[j].dst <<"]" << endl;
             LogStream  << "Count = " << PomConflicts << endl;
             
          } 
          
          
          
       }
    }  /* conflict test */
   } /* for (int step=0 ... */
    
  return LogStream.str();
} // end of PrintConflictPosition
//----------------------------------------------------------------------------- 



//---------------------------------------------------------------------------
#pragma package(smart_init)
