/*------------------------------------------------------------------*/
/* EvoCCs_scatter						    */
/*   Tool for evolutionary design of broadcast-based                */
/*   collective communications on wormhole networks                 */
/*                                                                  */
/* Author: Jiri Jaros						    */
/*	   jarosjir@fit.vutbr.cz                                    */
/* 	   Brno University of Technology			    */
/* 	   GPL Licence, 2010					    */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
/* Port_check.cpp - UMDA port check heuristics	 		    */
/*                - implementation file	                            */
/*------------------------------------------------------------------*/

#pragma hdrstop

#include <algorithm>

#include "Port_check.h"
#include "Topology.h"
#include "Chromozome.h"
#include "Random.h"

//---------------------------------------------------------------------------
extern Topology topology;  
extern Random RandomGenerator; 

//---------------------------------------------------------------------------


/**
 *  TUsageOfSlots::SetPortUtilization - Set Sender and Receiver port usage
*/
void TUsageOfSlots::SetPortUtilization()
{
  //-- check row's (Sender's) port ussage
  for (int row = 0; row < FRowCount; row++){
    for (int col = 0; col < FColumnCount; col ++)
    {
       if (row != col)
       {
          int step = (*Fchr)[GetStepIndex(row,col)];
          FRowPortUsage[row][step]++;
          FColumnPortUsage[col][step]++;
       } // if

     } // for col
   } // for row
}// end of TUsageOfSlots::SetPortUtilization
//---------------------------------------------------------------------------


/**
 *  TUsageOfSlots::TUsageOfSlots - Create an object and compute basic Ports' utilizations
*/
TUsageOfSlots::TUsageOfSlots(Chromozome* chr, int SenderCount, int ReceiverCount, int StepCount)
{
  FStepCount   = StepCount;
  FRowCount    = SenderCount;
  FColumnCount = ReceiverCount;
  Fchr = chr;
    //-- Init Senders' and Receives' maps
  FRowPortUsage.resize(FRowCount);
  FColumnPortUsage.resize(FColumnCount);


  SetPortUtilization();

}// end of TUsageOfSlots::TUsageOfSlots
//---------------------------------------------------------------------------


/**
 *  TUsageOfSlots::IsRowCollision - Is there a row collision?
*/
bool TUsageOfSlots::IsRowCollision(int Sender, int Step)
{
 return (FRowPortUsage[Sender][Step]      > topology.GetNumOfNeighbours(topology.Transmiters[Sender]));

}// end of TUsageOfSlots::IsRowCollision
//---------------------------------------------------------------------------



/**
 *  TUsageOfSlots::IsColumnCollision - Is there a column collision?
*/
bool TUsageOfSlots::IsColumnCollision(int Receiver, int Step)
{
 return (FColumnPortUsage[Receiver][Step] > topology.GetNumOfNeighbours(topology.Recievers[Receiver]));

}// end of TUsageOfSlots::IsColumnCollision 
//---------------------------------------------------------------------------


/**
 *  TUsageOfSlots::GetFreeSlots  
*/
void TUsageOfSlots::GetFreeSlots(TFreeSteps& FreeSteps, int Node, bool InRow)
{
  TPortUsage * PortUsage;

  PortUsage   = &FRowPortUsage[Node];
  int NodeID  = topology.Transmiters[Node];
  
  if (!InRow) {
    PortUsage = &FColumnPortUsage[Node];
    NodeID    = topology.Recievers[Node];    
  }

  FreeSteps.clear();


  for (int step = 0; step < FStepCount; step++){
    if ((* PortUsage)[step] < topology.GetNumOfNeighbours(Node)){
      FreeSteps.insert(step);
    }
  }
}// end of TUsageOfSlots::GetFreeSlots
//---------------------------------------------------------------------------


/**
 *  TUsageOfSlots::GetCorrectStep - Get correct time slot
*/
int TUsageOfSlots::GetCorrectStep(int Sender, int Receiver)
{
  TFreeSteps RowFreeSteps, ColumnFreeSteps, FreeSteps;
  insert_iterator<set<int, less<int> > > FreeSteps_ins(FreeSteps, FreeSteps.begin());


  GetFreeSlots(RowFreeSteps, Sender, true);
  GetFreeSlots(ColumnFreeSteps, Receiver, false);

  set_intersection(RowFreeSteps.begin(), RowFreeSteps.end(), ColumnFreeSteps.begin(), ColumnFreeSteps.end(), FreeSteps_ins);

  if (FreeSteps.size() > 0) {
    int  RndIdx = RandomGenerator.IntRandom(RowFreeSteps.size());
    set<int>::iterator it = FreeSteps.begin();
    for (int i =1 ; i< RndIdx; i++)  it++;
    
    return *it; 
  
  }      
  else return -1;

}// end of TUsageOfSlots::GetCorrectStep
//---------------------------------------------------------------------------


/**
 *  TUsageOfSlots::GetAndSetCorrectSlot - Set Correct slot for communication
*/
int TUsageOfSlots::GetAndSetCorrectSlot(int Sender, int Receiver){

  int Step = GetCorrectStep(Sender, Receiver);

  if (Step != -1) {
    int PreviousStep = (*Fchr)[GetStepIndex(Sender, Receiver)];

    FRowPortUsage[Sender][PreviousStep]--;
    FColumnPortUsage[Receiver][PreviousStep]--;


    FRowPortUsage[Sender][Step]++;
    FColumnPortUsage[Receiver][Step]++;
  }

  return Step;
}// end of TUsageOfSlots::GetAndSetCorrectSlot
//---------------------------------------------------------------------------


/**
 *  TUsageOfSlots::SetPortUsage 
*/
void TUsageOfSlots::SetPortUsage(int Sender, int Receiver, int NewStep, int PreviousStep)
{
  FRowPortUsage[Sender][PreviousStep]--;
  FColumnPortUsage[Receiver][PreviousStep]--;

  FRowPortUsage[Sender][NewStep]++;
  FColumnPortUsage[Receiver][NewStep]++;

}// end of TUsageOfSlots::SetPortUsage
//---------------------------------------------------------------------------





#pragma package(smart_init)
