/*
								+----------------------------------+
								|                                  |
								| ***  Numerical math methods  *** |
								|                                  |
								|   Copyright  -tHE SWINe- 2008   |
								|                                  |
								|           Numerical.h            |
								|                                  |
								+----------------------------------+
*/

#ifndef __NUMERICAL_MATH_INCLUDED
#define __NUMERICAL_MATH_INCLUDED

/*
 *	template <class CFunctor, class CScalarType>
 *	class CGoldenCutSover
 *		- simple one-dimensional numerical sovler with max error parameter
 *		- using the golden cut method
 */
template <class CFunctor, class CScalarType>
class CGoldenCutSover {
protected:
	CFunctor m_function;
	CScalarType m_f_min, m_f_max;
	CScalarType m_f_max_error;
	CScalarType m_f_root;

public:
	/*
	 *	inline CGoldenCutSover::CGoldenCutSover(CFunctor function, CScalarType f_root,
	 *		CScalarType f_min, CScalarType f_max, CScalarType f_max_error)
	 *		- default constructor; function is root lies in range f_min to f_max (estimate),
	 *		  calculates root with maximal relative error f_max_error
	 */
	inline CGoldenCutSover(CFunctor function, CScalarType f_root,
		CScalarType f_min, CScalarType f_max, CScalarType f_max_error)
		:m_function(function), m_f_min(f_min), m_f_max(f_max),
		m_f_root(f_root), m_f_max_error(f_max_error * (f_max - f_min))
	{
		_ASSERTE(m_f_min < m_f_max);
	}

	/*
	 *	inline CGoldenCutSover::operator CScalarType () const
	 *		- evaluates root position
	 *		- note this might get stuck in infinite loop if unlucky
	 */
	inline operator CScalarType () const
	{
		float f_left_x = m_f_min;
		float f_right_x = m_f_max;
		// start in min / max

		CScalarType f_left_y = m_function(f_left_x) - m_f_root;
		CScalarType f_right_y = m_function(f_right_x) - m_f_root;
		// calculate function in both points

		const CScalarType f_theta = CScalarType(.6180339889579);
		// golden ratio

		do {
			if(fabs(f_left_y) < fabs(f_right_y)) {
				f_right_x = f_left_x + (f_right_x - f_left_x) * f_theta;
				f_right_y = m_function(f_right_x) - m_f_root;
			} else {
				f_left_x = f_left_x + (f_right_x - f_left_x) * (1 - f_theta);
				f_left_y = m_function(f_left_x) - m_f_root;
			}
			// shift one of points closer to root and recalculate function values
		} while(f_right_x - f_left_x > m_f_max_error);
		return (f_left_x + f_right_x) * .5;
	}
};

/*
 *	template <class CFunctor, class CScalarType>
 *	class CGoldenCutSover2
 *		- simple one-dimensional numerical sovler with max error parameter
 *		- difference between CGoldenCutSover and CGoldenCutSover2
 *		  is CGoldenCutSover2 finds intersection of function with zero
 *		  whereas CGoldenCutSover finds intersection with given constant
 *		- using the golden cut method
 */
template <class CFunctor, class CScalarType>
class CGoldenCutSover2 {
protected:
	CFunctor m_function;
	CScalarType m_f_min, m_f_max;
	CScalarType m_f_max_error;
	CScalarType m_f_root;

public:
	/*
	 *	inline CGoldenCutSover::CGoldenCutSover2(CFunctor function, CScalarType f_min,
	 *		CScalarType f_max, CScalarType f_max_error)
	 *		- default constructor; function is root lies in range f_min to f_max (estimate),
	 *		  calculates root with maximal relative error f_max_error
	 */
	inline CGoldenCutSover2(CFunctor function, CScalarType f_min,
		CScalarType f_max, CScalarType f_max_error)
		:m_function(function), m_f_min(f_min), m_f_max(f_max),
		m_f_root(f_root), m_f_max_error(f_max_error * (f_max - f_min))
	{
		_ASSERTE(m_f_min < m_f_max);
	}

	/*
	 *	inline CGoldenCutSover::operator CScalarType () const
	 *		- evaluates root position
	 *		- note this might get stuck in infinite loop if unlucky
	 */
	inline operator CScalarType () const
	{
		float f_left_x = m_f_min;
		float f_right_x = m_f_max;
		// start in min / max

		CScalarType f_left_y = m_function(f_left_x) - m_f_root;
		CScalarType f_right_y = m_function(f_right_x) - m_f_root;
		// calculate function in both points

		do {
			if(fabs(f_left_y) < fabs(f_right_y)) {
				f_right_x = f_left_x + (f_right_x - f_left_x) * f_theta;
				f_right_y = m_function(f_right_x) - m_f_root;
			} else {
				f_left_x = f_left_x + (f_right_x - f_left_x) * (1 - f_theta);
				f_left_y = m_function(f_left_x) - m_f_root;
			}
			// shift one of points closer to root and recalculate function values
		} while(f_right_x - f_left_x > m_f_max_error);
		return (f_left_x + f_right_x) * .5;
	}
};

/*
 *	template <class CFunctor>
 *	inline float f_Solve(CFunctor function, float f_root, float f_min, float f_max, float f_max_error)
 *		- finds function root (such x where function(x) == f_root) using golden cut method
 *		- f_min to f_max is estimate range where root should be
 *		- returns root with maximal relative error f_max_error
 *		  (or ends in infinite loop if it can't find one)
 */
template <class CFunctor>
inline float f_Solve(CFunctor function, float f_root, float f_min, float f_max, float f_max_error)
{
	return CGoldenCutSover<CFunctor, float>(function, f_root, f_min, f_max, f_max_error);
}

/*
 *	template <class CFunctor>
 *	inline float f_Solve(CFunctor function, float f_min, float f_max, float f_max_error)
 *		- finds function root (such x where function(x) == 0) using golden cut method
 *		- f_min to f_max is estimate range where root should be
 *		- returns root with maximal relative error f_max_error
 *		  (or ends in infinite loop if it can't find one)
 */
template <class CFunctor>
inline float f_Solve(CFunctor function, float f_min, float f_max, float f_max_error)
{
	return CGoldenCutSover2<CFunctor, float>(function, f_min, f_max, f_max_error);
}

/*
 *	template <class CFunctor, class CScalarType>
 *	class CIntegrator
 *		- simple one-dimensional integrator with max error parameter
 */
template <class CFunctor, class CScalarType>
class CIntegrator {
protected:
	CFunctor m_function;
	CScalarType m_f_min, m_f_scale;
	CScalarType m_f_max_error;

public:
	/*
	 *	inline CIntegrator::CIntegrator(CFunctor function, CScalarType f_min,
	 *		CScalarType f_max, CScalarType f_max_error)
	 *		- default constructor; function is function being integrated over range f_min to f_max
	 *		  with maximal relative error f_max_error
	 */
	inline CIntegrator(CFunctor function, CScalarType f_min,
		CScalarType f_max, CScalarType f_max_error)
		:m_function(function), m_f_min(f_min), m_f_scale(f_max - f_min), m_f_max_error(f_max_error)
	{}

	/*
	 *	inline CIntegrator::operator CScalarType () const
	 *		- evaluates function integral
	 *		- note this may take a long time if high precission
	 *		  is required and / or integration range is wide
	 */
	inline operator CScalarType () const
	{
		CScalarType f_scale = m_f_scale;
		CScalarType f_first_integral((m_function(m_f_min) +
			m_function(m_f_min + f_scale)) * f_scale * CScalarType(.5));
		// first integral approximation (trapezoid rule)

		if(m_f_scale == 0)
			return 0;

		for(int n_pass = 1; n_pass < 20; ++ n_pass) {
			CScalarType f_integral(f_first_integral * CScalarType(.5));
			// prepare integration variable

			f_scale *= CScalarType(.5);
			// scale is going down with every refinement

			CScalarType f_refinement(0), f_arg(f_scale + m_f_min);
			for(int i = 1, n = (1 << n_pass) + 1; i < n; i += 2, f_arg += 2 * f_scale)
				f_refinement += m_function(f_arg); // bloody hell optimized
			f_refinement *= f_scale;
			f_integral += f_refinement;
			// refine integral sollution

			if(fabs((f_integral - f_first_integral) / f_first_integral) <= m_f_max_error)
				return f_integral;
			// is error in required range now?

			f_first_integral = f_integral;
		}
		// adaptive subdivision to fit error

		return f_first_integral;
	}
};

/*
 *	template <class CFunctor>
 *	inline double f_Integrate(CFunctor function, double f_min, double f_max, double f_max_error)
 *		- integrates one-dimensional function over range f_min to f_max (inclusive)
 *		  while maintaining maximal relative error f_max_error
 *		- returns the value of the integral
 */
template <class CFunctor>
inline double f_Integrate(CFunctor function, double f_min, double f_max, double f_max_error)
{
	return CIntegrator<CFunctor, double>(function, f_min, f_max, f_max_error);
}

/*
 *	template <class CFunctor>
 *	inline float f_Integrate_s(CFunctor function, float f_min, float f_max, float f_max_error)
 *		- integrates one-dimensional function over range f_min to f_max (inclusive)
 *		  while maintaining maximal relative error f_max_error
 *		- returns the value of the integral
 */
template <class CFunctor>
inline float f_Integrate_s(CFunctor function, float f_min, float f_max, float f_max_error)
{
	return CIntegrator<CFunctor, float>(function, f_min, f_max, f_max_error);
}

#endif //__NUMERICAL_MATH_INCLUDED
