/*
								+---------------------------------+
								|                                 |
								|     ***   URI helpers   ***     |
								|                                 |
								|  Copyright   -tHE SWINe- 2010  |
								|                                 |
								|              URI.h              |
								|                                 |
								+---------------------------------+
*/

#ifndef __URI_UTILS_INCLUDED
#define __URI_UTILS_INCLUDED

#include "../StlUtils.h"
#include "../UniConv.h"

/**
 *	@brief utility functions for request URI transforms
 */
class CURI_Utils {
public:
	/**
	 *	@brief replaces non-URI characters by "% hex hex" escape sequences in UTF8
	 *
	 *	@param[out] r_s_dest is destination string for the URI
	 *	@param[in] r_s_uri is plain URI string (may contain non-us-english characters, which are in turn escaped)
	 *	@param[in] encoder is function object responsible for translating 8-bit characters to unicode
	 *
	 *	@return Returns true on success, false on failure (malformed codes / not enough memory).
	 */
	template <class CEncodeUnicode>
	static bool Escape_URI_UTF8(std::string &r_s_dest, const std::string &r_s_uri, CEncodeUnicode encoder)
	{
		r_s_dest.erase();
		// erase destination string

		for(size_t i = 0, n = r_s_uri.length(); i < n; ++ i) {
			int n_char = encoder(r_s_uri[i]);
			// translate character to unicode

			const char *p_s_gen_delims = ":/?#[]@";
			const char *p_s_sub_delims = "!$&\'()*+,;=";
			const char *p_s_unreserved = "-._~";
			if(n_char < 0 || n_char >= 0x80 || r_s_uri[i] != n_char || (!isdigit(n_char) && !isalpha(n_char) &&
			   !strchr(p_s_gen_delims, n_char) && !strchr(p_s_sub_delims, n_char) &&
			   !strchr(p_s_unreserved, n_char))) {
				std::string s_utf8;
				if(!CUniConv::UTF32_to_UTF8(&n_char, 1, s_utf8, false))
					return false;
				// translate character to utf-8

				if(!stl_ut::Reserve_NMore(r_s_dest, 3 * s_utf8.length()))
					return false;
				// make space for extra characters in uri

				for(int j = 0, m = s_utf8.length(); j < m; ++ j) {
					std::string s_number;
					if(!stl_ut::Format(s_number, "%%%02x", int(unsigned char(s_utf8[j]))) ||
					   !stl_ut::Append(r_s_dest, s_number))
						return false;
				}
				// append (escaped) utf-8
			} else {
				if(!stl_ut::Reserve_1More(r_s_dest))
					return false;
				r_s_dest += char(n_char);
				// append plain character (plain 7-bit english)
			}
		}

		return true;
	}

	/**
	 *	@brief replaces escape sequences in "% hex hex" UTF8 format by appropriate characters
	 *
	 *	@param[out] r_s_dest is destination string for the URI
	 *	@param[in] r_string is input URI, containing escape sequences
	 *	@param[in] decoder is function object responsible for translating unicode characters to 8-bit charset
	 *
	 *	@return Returns true on success, false on failure (malformed codes / not enough memory)
	 */
	template <class CDecodeUnicode>
	static bool Unescape_URI_UTF8(std::string &r_s_dest, const std::string &r_string, CDecodeUnicode decoder)
	{
		if(!stl_ut::Assign(r_s_dest, r_string))
			return false;

		for(;;) {
			int n_position;
			if((n_position = r_s_dest.find('%')) != std::string::npos) {
				uint8_t n_byte;
				if(!GetByte(r_s_dest, n_position + 1, n_byte))
					return false;
				// get first byte

				int n_byte_num;
				if((n_byte_num = CUniConv::n_UTF8_Char_Size(n_byte)) < 0)
					return false;
				// determine number of bytes

				if(n_byte_num == 1)
					r_s_dest[n_position] = (char)n_byte;
				else {
					uint8_t p_utf8[4];
					p_utf8[0] = n_byte;
					for(int i = 1; i < n_byte_num; ++ i) {
						r_s_dest.erase(n_position + 1, 1); // erase additional '%'
						if(!GetByte(r_s_dest, n_position + 1, p_utf8[i]))
							return false;
					}
					// read additional bytes

					int n_read_bytes;
					int n_character;
					if((n_character = CUniConv::n_UTF8_Code(p_utf8, n_byte_num, n_read_bytes)) < 0)
						return false;
					_ASSERTE(n_byte_num == n_read_bytes);
					// decode utf-8

					r_s_dest[n_position] = decoder(n_character);
					// translate to local charset
				}
			} else
				break;
		}
		// parse %xx character codes

		return true;
	}

	/**
	 *	@brief gets filename along with path, extracted from p_s_uri; handle absolute URI's
	 *
	 *	@param[out] r_s_filename will contaion output filename
	 *	@param[in] r_s_uri is URI of a file (unescaped)
	 *	@param[in] p_s_default is default filename for cases p_s_uri doesn't contain it
	 *
	 *	@return Returns true on success, false on failure.
	 */
	static bool Get_FileName_Part(std::string &r_s_filename, const std::string &r_s_uri, const char *p_s_default = "index.html");

protected:
	static bool GetByte(std::string &r_string, int n_position, uint8_t &r_n_byte);
};

#endif //__URI_UTILS_INCLUDED
