#ifndef __BLOOM_H__
#define __BLOOM_H__

#include <stddef.h>
#include <stdbool.h>
#include <stdint.h>

/**
 * Bloom Filter
 * this is ordinary Bloom filter with bit array that supports
 * only adding of elements and membership querying
 */
struct bloom_filter_t {
  /* number of hash functions */
  unsigned int hash_functions;
  /* seeds for hash functions */
  uint32_t *hash_seeds;
  /* number of elements in bit field */
  unsigned int field_elements;
  /* bit field */
  unsigned char *field;
};

/**
 * bloom_filter_init
 * @param hash_functions number of hash functions to use
 * @param hash_seeds seeds for particular hash functions
 * @param field_elements number of bits in array
 * @return false if memory allocation failed, true otherwise
 */
bool bloom_filter_init(struct bloom_filter_t *filter,
                       unsigned int hash_functions,
                       uint32_t *hash_seeds,
                       unsigned int field_elements);

void bloom_filter_deinit(struct bloom_filter_t *filter);

/**
 * bloom_filter_add
 * adds the key of length key_length to the filter
 */
void bloom_filter_add(struct bloom_filter_t *filter,
                      const uint32_t *key,
                      size_t key_length);

/**
 * bloom_filter_query
 * queries if the filter contains the key
 * false positives are posible
 */
bool bloom_filter_query(struct bloom_filter_t *filter,
                        const uint32_t *key,
                        size_t key_length);

#endif // __BLOOM_H__
