////////////////////////////////////////////////////////////////////////////////////////////////////
//!\file include\CComplexMedian.h
//!
//!\author  Vit Stancl
//!\date 18.8.2011
//!\brief   Declares the complex median filter class. 
////////////////////////////////////////////////////////////////////////////////////////////////////
#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Image/mdsImage.h>
#include <MDSTk/Image/mdsImageFunctions.h>
#include <algorithm>
#include <vector>

class CComplexMedian
{
public:
   typedef mds::img::CComplexImage tImage;
   typedef mds::img::CComplexImagePtr tImagePtr;
   typedef mds::img::CComplexImage::tPixel tPixel;
   typedef std::vector< tPixel > tArray;
   typedef std::vector< tPixel >::iterator tIterator;

public:
   //! Create filter - constructor
   CComplexMedian( mds::tSize filter_size ) : m_filter_size( filter_size ), m_data_size( filter_size*filter_size ) { m_data.resize( m_data_size ); }

   //! Destructor
   ~CComplexMedian() { }

   //! Filter - use real part median
   void filterRe( const tImage & input, tImage & output );

   //! Filter - use real part median
   void filterIm( const tImage & input, tImage & output );

   //! Filter - use magnitude median
   void filterMag( const tImage & input, tImage & output );

protected:
   template < typename tpCompare >
   tPixel findMedian( tArray & data, tpCompare compareFunc );

protected:
   //! Filter size
   mds::tSize m_filter_size;

   //! Data array size
   mds::tSize m_data_size;

   //! Filtering array
   tArray m_data;

}; // class CComplexMedian

template < typename tpCompare >
CComplexMedian::tPixel CComplexMedian::findMedian( tArray & Data, tpCompare compareFunc )
{
    // Index of the median value
    const mds::tSize Size = mds::tSize(Data.size());
    const mds::tSize Median = Size >> 1; // Size / 2

    // Z algorithm initialization
    mds::tSize Left = 0;
    mds::tSize Right = Size - 1;

    // Median finding
    while( Left < Right )
    {
        tPixel Value = Data[Median];
        mds::tSize LeftValue = Left;
        mds::tSize RightValue = Right;

        while(  RightValue >= Median  && Median >= LeftValue  )
        {
            while( compareFunc( Data[LeftValue], Value ) )
            {
                ++LeftValue;
            }
            while( compareFunc( Value, Data[RightValue] ) )
            {
                --RightValue;
            }

            tPixel Temp = Data[LeftValue];
            Data[LeftValue] = Data[RightValue];
            Data[RightValue] = Temp;

            ++LeftValue;
            --RightValue;
        }

        if( RightValue < Median )
        {
            Left = LeftValue;
        }
        if( Median < LeftValue )
        {
            Right = RightValue;
        }
    }

    return Data[Median];
}