////////////////////////////////////////////////////////////////////////////////////////////////////
//\file  D:\BioMarker\Software\Iterative\include\DividingFilter.h
//
//\brief Declares the dividing filter class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef DividingFilter_H_included
#define DividingFilter_H_included

#include "ComplexFilter.h"
#include "RichardsonLucy.h"
#include "LamGoodman.h"
#include "MaximumLikelihood.h"
#include "DenoisedMLE.h"
#include "Windowing.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
//\class CDividingFilter
//
//\brief Dividing filter. 
////////////////////////////////////////////////////////////////////////////////////////////////////

class CDividingFilter : public CComplexFilter
{
public:
	//! Used filter type
	enum EFilterType
	{
		FT_LG,	// Lam-Goodman
		FT_RL,  // Richardson-Lucy
		FT_MLE,  // Maximum likelihood estimation
      FT_DMLE  // Denoised maximum likelihood estimation
	};

protected:
   //! Divided images pointers type
   typedef std::vector< tImagePtr > tImagesVector;

   //! One tile descriptor
   struct STileDescriptror
   {
   public:

      /// Subtiles positions in the source image. 
      // 
      // Destination is ordered
      // 0 1
      // 2 3
      int x[ 4 ], y[ 4 ];

      //! Subtile width and height (all are the same).
      int w[4], h[4];

      //! Where start in the windowing function
      int wx[4], wy[4];

      //! constructor
      STileDescriptror() { for( int i = 0; i < 4; ++i ) { x[ i ] = y[ i ] = w[i] = h[i] = wx[i] = wy[i] = 0; } }
   };

   //! Sizes vector
   typedef std::vector< STileDescriptror > tSizesVector;

public:
   //! Constructor
   CDividingFilter( EFilterType filter = FT_MLE, bool padded = false);

   //! Initializing constructor
   CDividingFilter( const tIImage & input, const tIImage & psf, EFilterType filter = FT_RL, bool padded = false  );

   //! Destructor
   ~CDividingFilter();

   //! Initialize - set input, set PSF 
   virtual bool init( const tIImage & input, const tIImage & psf, bool padded = false  );

   //! Do one iteration
   void step();

   //! Filter 
   int filter( mds::tSize max_steps);

   //! Get filter output
   virtual void getOutput( tImage & image );

   //! Get filter output - frequency domain
   virtual void getOutputFD( tImage & image );

   //! Set used filter
   void setFilterType( EFilterType type ) { m_filterType = type; }

protected:
   //! Initialize tile sizes array
   bool sizes();

   //! Initialize tile sizes array - padded version
   bool sizesPadded();

   //! Split current image to the tiles
   void split( const tImage & src, tImage & dst, const STileDescriptror & tile );

   //! Split image with paddings
   void splitPadded( const tImage & src, tImage & dst, const STileDescriptror & tile );

   //! Cut padded tile size and position to the image size
   void cutTile( STileDescriptror & tile );

   //! Concatenate vector back to the image
   void concatenate( const tImage & src, tImage & dst, const STileDescriptror & tile );

   //! Concatenate back padded tiles
   void concatenatePadded( const tImage & src, tImage & dst, const STileDescriptror & tile );

   //! Add values from the window to the image
   void addWindowed( const tImage & src, tImage & dst, const CWindow & window, int sx, int sy, int dx, int dy, int tx, int ty, int w, int h );

   //! Create filters
   bool createFilters( EFilterType type );

protected:
   //! Image sizes array
   tSizesVector m_sizes;

   //! Dividing parameters
   mds::tSize m_divideW, m_divideH;

   // Tile width and height
   mds::tSize m_tileW, m_tileH;

   // Subtile width and height
   mds::tSize m_subtileW, m_subtileH;

   //! Subfilters array
   CComplexFilter ** m_filters;

   //! Padding
   mds::tSize m_tilePaddingW, m_tilePaddingH;

   //! Windowing function
   CTukeyWindow m_window;

   //! Use padded version
   bool m_padded;

   //! Used filter type
   EFilterType m_filterType;
};


// DividingFilter_H_included
#endif

  