////////////////////////////////////////////////////////////////////////////////////////////////////
//\file  D:\BioMarker\Software\Iterative\include\NoiseEstimation.h
//
//\brief Declares the noise estimation class. 
// 
// Based on:
// Restoration from image degraded by white noise based on iterative spectral subtraction method
// Author: Kobayashi, T.,   Shimamura, T.,   Hosoya, T.,   Takahashi, Y.   
// This paper appears in: Circuits and Systems, 2005. ISCAS 2005. IEEE International Symposium on 
// Issue Date: 23-26 May 2005 
// On page(s): 6268 - 6271 Vol. 6 
// Print ISBN: 0-7803-8834-8 
// References Cited: 6 
// INSPEC Accession Number: 8633174 
// Digital Object Identifier: 10.1109/ISCAS.2005.1466073  
// Date of Current Version: 25 ervenec 2005 
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef NoiseEstimation_H_included
#define NoiseEstimation_H_included

#include <MDSTk/Image/mdsImage.h>


////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	Noise estimator - abstract class. 
////////////////////////////////////////////////////////////////////////////////////////////////////
class CNoiseEstimatorBase 
{
public:
   //! Input/output image type
   typedef mds::img::CFImage tImage;

   //! Pixel type
   typedef mds::img::CFImage::tPixel tPixel;

   //! Smart pointer type
   typedef mds::img::CFImagePtr tImagePtr;

   //! Complex image type
   typedef mds::img::CComplexImage tCImage;

   //! Complex pixel type
   typedef mds::img::CComplexImage::tPixel tCPixel;

   //! Complex image pointer type
   typedef mds::img::CComplexImagePtr tCImagePtr;

public:
	//! Estimate image noise
	virtual bool estimate( const tImage & input, tImage & output ) { return false; }

	//! Estimate image noise - from spectrum to spectrum
	virtual bool estimate( const tCImage & input, tCImage & output ) { return false; }

protected:
	//! Fast Fourier transform
	void fft( const tCImage & src, tCImage & dst );

	//! Inverse Fourier transform
	void ifft( const tCImage & src, tCImage & dst );
};

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	CFBDP. 
////////////////////////////////////////////////////////////////////////////////////////////////////
class CFBDP : public CNoiseEstimatorBase
{
public:

	//! Constructor
	CFBDP( float parameter = 50, int tile_size = 32 ) : m_tile_size( tile_size ), m_p( parameter ) {}

	//! Set tile size
	void setTileSize( mds::tSize size ) { m_tile_size = size; }

	//! Estimate image noise
	virtual bool estimate( const tImage & input, tImage & output );

	//! Estimate image noise - from spectrum to spectrum
	virtual bool estimate( const tCImage & input, tCImage & output );


protected:
	//! Get tile mean
	float getTileMean( mds::tSize tx, mds::tSize ty, const tImage & data );

	//! Fill complex tile with value
	void fillTile( mds::tSize tx, mds::tSize ty, tCImage & data, tCPixel value );

	//! Copy tile
	void copyTile( mds::tSize sx, mds::tSize sy, mds::tSize dx, mds::tSize dy, const tCImage & src, tCImage & dst );

	//! Create corner tile
	void createCornerTile( const tCImage & data, bool shifted = false );

protected:
	//! Tile size 
	mds::tSize m_tile_size;

	//! Thresholding parameter
	float m_p;

	//! Corner tile
	tCImagePtr m_corner_tile;

protected:
	//! Remove negative pixels operator
	struct CCutNegative
	{
	public:
		// Operator
		void operator()( tCPixel & p )
		{
			if( p.re() < 0.0f ) p = 0.0f;
		}
	};

};

// NoiseEstimation_H_included
#endif


