////////////////////////////////////////////////////////////////////////////////////////////////////
//!\file include\TestFDF2D.h
//!
//!\brief   Declares the test fdf 2 d class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef TestFDF_H_included
#define TestFDF_H_included

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Module/mdsModule.h>
#include "LamGoodman.h"
#include "RichardsonLucy.h"
#include "DividingFilter.h"
#include "MaximumLikelihood.h"
#include "DenoisedMLE.h"
#include <MDSTk/ImageIO/mdsPNG.h>


class CTestIterative : public mds::mod::CModule
{
   public:
    //! Smart pointer type
    //! - Declares type tSmartPtr
    MDS_SHAREDPTR( CTestIterative );

    enum EFilter
    {
       LAM_GOODMAN,			// Lam - Goodman iterative filter
	   RL,					// Richardson-Lucy		
	   MLE,					// Maximum likelihood
      DMLE,             // Denoised maximum likelihood
	   DIV_LG,				// Dividing filter using Lam/Goodman
	   DIV_RL,				// Dividing filter using Richardson/Lucy
	   DIV_MLE,				// Dividing filter using MLE
      DIV_DMLE,         // Dividing filter using denoised MLE

       UNKNOWN_FILTER,              // Unknown filter

       EFilter_Min = LAM_GOODMAN,   // Minimal and
       EFilter_Max = DIV_DMLE		// maximal values
    };

	//! Used windowing scheme
	enum EWindow
	{
		TUKEY,	

		UNKNOWN_WINDOW,	// Unknown or not defined window

		EWindow_Min = TUKEY,
		EWindow_Max = TUKEY
	};


public:
    //! Default constructor
    CTestIterative(const std::string& sDescription);

    //! Virtual destructor
    virtual ~CTestIterative();

protected:
   //---------------------------------------
   // Common methods

    //! Virtual method called on startup
    virtual bool startup();

    //! Virtual method called by the processing thread
    virtual bool main();

    //! Called on console shutdown
    virtual void shutdown();

    //! Called on writing a usage statement
    virtual void writeExtendedUsage(std::ostream& Stream);

    //! filter images
    void filter();

    //! Convert string filter name to the unsigned value
    EFilter decryptFilter( const std::string & name );

	//! Convert string window name to the enum
	EWindow decryptWindow( const std::string & name );
   
    //! Save PNG - normalize and save image
    void savePNG( const mds::img::CFImage &image, mds::mod::CChannel * channel );

    //! Save complex image as png picture
    void savePNG( const mds::img::CComplexImage &image, mds::mod::CChannel * channel );

    //! Save complex image as a png picture - fname version
    void savePNG( const mds::img::CComplexImage & image, const std::string & fname );

    //! Filter by Lam-Goodman method
    void lamGoodman( const mds::img::CFImage &input, const mds::img::CFImage &psf );

    //! Filter by Richardson-Lucy method
    void richardsonLucy( const mds::img::CFImage &input, const mds::img::CFImage &psf );

    //! Filter by maximum-likelihood method
    void maximumLikelihood( const mds::img::CFImage &input, const mds::img::CFImage &psf );

    //! Filter by denoising mle method
    void denoisedMLE( const mds::img::CFImage &input, const mds::img::CFImage &psf );

    //! Use dividing filter
    void dividingFilter( const mds::img::CFImage &input, const mds::img::CFImage &psf );
protected:
   //! Used filter
   EFilter m_used_filter;

   //! Used window type
   EWindow m_used_window;

   //! Data images 
   mds::img::CFImagePtr m_img_PSF, m_img_Input;

   //! Output image
   mds::img::CComplexImagePtr m_img_Output;

   //! Number of steps
   mds::tSize m_steps;

   //! Dividing filter tile size
   unsigned m_tile_size;

   //! Lam-Goodman filter
   CLamGoodman m_lg_filter;

   //! Richardson-Lucy filter
   CRichardsonLucy m_rl_filter;

   //! Experimental dividing filter
   CDividingFilter m_div_filter;

   //! Maximum likelihood filter
   CMLE m_mle_filter;

   //! Denoising maximum likelihood
   CDenoisedMLE m_dmle_filter;

   //! Use silent mode
   bool m_silent;

};

// Define a postfix increment operator for EFilter.
inline CTestIterative::EFilter operator++( CTestIterative::EFilter &rs, int ) {
   return rs = (CTestIterative::EFilter)(rs + 1);
}

// Define a postfix increment operator for EWindow.
inline CTestIterative::EWindow operator++( CTestIterative::EWindow &rs, int ) {
	return rs = (CTestIterative::EWindow)( rs + 1 );
}

//==============================================================================
/*!
 * Smart pointer to the application.
 */
typedef CTestIterative::tSmartPtr     CTestIterativePtr;





// TestFDF_H_included
#endif

