////////////////////////////////////////////////////////////////////////////////////////////////////
//!\file src\CComplexMedian.cpp
//!
//!\author  Vit Stancl
//!\date 18.8.2011
//!\brief   Implements the complex median class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "CComplexMedian.h"

//! Comparing operators
bool compareRe( const CComplexMedian::tPixel p1, const CComplexMedian::tPixel p2 ) { return p1.re() < p2.re(); }
bool compareIm( const CComplexMedian::tPixel p1, const CComplexMedian::tPixel p2 ) { return p1.im() < p2.im(); }
bool compareMag( const CComplexMedian::tPixel p1, const CComplexMedian::tPixel p2 ) { return p1.getMag() < p2.getMag(); }

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   ! Filter - use real part median. 
//!
//!\param   input             The input. 
//!\param [in,out]   output   The output. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CComplexMedian::filterRe(const tImage &input, tImage &output)
{
   // Image sizes
   mds::tSize w( input.getXSize() ), h( input.getYSize() );

   // Create input image margined copy
   tImage data( w, h, m_filter_size / 2 );
   data.fillMargin( tPixel( 0.0f ) );

   // Copy input image
   for( mds::tSize y = 0; y < h; ++y )
   {
      mds::base::memCopy< tPixel >( data.getPtr( 0, y ), input.getPtr( 0, y ), w );
   }

   // Where should be the median 
   mds::tSize center( m_data_size / 2 );

   // Do filtering
   for( mds::tSize y = 0; y < h; ++y )
   {
      for( mds::tSize x = 0; x < w; ++x )
      {
         // Fill array
         data.copyWindow( x, y, m_filter_size, &m_data[ 0 ] );

         // Write median
         output( x, y ) = findMedian( m_data, compareRe );
      }
   }
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   ! Filter - use real part median. 
//!
//!\param   input             The input. 
//!\param [in,out]   output   The output. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CComplexMedian::filterIm(const tImage &input, tImage &output)
{
	// Image sizes
   mds::tSize w( input.getXSize() ), h( input.getYSize() );

   // Create input image margined copy
   tImage data( w, h, m_filter_size / 2 );
   data.fillMargin( tPixel( 0.0f ) );

   // Copy input image
   for( mds::tSize y = 0; y < h; ++y )
   {
      mds::base::memCopy< tPixel >( data.getPtr( 0, y ), input.getPtr( 0, y ), w );
   }

   // Where should be the median 
   mds::tSize center( m_data_size / 2 );

   // Do filtering
   for( mds::tSize y = 0; y < h; ++y )
   {
      for( mds::tSize x = 0; x < w; ++x )
      {
         // Fill array
         data.copyWindow( x, y, m_data_size, &m_data[ 0 ] );

         // Write median
         output( x, y ) = findMedian( m_data, compareIm );
      }
   }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   ! Filter - use magnitude median. 
//!
//!\param   input             The input. 
//!\param [in,out]   output   The output. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CComplexMedian::filterMag(const tImage &input, tImage &output)
{
	// Image sizes
   mds::tSize w( input.getXSize() ), h( input.getYSize() );

   // Create input image margined copy
   tImage data( w, h, m_filter_size / 2 );
   data.fillMargin( tPixel( 0.0f ) );

   // Copy input image
   for( mds::tSize y = 0; y < h; ++y )
   {
      mds::base::memCopy< tPixel >( data.getPtr( 0, y ), input.getPtr( 0, y ), w );
   }

   // Where should be the median 
   mds::tSize center( m_data_size / 2 );

   // Do filtering
   for( mds::tSize y = 0; y < h; ++y )
   {
      for( mds::tSize x = 0; x < w; ++x )
      {
         // Fill array
         data.copyWindow( x, y, m_data_size, &m_data[ 0 ] );

         // Write median
         output( x, y ) = findMedian( m_data, compareMag );
      }
   }
}

