////////////////////////////////////////////////////////////////////////////////////////////////////
//\file  D:\BioMarker\Software\Iterative\input\ComplexFilter.cpp
//
//\brief Implements the complex filter class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "ComplexFilter.h"
#include <assert.h>



////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn CComplexFilter::CComplexFilter(void)
//
//\brief ! Constructor - simple. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CComplexFilter::CComplexFilter(void)
   : m_dataW( 0 ), m_dataH( 0 )
   , m_W( 0 ), m_H( 0 )
   , m_kernelW( 0 ), m_kernelH( 0 )
   , m_paddingW( 0 ), m_paddingH( 0 )
{
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn CComplexFilter::CComplexFilter(const tIImage &input, const tIImage &psf)
//
//\brief ! Constructor - initializing. 
//
//\param input The input. 
//\param psf   The psf. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CComplexFilter::CComplexFilter(const tIImage &input, const tIImage &psf)
{
   assert( init( input, psf ) );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CComplexFilter::init(const tIImage &input, const tIImage &psf)
//
//\brief ! Initialize - set input, set PSF. 
//
//\param input The input. 
//\param psf   The psf. 
//
//\return   true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CComplexFilter::init(const tIImage &input, const tIImage &psf)
{
   // Get data sizes
   m_dataW = input.getXSize(); m_dataH = input.getYSize();

   // Get kernel sizes
   m_kernelW = psf.getXSize(); m_kernelH = psf.getYSize();

   // Compute working image size - must be greater to handle cyclic nature of convolution 
   // using multiplication in the Fourier domain
   m_W = m_kernelW + m_dataW; m_H = m_kernelH + m_dataH;

   // Compute padding size
   m_paddingW = m_kernelW / 2; m_paddingH = m_kernelH / 2;

   // Create input image copy
   m_input = new tImage( m_W, m_H, 0 );

   if( m_input.get() == 0 )
      return false;

   // Fill image with zeroes
   m_input->fillEntire( tPixel( 0, 0 ) );

   double ps( 0.0 );

   // Copy input image
   int x, y;
   tPixel v;
   for( y = 0; y < m_dataH; ++y )
   {
      for( x = 0; x < m_dataW; ++x )
      {
         v = input.get( x, y );
         m_input->set( x + m_paddingW, y + m_paddingH, v );      // From the beginning?
         ps += v.re();
      }
   }

    // Left and right padding
   int xs, ys, xd, yd;
   for( y = 0; y < m_dataH; ++y )
   {
      ys = yd = y + m_paddingH;
      for( x = 0; x < m_paddingW; ++x )
      {
         xs = x + m_paddingW;
         xd = m_paddingW - 1 - x;
         
         v = m_input->get( xs, ys );
         ps += v.re();
         m_input->set( xd, yd, v );
         
         xs = m_paddingW + m_dataW - 1 - x;
         xd = x + m_paddingW + m_dataW;

         v = m_input->get( xs, ys );
         ps += v.re();
         m_input->set( xd, yd, v );
      }
   }

   // Top and bottom padding
   for( x = 0; x < m_dataW; ++x )
   {
      xs = xd = x + m_paddingW;
      for( y = 0; y < m_paddingH; ++y )
      {
         ys = y + m_paddingH;
         yd = m_paddingH - 1 - y;

         v = m_input->get( xs, ys );
         ps += v.re();
         m_input->set( xd, yd, v );

         ys = m_paddingH + m_dataH - 1 - y;
         yd = y + m_paddingH + m_dataH;

         v = m_input->get( xs, ys );
         ps += v.re();
         m_input->set( xd, yd, v );
      }
   }

   // Corners
   for( y = 0; y < m_paddingH; ++y )
   {
      for( x = 0; x < m_paddingW; ++x )
      {
         // Top left
         xs = x + m_paddingW; 
         ys = y + m_paddingH;
         xd = m_paddingW - 1 - x;
         yd = m_paddingH - 1 - y;
         
         v = m_input->get( xs, ys );
         ps += v.re();
         m_input->set( xd, yd, v );

         // Top right
         xs = m_paddingW + m_dataW - 1 - x;
         ys = y + m_paddingH;
         xd = m_paddingW + m_dataW + x;
         yd = m_paddingH - 1 - y;
         
         v = m_input->get( xs, ys );
         ps += v.re();
         m_input->set( xd, yd, v );

         // Botom left
         xs = x + m_paddingW; 
         ys = m_paddingH + m_dataH - 1 - y;
         xd = m_paddingW - 1 - x;
         yd = m_paddingH + m_dataH + y;
         
         v = m_input->get( xs, ys );
         ps += v.re();
         m_input->set( xd, yd, v );

         // Bottom right
         xs = m_paddingW + m_dataW - 1 - x;
         ys = m_paddingH + m_dataH - 1 - y;
         xd = m_paddingW + m_dataW + x;
         yd = m_paddingH + m_dataH + y;
         
         v = m_input->get( xs, ys );
         ps += v.re();
         m_input->set( xd, yd, v );


      }
   }

   // Normalize input image
   //*m_input *= 1.0/ps;

   // Create PSF image copy and shift it so the psf center is at the origin
   m_psf = new tImage( m_W, m_H, 0 );

   if( m_psf.get() == 0 )
      return false;

   m_psf->fillEntire( tPixel( 0.0f, 0.0f ) );
   int dx, dy;

   ps = 0.0;

   for( y = 0; y < m_kernelH; ++y )
   {
      for( x = 0; x < m_kernelW; ++x )
      {
         v = psf.get( x, y );

         dx = x - int(m_kernelW) / 2;
         dx = ( dx < 0 ) ? dx + m_W : dx;

         dy = y - int(m_kernelH) / 2;
         dy = ( dy < 0 ) ? dy + m_H : dy;

         m_psf->set( dx, dy, v );
         ps += v.re();
      }
   }

   // Normalize psf
   if( ps != 0 )
      *m_psf *= 1.0/ps;
//      *m_input *= 1.0/ps;

   // Create padded output image
   m_output = new tImage( m_W, m_H, 0 );
   m_output->fillEntire( tPixel( 0, 0 ) );

   // Create input image in the frequency domain
   m_inputFD = new tImage( m_W, m_H, 0 );
   if( m_inputFD.get() == 0 )
      return false;

   // Create psf function copy in th frequency domain
   m_psfFD = new tImage( m_W, m_H, 0 );
   if( m_psfFD.get() == 0 )
      return false;

   // Compute fft of input image and estimated psf 
   doFFT();

   return(true);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CComplexFilter::doFFT(void)
//
//\brief ! Do FFT operation on buffers. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::doFFT(void)
{
   tPixel sum1, sum2;

   // Convert input image
   fft( *m_input, *m_inputFD );

   // Convert psf
   fft( *m_psf, *m_psfFD );

   float scale( 1.0f / float( m_W * m_H ) );

   *m_input *= scale;
   *m_psf *= scale;

//   sum1 = mds::img::getSumOfSquares< tPixel, tImage >( * m_inputFD );
//   sum2 = mds::img::getSumOfSquares< tPixel, tImage >( * m_psfFD );
   // Rescale images
   
 //  *m_inputFD /= sum1;
 //  *m_psfFD /= sum2;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CComplexFilter::doIFFT(void)
//
//\brief ! Do inverse FFT operation on buffers. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::doIFFT(void)
{
   // Convert input image
   ifft( *m_inputFD, *m_output, true );

   // Convert psf
   ifft( *m_psfFD, *m_psf, true );	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CComplexFilter::fft(const tImage &src, tImage &dst)
//
//\brief ! Do fft on given image. 
//
//\param src            Source for the. 
//\param [in,out] dst   Destination for the. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::fft(const tImage &src, tImage &dst)
{
   // Image sizes array
   int imageSizes[2] = { src.getXSize(), src.getYSize() };

	fftwf_plan plan = fftwf_plan_many_dft( 2, imageSizes, 1, 
                                 (fftwf_complex * )src.getPtr( 0, 0 ), 
                                 imageSizes, 
                                 src.getXOffset(), 
                                 0,
                                 (fftwf_complex *)dst.getPtr( 0, 0 ),
                                 imageSizes,
                                 dst.getXOffset(),
                                 0,
                                 FFTW_FORWARD,
                                 FFTW_ESTIMATE );

   fftwf_execute( plan );
   fftwf_destroy_plan( plan );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CComplexFilter::ifft(const tImage &src, tImage &dst)
//
//\brief ! Do inverse fft on given image. 
//
//\param src            Source for the. 
//\param [in,out] dst   Destination for the. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::ifft(const tImage &src, tImage &dst, bool bRescale)
{
// Image sizes array
   int imageSizes[2] = { src.getXSize(), src.getYSize() };

	fftwf_plan plan = fftwf_plan_many_dft( 2, imageSizes, 1, 
                                 (fftwf_complex * )src.getPtr( 0, 0 ), 
                                 imageSizes, 
                                 src.getXOffset(), 
                                 0,
                                 (fftwf_complex *)dst.getPtr( 0, 0 ),
                                 imageSizes,
                                 dst.getXOffset(),
                                 0,
                                 FFTW_BACKWARD,
                                 FFTW_ESTIMATE );

   fftwf_execute( plan );
   fftwf_destroy_plan( plan );	

   if( bRescale )
      dst /= float( src.getXSize() * src.getYSize() );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CComplexFilter::normalizeSq(tImage &image)
//
//\brief ! Normalize image by squared sum. 
//
//\param [in,out] image The image. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::normalizeSq(tImage &image)
{
   tPixel sum = mds::img::getSumOfSquares< tPixel, tImage >( image );
   image /= sum.re();
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CComplexFilter::copy(const tImage &src, tImage &dst)
//
//\brief ! Copy one buffer to the other. 
//
//\param src            Source for the. 
//\param [in,out] dst   Destination for the. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::copy(const tImage &src, tImage &dst)
{
   mds::tSize w( src.getXSize() ), h( src.getYSize() );

   for( mds::tSize y = 0; y < h; ++y )
   {
      mds::base::memCopy( dst.getPtr( 0, y ), src.getPtr( 0, y ), w );
   }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CRichardsonLucy::multiply(const tImage &op1, const tImage &op2, tImage &result)
//
//\brief ! Element by element multiply images: result = op1 * op2. 
//
//\param op1               The first operation. 
//\param op2               The second operation. 
//\param [in,out] result   The result. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::multiply(const tImage &op1, const tImage &op2, tImage &result)
{
  for( mds::tSize y = 0; y < m_H; ++y )
  {
     for( mds::tSize x = 0; x < m_W; ++x )
     {
        result( x, y ) = op1( x, y ) * op2( x, y );
     }
  }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CRichardsonLucy::divide(const tImage &op1, const tImage &op2, tImage &result,
//tPixel high)
//
//\brief ! Element by element divide images: result = op1/op2. 
//
//\param op1               The first operation. 
//\param op2               The second operation. 
//\param [in,out] result   The result. 
//\param high              The high. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::divide(const tImage &op1, const tImage &op2, tImage &result, tPixel high)
{
  for( mds::tSize y = 0; y < m_H; ++y )
  {
     for( mds::tSize x = 0; x < m_W; ++x )
     {
        result( x, y ) = op1( x, y ) / op2( x, y );
     }
  }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CComplexFilter::add(const tImage &op1, const tImage &op2, tImage &result)
//
//\brief ! Element by element addition: result = op1 + op2. 
//
//\param op1               The first operation. 
//\param op2               The second operation. 
//\param [in,out] result   The result. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::add(const tImage &op1, const tImage &op2, tImage &result)
{
	for( mds::tSize y = 0; y < m_H; ++y )
  {
     for( mds::tSize x = 0; x < m_W; ++x )
     {
        result( x, y ) = op1( x, y ) + op2( x, y );
     }
  }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CComplexFilter::sub(const tImage &op1, const tImage &op2, tImage &result)
//
//\brief ! Element by element subtraction: result = op1 - op2. 
//
//\param op1               The first operation. 
//\param op2               The second operation. 
//\param [in,out] result   The result. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::sub(const tImage &op1, const tImage &op2, tImage &result)
{
	for( mds::tSize y = 0; y < m_H; ++y )
  {
     for( mds::tSize x = 0; x < m_W; ++x )
     {
        result( x, y ) = op1( x, y ) - op2( x, y );
     }
  }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CComplexFilter::asub(const tImage &op1, const tImage &op2, tImage &result)
//
//\brief ! Element by element distance calculation: result = abs( op1 - op2 ) 
//
//\param op1               The first operation. 
//\param op2               The second operation. 
//\param [in,out] result   The result. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::asub(const tImage &op1, const tImage &op2, tImage &result)
{
   tPixel v, o1, o2;
	for( mds::tSize y = 0; y < m_H; ++y )
  {
     for( mds::tSize x = 0; x < m_W; ++x )
     {
        o1 = op1( x, y );
        o2 = op2( x, y );
        v.re() = abs( o1.re() - o2.re() );
        v.im() = abs( o1.im() - o2.im() );

        result( x, y ) = v;
     }
  }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CComplexFilter::copywindow(const tImage &src, tImage &dst, mds::tSize x, mds::tSize y,
//mds::tSize w, mds::tSize h)
//
//\brief ! Copy subwindow. 
//
//\param src            Source for the. 
//\param [in,out] dst   Destination for the. 
//\param x              The x coordinate. 
//\param y              The y coordinate. 
//\param w              The w. 
//\param h              The. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CComplexFilter::copyWindow(const tImage &src, tImage &dst, mds::tSize x, mds::tSize y, mds::tSize w, mds::tSize h)
{
   // Get image sizes
   mds::tSize sw( src.getXSize() ), sh( src.getYSize() );
   mds::tSize dw( dst.getXSize() ), dh( dst.getYSize() );

   // Do tests
   if( x > sw || y > sh )
      return false;

   if( w > dw || h > dh )
      return false;

   // Compute common part
   int cw( mds::math::getMin( int( sw ) - int( x ), w ) );
   int ch( mds::math::getMin( int( sh ) - int( y ), h ) );

   for( mds::tSize yp = 0; y < ch; ++y )
   {
      mds::base::memCopy( dst.getPtr( 0, yp ), src.getPtr( x, y + yp ), cw );  
   }

   return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CComplexFilter::copyWindow(const tImage &src, tImage &dst, mds::tSize srcx,
//mds::tSize srcy, mds::tSize dstx, mds::tSize dsty, mds::tSize w, mds::tSize h)
//
//\brief ! Copy subwindow to the location. 
//
//\param src            Source for the. 
//\param [in,out] dst   Destination for the. 
//\param srcx           The srcx. 
//\param srcy           The srcy. 
//\param dstx           The dstx. 
//\param dsty           The dsty. 
//\param w              The w. 
//\param h              The. 
//
//\return   true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CComplexFilter::copyWindow(const tImage &src, tImage &dst, mds::tSize srcx, mds::tSize srcy, mds::tSize dstx, mds::tSize dsty, mds::tSize w, mds::tSize h)
{
   // Get image sizes
   mds::tSize sw( src.getXSize() ), sh( src.getYSize() );
   mds::tSize dw( dst.getXSize() ), dh( dst.getYSize() );

   // test source image size
   if( sw < srcx + w || sh < srcy + h )
      return false;

   // Test destination image size
   if( dw < dstx + w || dh < dsty + h )
      return false;
	
   // Copy data
   for( mds::tSize y = 0; y < h; ++y )
   {
      mds::base::memCopy( dst.getPtr( dstx, dsty + y ), src.getPtr( srcx, srcy + y ), w );  
   }

	return(true);
}


////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CComplexFilter::fft(const tImage &src, tImage &dst)
//
//\brief ! Do fft on given image. 
//
//\param src            Source for the. 
//\param [in,out] dst   Destination for the. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::fftf(const tIImage &src, tIImage &dst)
{
   tImage copy( src.getXSize(), src.getYSize(), 0 );
   copy.convert( src );

   // Image sizes array
   int imageSizes[2] = { src.getXSize(), src.getYSize() };

	fftwf_plan plan = fftwf_plan_many_dft( 2, imageSizes, 1, 
                                 (fftwf_complex * )copy.getPtr( 0, 0 ), 
                                 imageSizes, 
                                 src.getXOffset(), 
                                 0,
                                 (fftwf_complex *)copy.getPtr( 0, 0 ),
                                 imageSizes,
                                 dst.getXOffset(),
                                 0,
                                 FFTW_FORWARD,
                                 FFTW_ESTIMATE );

   fftwf_execute( plan );
   fftwf_destroy_plan( plan );

   dst.convert( copy );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CComplexFilter::ifft(const tImage &src, tImage &dst)
//
//\brief ! Do inverse fft on given image. 
//
//\param src            Source for the. 
//\param [in,out] dst   Destination for the. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CComplexFilter::ifftf(const tIImage &src, tIImage &dst)
{
   tImage copy( src.getXSize(), src.getYSize(), 0 );
   copy.convert( src );

// Image sizes array
   int imageSizes[2] = { src.getXSize(), src.getYSize() };

	fftwf_plan plan = fftwf_plan_many_dft( 2, imageSizes, 1, 
                                 (fftwf_complex * )copy.getPtr( 0, 0 ), 
                                 imageSizes, 
                                 src.getXOffset(), 
                                 0,
                                 (fftwf_complex *)copy.getPtr( 0, 0 ),
                                 imageSizes,
                                 dst.getXOffset(),
                                 0,
                                 FFTW_BACKWARD,
                                 FFTW_ESTIMATE );

   fftwf_execute( plan );
   fftwf_destroy_plan( plan );

    dst.convert( copy );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Square all values. 
//!
//!\param	src			Source for the. 
//!\param [in,out]	dst	Destination for the. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CComplexFilter::sq(const tImage &src, tImage &dst)
{
  for( mds::tSize y = 0; y < m_H; ++y )
  {
     for( mds::tSize x = 0; x < m_W; ++x )
     {
        dst( x, y ) = src( x, y )*src( x, y );
     }
  }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Create power spectrum. 
//!
//!\param	src			Source for the. 
//!\param [in,out]	dst	Destination for the. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CComplexFilter::pow(const tImage &src, tImage &dst)
{
	for( mds::tSize y = 0; y < m_H; ++y )
  {
     for( mds::tSize x = 0; x < m_W; ++x )
     {
		 dst( x, y ) = src( x, y ).getNorm();
     }
  }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Initialize - get frequency domain data. 
//!
//!\param	input	The input. 
//!\param	psf		The psf. 
//!
//!\return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////
bool CComplexFilter::initFD(const tImage &input, const tImage &psf)
{
	// psf and input sizes MUST be the same...
   
   mds::tSize iw( input.getXSize() ), ih( input.getYSize() );
   mds::tSize pw( psf.getXSize() ), ph( input.getYSize() );

   if( iw != pw || ih != ph )
      return false;

   // Set sizes
   m_W = iw; m_H = ph;

   // Create complex filter buffers
   m_input = new tImage( m_W, m_H, 0 );
   m_inputFD = new tImage( m_W, m_H, 0 );
   m_psf = new tImage( m_W, m_H, 0 );
   m_psfFD = new tImage( m_W, m_H, 0 );
   m_output = new tImage( m_W, m_H, 0 );

   // Copy input to the m_inputFD
   copy( input, *m_inputFD );

   // Do ifft to create spatial domain image
   ifft( *m_inputFD, *m_input, true );

   // Copy psf to the m_psfFD
   copy( psf, *m_psfFD );

   // Do ifft
   ifft( *m_psfFD, *m_psf, true );

   // Set padding to 0
   m_paddingH = m_paddingW = 0;

   // Set kernel to m_W, m_H
   m_kernelW = 0;
   m_kernelH = 0;

   // Set data sizes
   m_dataW = m_W;
   m_dataH = m_H;

   return(true);
}

