////////////////////////////////////////////////////////////////////////////////////////////////////
//!\file src\DenoisedMLE.cpp
//!
//!\author  Vit Stancl
//!\date 22.3.2011
//!\brief   Implements the denoised mle class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "DenoisedMLE.h"
#include "Constraint.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn CDenoisedMLE::CDenoisedMLE(void)
//
//\brief ! Constructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CDenoisedMLE::CDenoisedMLE(void)
{
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CDenoisedMLE::init(const tIImage &input, const tIImage &psf)
//
//\brief ! Initialize - set input, set PSF. 
//
//\param input The input. 
//\param psf   The psf. 
//
//\return   true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CDenoisedMLE::init(const tIImage &input, const tIImage &psf)
{
   // Call base class initialization
   CComplexFilter::init( input, psf );

   // Create input image copies
   m_wG = new tImage( m_W, m_H, 0 );
   m_wGfd = new tImage( m_W, m_H, 0 );

   copy( *m_input, *m_wG );
   copy( *m_inputFD, *m_wGfd );

   // Create psf copies
   m_wP = new tImage( m_W, m_H, 0 );
   m_wPfd = new tImage( m_W, m_H, 0 );

   copy( *m_psf, *m_wP );
   copy( *m_psfFD, *m_wPfd );

   // Create transformed psf copies
   m_wPT = new tImage( m_W, m_H, 0 );
   m_wPTfd = new tImage( m_W, m_H, 0 );

   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         m_wPT->set( m_W - 1 - x, m_H - 1 - y, m_wP->get( x, y ) );
//         m_wPTfd->set( m_W - 1 - x, m_H - 1 - y, m_wPfd->get( x, y ) );
      }
   }
	
   fft( *m_wPT, *m_wPTfd );

   // Create transformed input copies
   m_wGT = new tImage( m_W, m_H, 0 );
   m_wGTfd = new tImage( m_W, m_H, 0 );

   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         m_wGT->set( m_W - 1 - x, m_H - 1 - y, m_wG->get( x, y ) );
//         m_wPTfd->set( m_W - 1 - x, m_H - 1 - y, m_wPfd->get( x, y ) );
      }
   }
	
   fft( *m_wGT, *m_wGTfd );

   // Create shared image
   m_shared = new tImage( m_W, m_H, 0 );

	return(true);
}


////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CDenoisedMLE::computeShared(void)
//
//\brief ! Compute shared part. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CDenoisedMLE::computeShared(void)
{
   tImage buffer( m_W, m_H, 0 );
   
   // Prepare convolution - do fft of the operands
   fft( *m_wG, *m_wGfd );
   fft( *m_wP, *m_wPfd );

   // Multiply
   multiply( *m_wGfd, *m_wPfd, *m_shared );

   // Do ifft on the result
   ifft( *m_shared, *m_shared, true );

   // Compute noise image
   sub( *m_input, *m_shared, buffer );

   // Denoise
   denoise( buffer );

   // Finalize shared part
   add( *m_shared, buffer, buffer );
   divide( buffer, *m_shared, *m_shared );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CDenoisedMLE::step(void)
//
//\brief ! Do one iteration. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CDenoisedMLE::step(void)
{
	// Create some buffers - estimated image and kernel
   tImage eg( m_W, m_H, 0 ), ep( m_W, m_H, 0 );

	// Create current image estimate and current psf estimate transformed version
   tPixel v, sumg( 0.0f, 0.0f );
   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         v = m_wG->get( x, y );
         sumg += v;
         m_wGT->set( m_W - 1 - x, m_H - 1 - y, v );
         m_wPT->set( m_W - 1 - x, m_H - 1 - y, m_wP->get( x, y ) );
      }
   }

   fft( *m_wGT, *m_wGTfd );
   fft( *m_wPT, *m_wPTfd );

   // Get current P and G spectrum
   fft( *m_wG, *m_wGfd );
   fft( *m_wP, *m_wPfd );

   // Create "shared part" - I/(G*P)
  computeShared();

   // ESTIMATE IMAGE
   fft( *m_shared, eg );

   multiply( eg, *m_wPTfd, eg );
   ifft( eg, eg, true );
   multiply( *m_wG, eg, eg );
   

   // ESTIMATE PSF
   fft( *m_shared, ep );

   multiply( ep, *m_wPTfd, ep );
   ifft( ep, ep, true );

   multiply( *m_wP, ep, ep );
   //ep /= sumg;

   // Normalize psf estimate
   tPixel sum( mds::img::getSum< tPixel, tImage >( *m_wP ) );
   ep /= sum; // *= sum.getConj() / sum.getNorm();

   // Copy output
   copy( eg, *m_wG );
   copy( ep, *m_wP );
	
}
   

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn int CDenoisedMLE::filter(mds::tSize max_steps)
//
//\brief ! Filter. 
//
//\param max_steps   The maximum steps. 
//
//\return   . 
////////////////////////////////////////////////////////////////////////////////////////////////////

int CDenoisedMLE::filter(mds::tSize max_steps)
{
   mds::tSize current_step( 0 );	

   while( current_step < max_steps )
   {
      step();
      ++current_step;
   }

	return( max_steps );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CDenoisedMLE::getOutput(tImage &image)
//
//\brief ! Get filter output. 
//
//\param [in,out] image The image. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CDenoisedMLE::getOutput(tImage &image)
{
// Convert working image to the spatial domain
   // ifft( *m_wGfd, *m_wG );
   // ifft( *m_wPSFfd, *m_wPSF );

   // Copy output window (without padding)	
   for( mds::tSize y = 0; y < m_dataH; ++y )
   {
      mds::base::memCopy( image.getPtr( 0, y ), m_wG->getPtr( m_paddingW, y + m_paddingH ), m_dataW );
//     mds::base::memCopy( image.getPtr( 0, y ), m_wP->getPtr( m_paddingW, y + m_paddingH ), m_dataW );
 //     mds::base::memCopy( image.getPtr( 0, y ), m_wP->getPtr( 0, y ), m_dataW );
   }	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Get filter output - frequency domain. 
//!
//!\param [in,out]	image	The image. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CDenoisedMLE::getOutputFD(tImage &image)
{
	// Copy output window (without padding)	
   for( mds::tSize y = 0; y < m_dataH; ++y )
   {
      mds::base::memCopy( image.getPtr( 0, y ), m_wGfd->getPtr( m_paddingW, y + m_paddingH ), m_dataW );
     //mds::base::memCopy( image.getPtr( 0, y ), m_wPSF->getPtr( m_paddingW, y + m_paddingH ), m_dataW );
   }	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Initialize - use frequency domain data. 
//!
//!\param	input	The input. 
//!\param	psf		The psf. 
//!
//!\return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////
bool CDenoisedMLE::initFD(const tImage &input, const tImage &psf)
{
	
	// Call base class initialization
   CComplexFilter::initFD( input, psf );

   // Create input images copies
   m_wG = new tImage( m_W, m_H, 0 );
   m_wGfd = new tImage( m_W, m_H, 0 );

   copy( *m_input, *m_wG );
   copy( *m_inputFD, *m_wGfd );

   // Create psf copies
   m_wP = new tImage( m_W, m_H, 0 );
   m_wPfd = new tImage( m_W, m_H, 0 );

   copy( *m_psf, *m_wP );
   copy( *m_psfFD, *m_wPfd );

   // Create transformed psf copies
   m_wPT = new tImage( m_W, m_H, 0 );
   m_wPTfd = new tImage( m_W, m_H, 0 );

   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         m_wPT->set( m_W - 1 - x, m_H - 1 - y, m_wP->get( x, y ) );
//         m_wPTfd->set( m_W - 1 - x, m_H - 1 - y, m_wPfd->get( x, y ) );
      }
   }
	
   fft( *m_wPT, *m_wPTfd );

   // Create transformed input copies
   m_wGT = new tImage( m_W, m_H, 0 );
   m_wGTfd = new tImage( m_W, m_H, 0 );

   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         m_wGT->set( m_W - 1 - x, m_H - 1 - y, m_wG->get( x, y ) );
//         m_wPTfd->set( m_W - 1 - x, m_H - 1 - y, m_wPfd->get( x, y ) );
      }
   }
	
   fft( *m_wGT, *m_wGTfd );

   // Create shared image
   m_shared = new tImage( m_W, m_H, 0 );

	return(true);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   ! Denoise. 
//!
//!\param [in,out]   image The image. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CDenoisedMLE::denoise(tImage &image)
{
	
}

