////////////////////////////////////////////////////////////////////////////////////////////////////
//\file  D:\BioMarker\Software\Iterative\src\DividingFilter.cpp
//
//\brief Implements the dividing filter class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "DividingFilter.h"
#include <MDSTk/Image/mdsImageFunctions.h>

#define TILE_SIZE 8

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CDividingFilter::sizes(void)
//
//\brief ! Initialize image sizes. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CDividingFilter::sizes(void)
{
   // Sizes must be divisible by 4
   if( m_W % 4 != m_H % 4 != 0 )
      return false;
  

   // Try to compute tiles size
   m_tileW = m_W / ( m_divideW );
   m_tileH = m_H / ( m_divideH );
   m_subtileW = m_tileW / 2;
   m_subtileH = m_tileH / 2;


   // Create and fill sizes array
   m_sizes.resize( m_divideW * m_divideH );
   int vecpos( 0 );

   for( mds::tSize y = 0; y < m_divideH; ++y )
   {
      for( mds::tSize x = 0; x < m_divideW; ++x )
      {
         // Get current size
         STileDescriptror & tile( m_sizes[ vecpos++ ] );

         // Top left subtile position
         tile.x[ 0 ] = x * m_subtileW;
         tile.y[ 0 ] = y * m_subtileH;

         // Top right subtile position
         tile.x[ 1 ] = ( 2*m_divideW - x - 1 ) * m_subtileW;
         tile.y[ 1 ] = y * m_subtileH;

         // Bottom left 
         tile.x[ 2 ] = x * m_subtileW;
         tile.y[ 2 ] = ( 2*m_divideH - y - 1 ) * m_subtileH;

         // Bottom right
         tile.x[ 3 ] = ( 2*m_divideW - x - 1 ) * m_subtileW;
         tile.y[ 3 ] = ( 2*m_divideH - y - 1 ) * m_subtileH;
      }
   }

   return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CDividingFilter::split(void)
//
//\brief ! Split current image to the tiles. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CDividingFilter::split(const tImage & src, tImage & dst, const STileDescriptror & tile )
{
   // Copy subtiles from the source image
   // Top left
   copyWindow( src, dst, tile.x[ 0 ], tile.y[ 0 ], 0, 0, m_subtileW, m_subtileH );
   // Top right
   copyWindow( src, dst, tile.x[ 1 ], tile.y[ 1 ], m_subtileW, 0, m_subtileW, m_subtileH );
   // Bottom left
   copyWindow( src, dst, tile.x[ 2 ], tile.y[ 2 ], 0, m_subtileH, m_subtileW, m_subtileH );
   // Bottom right
   copyWindow( src, dst, tile.x[ 3 ], tile.y[ 3 ], m_subtileW, m_subtileH, m_subtileW, m_subtileH );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CDividingFilter::init(const tIImage &input, const tIImage &psf)
//
//\brief ! Initialize - set input, set PSF. 
//
//\param input The input. 
//\param psf   The psf. 
//
//\return   true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CDividingFilter::init(const tIImage &input, const tIImage &psf, bool padded /* = false */ )
{
   m_padded = padded;

   // Initialize base class
   if( ! CComplexFilter::init( input, psf ) )
      return false;

   // Create filters array
   if( ! createFilters( m_filterType ) )
	  return false;

   if( ! padded )
   {
      // Compute splitting sizes
      if( ! sizes() )
         return false;

      // Create image buffers
      tImage tinput( m_tileW, m_tileH, 0 );
      tImage tpsf( m_tileW, m_tileH, 0 );

      tSizesVector::iterator i;
      int c( 0 );
      for( i = m_sizes.begin(); i != m_sizes.end(); ++i )
      {
         // Do subdividing - data
         split( *m_inputFD, tinput, *i );

         // Do subdividing - psf
         split( *m_psfFD, tpsf, *i );

         // Initialize subfilter in frequency domain
         m_filters[ c ]->initFD( tinput, tpsf ); 

         ++c;
      }   
	
	   return(true);
   }

   // Compute splitting sizes
   if( ! sizesPadded() )
      return false;


   // Create image buffers
   tImage tinput;
   tImage tpsf;

   // Current tile sizes
   int w( 0 ), h( 0 );

   tSizesVector::iterator i;
   int c( 0 );
   for( i = m_sizes.begin(); i != m_sizes.end(); ++i )
   {
	  // Get current tile size	
	  w = i->w[ 0 ] + i->w[ 1 ];
	  h = i->h[ 0 ] + i->h[ 2 ];

	  // Resize buffers, if different.
	  if( w != tinput.getXSize() || h != tinput.getYSize() )
	  {
		  tinput.create( w, h, 0 );
		  tpsf.create( w, h, 0 );
	  }

      // Do subdividing - data
      splitPadded( *m_inputFD, tinput, *i );

      // Do subdividing - psf
      splitPadded( *m_psfFD, tpsf, *i );

      // Initialize subfilter in frequency domain
      m_filters[ c ]->initFD( tinput, tpsf ); 

      ++c;
   }   

   return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CDividingFilter::concatenate(void)
//
//\brief ! Concatenate vector back to the image. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CDividingFilter::concatenate(const tImage & src, tImage & dst, const STileDescriptror & tile )
{
      // Copy subtiles from the source image
      // Top left
      copyWindow( src, dst, 0, 0, tile.x[ 0 ], tile.y[ 0 ], m_subtileW, m_subtileH );
      // Top right
      copyWindow( src, dst, m_subtileW, 0, tile.x[ 1 ], tile.y[ 1 ], m_subtileW, m_subtileH );
      // Bottom left
      copyWindow( src, dst, 0, m_subtileH, tile.x[ 2 ], tile.y[ 2 ], m_subtileW, m_subtileH );
      // Bottom right
      copyWindow( src, dst, m_subtileW, m_subtileH, tile.x[ 3 ], tile.y[ 3 ], m_subtileW, m_subtileH );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn CDividingFilter::CDividingFilter(void)
//
//\brief ! Constructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CDividingFilter::CDividingFilter( EFilterType filter, bool padded /* = false */)
   : m_filterType( filter )
   , m_divideW( TILE_SIZE )
   , m_divideH( TILE_SIZE )
   , m_filters( 0 )
   , m_tileW( 0 )
   , m_tileH( 0 )
   , m_subtileW( 0 )
   , m_subtileH( 0 )
   , m_window( 1, 0 )
   , m_padded( padded )
   , m_tilePaddingW( 4 )
   , m_tilePaddingH( 4 )
{
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn CDividingFilter::~CDividingFilter(void)
//
//\brief ! Destructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CDividingFilter::~CDividingFilter(void)
{
	if( m_filters != 0 )
	{
	  // delete filters
	  for( mds::tSize i = 0; i < m_divideW * m_divideH; ++i )
		  delete m_filters[ i ];

	  // delete array
      delete [] m_filters;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CDividingFilter::step(void)
//
//\brief ! Do one iteration. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CDividingFilter::step(void)
{
   for( int c = 0; c < m_divideW * m_divideH; ++c )
   {
      m_filters[ c ]->step();
   }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn int CDividingFilter::filter(mds::tSize max_steps)
//
//\brief ! Filter. 
//
//\param max_steps   The maximum steps. 
//
//\return   . 
////////////////////////////////////////////////////////////////////////////////////////////////////

int CDividingFilter::filter(mds::tSize max_steps)
{
   for( int c = 0; c < m_divideW * m_divideH; ++c )
   {
      m_filters[ c ]->filter( max_steps );
   }

	return(max_steps);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CDividingFilter::getOutput(tImage &image)
//
//\brief ! Get filter output. 
//
//\param [in,out] image The image. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CDividingFilter::getOutput(tImage &image)
{
	// Create image buffers
	tImage buffer( m_tileW + m_paddingW * 2, m_tileH + m_paddingH * 2, 0 );

   tSizesVector::iterator i;
   int c( 0 );
   for( i = m_sizes.begin(); i != m_sizes.end(); ++i )
   {
      // Get subimage
      m_filters[ c ]->getOutputFD( buffer );

      // Place it to the real output image
      if( m_padded )
         concatenatePadded( buffer, *m_inputFD, *i );
      else
         concatenate( buffer, *m_inputFD, *i );

      ++c;
   }

   float sum(0.0f);
   // Compute square root of all cells
   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         sum += m_inputFD->get( x, y ).getNorm();
      }
   }

  // if( sum != 0.0 )
  //    *m_inputFD /= sum;

   // Do ifft
   ifft( *m_inputFD, *m_input, true );

   // No negative pixels
   for( int i = 0; i < m_H; ++i )
   {
      for( int j = 0; j < m_W; ++j )
      {
         if( m_input->get( i, j ).re() < 0 )
         {
            m_input->set( i, j, tPixel( 0, 0 ) );
         }
      }
   }
   
   // Copy output window (without padding)	
   for( mds::tSize y = 0; y < m_dataH; ++y )
   {
	   for( mds::tSize x = 0; x < m_dataW; ++x )
	   {
		   image( x, y ) = m_input->get( x + m_paddingW, y + m_paddingH );

//      mds::base::memCopy( image.getPtr( 0, y ), m_inputFD->getPtr( m_paddingW, y + m_paddingH ), m_dataW );
     //mds::base::memCopy( image.getPtr( 0, y ), m_wPSF->getPtr( m_paddingW, y + m_paddingH ), m_dataW );
		}
   }

   normalizeSq( image );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CDividingFilter::sizesPadded(void)
//
//\brief ! Initialize tile sizes array - padded version. 
//
//\return   true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CDividingFilter::sizesPadded(void)
{
	// Sizes must be divisible by 4
   if( m_W % 4 != m_H % 4 != 0 )
      return false;
  

   // Try to compute tiles size


   m_tileW = m_W / ( m_divideW );
   m_tileH = m_H / ( m_divideH );
   m_subtileW = m_tileW / 2;
   m_subtileH = m_tileH / 2;

   // Create windowing function image
   m_window.init( m_tileW, m_tileH, m_tilePaddingW, m_tilePaddingH );

   // Create and fill sizes array
   m_sizes.resize( m_divideW * m_divideH );
   int vecpos( 0 );

   for( mds::tSize y = 0; y < m_divideH; ++y )
   {
      for( mds::tSize x = 0; x < m_divideW; ++x )
      {

         // Get current size
         STileDescriptror & tile( m_sizes[ vecpos++ ] );

         // Top left subtile position
         tile.x[ 0 ] = x * m_subtileW;
         tile.y[ 0 ] = y * m_subtileH;

         // Top right subtile position
         tile.x[ 1 ] = ( 2*m_divideW - x - 1 ) * m_subtileW;
         tile.y[ 1 ] = y * m_subtileH;

         // Bottom left 
         tile.x[ 2 ] = x * m_subtileW;
         tile.y[ 2 ] = ( 2*m_divideH - y - 1 ) * m_subtileH;

         // Bottom right
         tile.x[ 3 ] = ( 2*m_divideW - x - 1 ) * m_subtileW;
         tile.y[ 3 ] = ( 2*m_divideH - y - 1 ) * m_subtileH;

         for( int i = 0; i < 4; ++i )
         {
            // Sizes - padded from all four sides
            tile.w[i] = m_subtileW + 2 * m_tilePaddingW;
            tile.h[i] = m_subtileH + 2 * m_tilePaddingH;

            // Move window 
            tile.x[ i ] -= m_tilePaddingW;
            tile.y[ i ] -= m_tilePaddingH;
            
         }

         // Do tile clipping
         cutTile( tile );

      }
   }

   return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CDividingFilter::cutTile(int &x, int &y, int &w, int &h)
//
//\brief ! Cut padded tile size and position to the image size. 
//
//\param [in,out] x  The x coordinate. 
//\param [in,out] y  The y coordinate. 
//\param [in,out] w  The w. 
//\param [in,out] h  The. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CDividingFilter::cutTile(STileDescriptror & tile)
{
   for( int i = 0; i < 4; ++i )
   {

	   if( tile.x[i] < 0 )
      {
         tile.w[i] += tile.x[i]; // shorten tile width
         tile.wx[i] -= tile.x[i]; // Move to the right in the windowing function
         tile.x[i] = 0;
      }

      if( tile.y[i] < 0 )
      {
         tile.h[i] += tile.y[i]; // shorten tile height
         tile.wy[i] -= tile.y[i]; // move down in the windowing function tile
         tile.y[i] = 0;
      }

      if( (tile.x[i] + tile.w[i]) > m_W )
      {
         tile.w[i] = m_W - tile.x[i];
      }

      if( (tile.y[i] + tile.h[i]) > m_H )
      {
         tile.h[i] = m_H - tile.y[i];
      }
   }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CDividingFilter::splitPadded(const tImage &src, tImage &dst,
//const STileDescriptror &tile)
//
//\brief ! Split image with paddings. 
//
//\param src            Source for the. 
//\param [in,out] dst   Destination for the. 
//\param tile           The tile. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CDividingFilter::splitPadded(const tImage &src, tImage &dst, const STileDescriptror &tile)
{
   split( src, dst, tile );	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CDividingFilter::concatenatePadded(const tImage &src, tImage &dst,
//const STileDescriptror &tile)
//
//\brief ! Concatenate back padded tiles. 
//
//\param src            Source for the. 
//\param [in,out] dst   Destination for the. 
//\param tile           The tile. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CDividingFilter::concatenatePadded(const tImage &src, tImage &dst, const STileDescriptror &tile)
{
	addWindowed( src, dst, m_window, 0, 0, tile.x[0], tile.y[0], tile.wx[0], tile.wy[0], tile.w[0], tile.h[0] );
	addWindowed( src, dst, m_window, tile.w[0], 0, tile.x[1], tile.y[1], tile.wx[1], tile.wy[1], tile.w[1], tile.h[1] );
	addWindowed( src, dst, m_window, 0, tile.h[0], tile.x[0], tile.y[2], tile.wx[2], tile.wy[2], tile.w[2], tile.h[2] );
	addWindowed( src, dst, m_window, tile.w[0], tile.h[0], tile.x[3], tile.y[3], tile.wx[3], tile.wy[3], tile.w[3], tile.h[3] );
	/*
	for( int i = 0; i < 4; ++i )
   {
      int tx(0), ty(0);
      for( int y = tile.y[i]; y < tile.y[i] + tile.h[i]; ++y, ++ty )
      {
         tx = 0;
         for( int x = tile.x[i]; x < tile.x[i] + tile.w[i]; ++x, ++tx )
         {
            // Apply windowing and add it to the destination image
            dst( x, y ) += m_window.apply< tPixel >( tx + tile.wx[i], ty + tile.wy[i], src.get( tx, ty ) );
         }
      }
   }
   */
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Add values from the window to the image. 
//!
//!\param	src			Source for the. 
//!\param [in,out]	dst	Destination for the. 
//!\param	window		The window. 
//!\param	sx			The sx. 
//!\param	sy			The sy. 
//!\param	dx			The dx. 
//!\param	dy			The dy. 
//!\param	tx			The transmit. 
//!\param	ty			The ty. 
//!\param	w			The width. 
//!\param	h			The height. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CDividingFilter::addWindowed(const tImage &src, tImage &dst, const CWindow &window, int sx, int sy, int dx, int dy, int tx, int ty, int w, int h)
{
	for( int y = 0; y < h; ++y )
	{
		for( int x = 0; x < w; ++x )
		{
			dst( dx + x, dy + y ) = window.apply< tPixel > ( tx + x, ty + y, src( sx + x, sy + y ) );
		}
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Create filters. 
//!
//!\param	type	The type. 
//!
//!\return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////
bool CDividingFilter::createFilters(EFilterType type)
{
	mds::tSize arraySize( m_divideW * m_divideH );
	
	m_filters = new CComplexFilter * [arraySize];
	
	switch( type )
	{
	case FT_LG:
		
		for( mds::tSize i = 0; i < arraySize; ++i )
			m_filters[ i ] = new CLamGoodman();
		break;

	case FT_RL:
		
		for( mds::tSize i = 0; i < arraySize; ++i )
			m_filters[ i ] = new CRichardsonLucy();
		break;

	case FT_MLE:
		
		for( mds::tSize i = 0; i < arraySize; ++i )
			m_filters[ i ] = new CMLE();
		break;

	case FT_DMLE:
		
		for( mds::tSize i = 0; i < arraySize; ++i )
         m_filters[ i ] = new CDenoisedMLE();
		break;

	default:
		return false; // something is wrong...
	}

	return( m_filters != 0 );


}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Get filter output - frequency domain. 
//!
//!\param [in,out]	image	The image. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CDividingFilter::getOutputFD(tImage &image)
{
// Create image buffers
	tImage buffer( m_tileW + m_paddingW * 2, m_tileH + m_paddingH * 2, 0 );

   tSizesVector::iterator i;
   int c( 0 );
   for( i = m_sizes.begin(); i != m_sizes.end(); ++i )
   {
      // Get subimage
      m_filters[ c ]->getOutputFD( buffer );

      // Place it to the real output image
      if( m_padded )
         concatenatePadded( buffer, *m_inputFD, *i );
      else
         concatenate( buffer, *m_inputFD, *i );

      ++c;
   }

   float sum(0.0f);
   // Compute square root of all cells
   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         sum += m_inputFD->get( x, y ).getNorm();
      }
   }

  // if( sum != 0.0 )
  //    *m_inputFD /= sum;

   // Do ifft
   ifft( *m_inputFD, *m_input, true );

   // No negative pixels
   for( int i = 0; i < m_H; ++i )
   {
      for( int j = 0; j < m_W; ++j )
      {
         if( m_input->get( i, j ).re() < 0 )
         {
            m_input->set( i, j, tPixel( 0, 0 ) );
         }
      }
   }

   fft( *m_input, image );
}

