////////////////////////////////////////////////////////////////////////////////////////////////////
//\file  D:\BioMarker\Software\Iterative\src\NoiseEstimation.cpp
//
//\brief Implements the noise estimation class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "NoiseEstimation.h"
#include <fftw3.h>

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Estimate image noise. 
//!
//!\param	input			The input image. 
//!\param [in,out]	output	The output image. 
////////////////////////////////////////////////////////////////////////////////////////////////////
bool CFBDP::estimate(const tImage &input, tImage &output)
{
	// Get image size
	mds::tSize x, y, w( input.getXSize() ), h( input.getYSize() );


	// Step 1 - compute DFT of the input image, compute power spectrum and logarithmic power spectrum
	// compute threshold
	//-----------------------------------------------------------------------------------------------
	tCImagePtr ibuffer( new tCImage( w, h, 0 ) );
	tCImagePtr obuffer( new tCImage( w, h, 0 ) );

	// convert input image to the complex one
	for( y = 0; y < h; ++y ) {
		for( x = 0; x < w; ++x ) {

			ibuffer->set( x, y, tCPixel( input( x, y ) ) );
		}
	}

	// Create input image spectrum
	fft( *ibuffer, *ibuffer );

	// do estimation from the spectra
	if( ! estimate( *ibuffer, *obuffer ) )
		return false;

	// Do ifft of the output image
	ifft( *obuffer, *obuffer );

	// convert complex output image
	for( y = 0; y < h; ++y ) {
		for( x = 0; x < w; ++x ) {

			output.set( x, y, obuffer->get( x, y ).re() );
		}
	}

	return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Get tile mean. 
//!
//!\param	tx		The transmit. 
//!\param	ty		The ty. 
//!\param	data	The data. 
//!
//!\return	The tile mean. 
////////////////////////////////////////////////////////////////////////////////////////////////////
float CFBDP::getTileMean(mds::tSize tx, mds::tSize ty, const tImage &data)
{
	mds::tSize x, y, sx( tx*m_tile_size ), sy( ty * m_tile_size );
	mds::tSize ex( sx + m_tile_size ), ey( sy + m_tile_size );

	float mean( 0.0f );

	for( y = sy; y < ey; ++y ) {
		for( x = sx; x < ex; ++x ) {
			mean += data( x, y );
		}
	}


	return( mean / float( m_tile_size * m_tile_size ) );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Fill complex tile with value. 
//!
//!\param	tx				The transmit. 
//!\param	ty				The ty. 
//!\param [in,out]	data	The data. 
//!\param	value			The value. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CFBDP::fillTile(mds::tSize tx, mds::tSize ty, tCImage &data, tCPixel value)
{
	mds::tSize x, y, sx( tx*m_tile_size ), sy( ty * m_tile_size );
	mds::tSize ex( sx + m_tile_size ), ey( sy + m_tile_size );


	for( y = sy; y < ey; ++y ) {
		for( x = sx; x < ex; ++x ) {
			data.set( x, y, value );
		}
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Fast Fourier transform. 
//!
//!\param	src			Source for the. 
//!\param [in,out]	dst	Destination for the. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CNoiseEstimatorBase::fft(const tCImage &src, tCImage &dst)
{
	// Image sizes array
   int imageSizes[2] = { src.getXSize(), src.getYSize() };

	fftwf_plan plan = fftwf_plan_many_dft( 2, imageSizes, 1, 
                                 (fftwf_complex * )src.getPtr( 0, 0 ), 
                                 imageSizes, 
                                 src.getXOffset(), 
                                 0,
                                 (fftwf_complex *)dst.getPtr( 0, 0 ),
                                 imageSizes,
                                 dst.getXOffset(),
                                 0,
                                 FFTW_FORWARD,
                                 FFTW_ESTIMATE );

   fftwf_execute( plan );
   fftwf_destroy_plan( plan );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Inverse Fourier transform. 
//!
//!\param	src			Source for the. 
//!\param [in,out]	dst	Destination for the. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CNoiseEstimatorBase::ifft(const tCImage &src, tCImage &dst)
{
	// Image sizes array
   int imageSizes[2] = { src.getXSize(), src.getYSize() };

	fftwf_plan plan = fftwf_plan_many_dft( 2, imageSizes, 1, 
                                 (fftwf_complex * )src.getPtr( 0, 0 ), 
                                 imageSizes, 
                                 src.getXOffset(), 
                                 0,
                                 (fftwf_complex *)dst.getPtr( 0, 0 ),
                                 imageSizes,
                                 dst.getXOffset(),
                                 0,
                                 FFTW_BACKWARD,
                                 FFTW_ESTIMATE );

   fftwf_execute( plan );
   fftwf_destroy_plan( plan );	

 //  if( bRescale )
     dst /= float( src.getXSize() * src.getYSize() );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Estimate image noise - from spectrum to spectrum. 
//!
//!\param	input			The input. 
//!\param [in,out]	output	The output. 
//!
//!\return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////
bool CFBDP::estimate(const tCImage &input, tCImage &output)
{
	mds::tSize x, y;

	// Do some tests
	mds::tSize w( input.getXSize() ), h( input.getYSize() );
	if( w % m_tile_size != 0 || h % m_tile_size != 0 )
		return false;

	// Prepare corner tile
	createCornerTile( input, false );

	// Step 1 - compute DFT of the input image, compute power spectrum and logarithmic power spectrum
	// compute threshold
	//-----------------------------------------------------------------------------------------------
	tCImagePtr buffer( new tCImage( w, h, 0 ) );
	tImagePtr power_spectrum( new tImage( w, h, 0 ) );
	tImagePtr log_spectrum( new tImage( w, h, 0 ) );

	float gmin, gmax, mean( 0.0f );
	float v = log10( input( 0, 0 ).getNorm() );
	gmin = gmax = v;

	// compute power spectrum, logarithmic power spectrum, minimal and maximal values
	for( y = 0; y < h; ++y ) {
		for( x = 0; x < w; ++x ) {
			v = input( x, y ).getNorm();
			power_spectrum->set( x, y, v );

			v = log10( v );
			log_spectrum->set( x, y, v );

			if( v > gmax ) gmax = v;
			if( v < gmin ) gmin = v;

			mean += v;
		}
	}

	mean /= float( w * h );

	// Compute threshold value
	float threshold( (gmax - gmin) / 100.0f * m_p + gmin);

	// Step 2 - divide logarithmic spectral image to the tiles, compute tile average value, 
	// use threshold and fill output image
	//-------------------------------------------------------------------------------------
	
	mds::tSize tw( w / m_tile_size ), th( h / m_tile_size ), tx, ty;
	float gmean;

	int low( 0 ), high( 0 );

	// for all tiles
	for( ty = 0; ty < th; ++ty ) {
		for( tx = 0; tx < tw; ++tx ) {

			// compute mean value
			gmean = getTileMean( tx, ty, *log_spectrum );

			// Step 3 - Compare mean with threshold, fill tile
			//-----------------------------------------------------------------------------
			if( gmean < threshold )
			{
				// It is P_high - copy tile
				copyTile( tx, ty, tx, ty, input, *buffer );
				++high;
			}else{
				// It is P_low - fill tile with averaged corner tile
				// fillTile( tx, ty, *buffer, tCPixel( 0.0f ) );
				copyTile( 0, 0, tx, ty, *m_corner_tile, *buffer );
				++low;
			}

		}
	}

	MDS_LOG( "Noise estimation: " << low << ", " << high << std::endl );

	// Try to create normal image from the spectrum
	ifft( *buffer, *buffer );

	// Limit new image - remove negative pixels
	buffer->forEach( CCutNegative() );

	// Transform it back to the spectral
	fft( *buffer, output );

	// That's all folks...
	return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Copy tile. 
//!
//!\param	sx			The sx. 
//!\param	sy			The sy. 
//!\param	dx			The dx. 
//!\param	dy			The dy. 
//!\param	src			Source for the. 
//!\param [in,out]	dst	Destination for the. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CFBDP::copyTile(mds::tSize sx, mds::tSize sy, mds::tSize dx, mds::tSize dy, const tCImage &src, tCImage &dst)
{
	mds::tSize sx0( sx * m_tile_size ), sy0( sy * m_tile_size ), dx0( dx * m_tile_size ), dy0( dy * m_tile_size );

	for(mds::tSize  i = 0; i < m_tile_size; i++ ) {
		mds::base::memCopy< tCPixel >( dst.getPtr( dx0, dy0 ), src.getPtr( sx0, sy0 ), m_tile_size );
		++dy0; ++sy0;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Create corner tile. 
//!
//!\param	data			The data. 
//!\param [in,out]	tile	The tile. 
//!\param	shifted			true to shifted. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CFBDP::createCornerTile(const tCImage &data, bool shifted)
{
	m_corner_tile = new tCImage( m_tile_size, m_tile_size, 0 );

	mds::tSize w( data.getXSize() ), h( data.getYSize() );
	mds::tSize cx[4], cy[4];

	if( shifted )
	{
		cx[ 0 ] = cx[ 1 ] = 0; 
		cx[ 2 ] = cx[ 3 ] = w - m_tile_size;

		cy[ 0 ] = cy[ 2 ] = 0;
		cy[ 1 ] = cy[ 3 ] = h - m_tile_size;
	}	
	else
	{
		cx[ 0 ] = cx[ 1 ] = w / 2 - m_tile_size;
		cx[ 2 ] = cx[ 3 ] = w / 2;

		cy[ 0 ] = cy[ 2 ] = h / 2 - m_tile_size;
		cy[ 1 ] = cy[ 3 ] = h / 2;
	}


	for( mds::tSize y = 0; y < m_tile_size; ++y )
	{
		for( mds::tSize x = 0; x < m_tile_size; ++x )
		{
			tCPixel mean = data( x + cx[ 0 ], y + cy[ 0 ] ) + data( x + cx[ 1 ], y + cy[ 1 ] )
						+data( x + cx[ 2 ], y + cy[ 2 ] ) + data( x + cx[ 3 ], y + cy[ 3 ] );

			mean /= 4.0f;

			m_corner_tile->set( x, y, mean );
		}
	}
}

