////////////////////////////////////////////////////////////////////////////////////////////////////
//\file  D:\BioMarker\Software\Iterative\src\RichardsonLucy.cpp
//
//\brief Implements the richardson lucy class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "RichardsonLucy.h"
#include "Constraint.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn CRichardsonLucy::CRichardsonLucy(void)
//
//\brief ! Constructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CRichardsonLucy::CRichardsonLucy(void)
{
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn CRichardsonLucy::CRichardsonLucy(const tIImage &input, const tIImage &psf)
//
//\brief ! Initializing constructor. 
//
//\param input The input. 
//\param psf   The psf. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CRichardsonLucy::CRichardsonLucy(const tIImage &input, const tIImage &psf)
{
   assert( init( input, psf ) );	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CRichardsonLucy::init(const tIImage &input, const tIImage &psf)
//
//\brief ! Initialize - set input, set PSF. 
//
//\param input The input. 
//\param psf   The psf. 
//
//\return   true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CRichardsonLucy::init(const tIImage &input, const tIImage &psf)
{
	// Do base initialization
   if( ! CComplexFilter::init( input, psf ) )
      return false;

   // Create blurred image buffer
   m_B = new tImage( m_W, m_H, 0 );
   m_Bf = new tImage( m_W, m_H, 0 );

   // Create ratios image buffer
   m_R = new tImage( m_W, m_H, 0 );
   m_Rf = new tImage( m_W, m_H, 0 );

   // Create corrections image buffer
   m_C = new tImage( m_W, m_H, 0 );
   m_Cf = new tImage( m_W, m_H, 0 );

   // Create current estimation buffer
   m_E = new tImage( m_W, m_H, 0 );
   m_Ef = new tImage( m_W, m_H, 0 );

   // Create complex conjugate psf image
   m_PSFCf = new tImage( m_W, m_H, 0 );


   // Create copy of the complex conjugate of the psf function
   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         (*m_PSFCf)( x, y ) = m_psfFD->get( x, y ).getConj();
      }
   }

   // Copy current estimation in spatial domain
   //copy( *m_input, *m_E );

   tPixel sum( 0.0f, 0.0f ), v;
   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         v = m_psfFD->get( x, y );
         sum += v;
         (*m_Ef)( x, y ) = m_inputFD->get( x, y ) * v;
      }
   }

  // *m_Ef /= sum;
   ifft( *m_Ef, *m_E, true );

	return( true );

}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CRichardsonLucy::step(void)
//
//\brief ! Do one iteration. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CRichardsonLucy::step(void)
{
   // B = E conv PSF
   fft( *m_E, *m_Ef );	
   multiply( *m_Ef, *m_psfFD, *m_Bf );
   ifft( *m_Bf, *m_B, true );

   // R = O / B
   divide( *m_input, *m_B, *m_R );

   // C = R conv PSFconj
   fft( *m_R, *m_Rf );
   multiply( *m_Rf, *m_PSFCf, *m_Cf );
   ifft( *m_Cf, *m_C, true );


   // E = E * C
   multiply( *m_E, *m_C, *m_E );

}



////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CRichardsonLucy::getOutput(tImage &image)
//
//\brief ! Get filter output. 
//
//\param [in,out] image The image. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CRichardsonLucy::getOutput(tImage &image)
{
   // Copy output window (without padding)	
   for( mds::tSize y = 0; y < m_dataH; ++y )
   {
      mds::base::memCopy( image.getPtr( 0, y ), m_E->getPtr( m_paddingW, y + m_paddingH ), m_dataW );
     //mds::base::memCopy( image.getPtr( 0, y ), m_wPSF->getPtr( m_paddingW, y + m_paddingH ), m_dataW );
   }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn int CRichardsonLucy::filter(mds::tSize max_steps)
//
//\brief ! Filter. 
//
//\param max_steps   The maximum steps. 
//
//\return   . 
////////////////////////////////////////////////////////////////////////////////////////////////////

int CRichardsonLucy::filter(mds::tSize max_steps)
{
	for( mds::tSize s = 0; s < max_steps; ++s )
      step();

	return(0);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn bool CRichardsonLucy::initFD(const tImage &input, const tImage &psf)
//
//\brief ! Initialize - get frequency domain data. 
//
//\param input The input. 
//\param psf   The psf. 
//
//\return   true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CRichardsonLucy::initFD(const tImage &input, const tImage &psf)
{
   CComplexFilter::initFD( input, psf );

   // Create blurred image buffer
   m_B = new tImage( m_W, m_H, 0 );
   m_Bf = new tImage( m_W, m_H, 0 );

   // Create ratios image buffer
   m_R = new tImage( m_W, m_H, 0 );
   m_Rf = new tImage( m_W, m_H, 0 );

   // Create corrections image buffer
   m_C = new tImage( m_W, m_H, 0 );
   m_Cf = new tImage( m_W, m_H, 0 );

   // Create current estimation buffer
   m_E = new tImage( m_W, m_H, 0 );
   m_Ef = new tImage( m_W, m_H, 0 );

   // Create complex conjugate psf image
   m_PSFCf = new tImage( m_W, m_H, 0 );

   // Create copy of the complex conjugate of the psf function
   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         (*m_PSFCf)( x, y ) = psf.get( x, y ).getConj();
      }
   }

   // Copy current estimation in spatial domain
   //copy( *m_input, *m_E );

   tPixel sum( 0.0f, 0.0f ), v;
   for( mds::tSize y = 0; y < m_H; ++y )
   {
      for( mds::tSize x = 0; x < m_W; ++x )
      {
         v = psf.get( x, y );
         sum += v;
         (*m_Ef)( x, y ) = input.get( x, y ) * v;
      }
   }

   //*m_Ef /= sum;

   // Back transform
   ifft( *m_Ef, *m_E, true );

	return( true );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void CRichardsonLucy::getOutputFD(tImage &image)
//
//\brief ! Get filter output - frequency domain. 
//
//\param [in,out] image The image. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CRichardsonLucy::getOutputFD(tImage &image)
{
	// Copy output window (without padding)	
   for( mds::tSize y = 0; y < m_dataH; ++y )
   {
      mds::base::memCopy( image.getPtr( 0, y ), m_Ef->getPtr( m_paddingW, y + m_paddingH ), m_dataW );
     //mds::base::memCopy( image.getPtr( 0, y ), m_wPSF->getPtr( m_paddingW, y + m_paddingH ), m_dataW );
   }
}

