////////////////////////////////////////////////////////////////////////////////////////////////////
//!\file src\TestFDF2D.cpp
//!
//!\brief   Implements the test fdf 2 d class. 
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "testIterative.h"
#include <MDSTk/Image/mdsImage.h>

//#define _DEBUG


//==============================================================================
/*
 * Global module constants.
 */

//! Module description
const std::string MODULE_DESCRIPTION    = "Iterative filter testing module.";

//! Additional command line arguments
const std::string MODULE_ARGUMENTS      = "filter:steps:window:noise:tile:silent";

//! Mode argument names
const std::string MODULE_ARG_MODE = "filter";
const std::string MODULE_ARG_STEPS = "steps";
const std::string MODULE_ARG_TILESIZE = "tile";
const std::string MODULE_ARG_WINDOW = "window";
const std::string MODULE_ARG_SILENT = "silent";


//! Possible filters
const std::string FILTERS[] = { "lg", "rl", "mle", "dmle", "divlg", "divrl", "divmle", "divdmle" };

//! Possible windowings
const std::string WINDOWS[] = { "tukey" };

//! Noise estimation type
const std::string NOISEESTIMATION[] = { "none" };

//! Default tile size
const unsigned default_tile_size = 1;

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   ! Default constructor. 
//!
//!\param   sDescription   The description. 
////////////////////////////////////////////////////////////////////////////////////////////////////
CTestIterative::CTestIterative(const std::string&sDescription)
   : mds::mod::CModule( sDescription )
   , m_used_filter( LAM_GOODMAN )
   , m_silent( false )
{
   allowArguments(MODULE_ARGUMENTS);	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   ! Virtual destructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////
CTestIterative::~CTestIterative(void)
{
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   ! Virtual method called on startup. 
//!
//!\return  true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////
bool CTestIterative::startup(void)
{
   // Note
   MDS_LOG_NOTE("Module startup");

   // Test of existence of input and output channel
   if( getNumOfInputs() != 2  )
   {
      MDS_CERR('<' << m_sFilename << "> Wrong number of input channels" << std::endl);
      return false;
   }

   // Get used filter
   std::string mode( "lg" );
   m_Arguments.value( MODULE_ARG_MODE, mode );
   m_used_filter = decryptFilter( mode );

   if( m_used_filter == UNKNOWN_FILTER )
   {
      MDS_CERR('<' << m_sFilename << "> Unknown filter parameter: " << mode << std::endl);
      return false;
   }

   //! Get used windowing system
   std::string window( "tukey" );
   m_Arguments.value( MODULE_ARG_WINDOW, window );
   m_used_window = decryptWindow( window );

   if( m_used_window == UNKNOWN_WINDOW )
   {
	  MDS_CERR('<' << m_sFilename << "> Unknown window parameter: " << window << std::endl);
      return false;
   }

   // Get number of steps
   unsigned steps( 1 );
   m_Arguments.value( MODULE_ARG_STEPS, steps );
   m_steps = ( steps > 0 ) ? steps : 1;

   //! Get tile size
   unsigned tilesize( default_tile_size );
   m_Arguments.value( MODULE_ARG_TILESIZE, tilesize );
   m_tile_size = tilesize;
    
   // Try if silent mode is on
   m_silent = m_Arguments.check( MODULE_ARG_SILENT );

   // O.K.
   return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   ! Called on console shutdown. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CTestIterative::shutdown(void)
{
   // Note
   MDS_LOG_NOTE("Module shutdown");	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   ! Called on writing a usage statement. 
//!
//!\param [in,out]   Stream   the stream. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CTestIterative::writeExtendedUsage(std::ostream&Stream)
{
	MDS_CERR("Extended usage: " << std::endl );
   MDS_CERR( "iterative -i file:Image.png::file:psf.png -filter filter_name [-steps number] [-silent] [-tile number]" << std::endl );
   MDS_CERR( "Options: " << std::endl );
   MDS_CERR( "-filter filter_name - specifies filter used" << std::endl );
   MDS_CERR( "          One of the following values can be used:" << std::endl );
   MDS_CERR( "          " << "lg - Lam/Goodman filter in the frequency domain." << std::endl );
   MDS_CERR( "          " << "rl - Richardson/Lucy algorithm in the spatial domain." <<std::endl );
   MDS_CERR( "          " << "mle - Maximum likelihood method in the spatial domain." << std::endl );
   MDS_CERR( "          " << "dmle - Maximum likelihood method in the spatial domain with some denoising optimalisations." << std::endl );
   MDS_CERR( "          " << "divlg - tile based filtering (tiling in the frequency domain)." << std::endl );
   MDS_CERR( "          " << "divrl - tile based filtering (tiling in the frequency domain)." <<  std::endl );
   MDS_CERR( "          " << "divmle - tile based filtering (tiling in the frequency domain)." <<  std::endl );
   MDS_CERR( "          " << "divdmle - tile based filtering (tiling in the frequency domain)." <<  std::endl );
   MDS_CERR( "-steps <1, N> - specifies number of iterations    " <<  std::endl );
   MDS_CERR( "-silent - do not write information on screen" <<  std::endl );
   MDS_CERR( "-tile <1, N> - size of the tile when using tile based filtering. Please consider that both image sizes must be dividible by this number." << std::endl );
   MDS_CERR( std::endl );
   MDS_CERR( "Input: Two png images (input image, first psf estimation image." << std::endl );
   MDS_CERR( "Output: Estimated image (for every iteration step)." << std::endl );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   ! Virtual method called by the processing thread. 
//!
//!\return  Exit-code for the process - 0 for success, else an error code. 
////////////////////////////////////////////////////////////////////////////////////////////////////
bool CTestIterative::main(void)
{
	MDS_LOG_NOTE("Module main method start");

   // I/O channels
   mds::mod::CChannel *pIChannel0 = getInput(0);
   mds::mod::CChannel *pIChannel1 = getInput(1); 
   mds::mod::CChannel *pOChannel  = getOutput(0);

   // Is any input?
   if( !( pIChannel0->isConnected() && pIChannel1->isConnected() ) )
   {
      return false;
   }

   

   // Create loading buffer
   mds::img::CDImagePtr buffer( new mds::img::CDImage( 1, 1 ) );
   
   // Wait for data - input image
    if( pIChannel0->wait(1000) )
    {

       if( mds::img::loadPNG( *buffer.get(), *pIChannel0 ) )
       {
         // Copy data to the float
         m_img_Input = new mds::img::CFImage( buffer->getXSize(), buffer->getYSize() );
         float min( mds::img::getMin< float, mds::img::CDImage>( *buffer ) );
         float max( mds::img::getMax< float, mds::img::CDImage>( *buffer ) );

         float scale( 1.0 );
         if( max > min )
            scale = 2.0f / ( max - min );

         for( int y = 0; y < buffer->getYSize(); ++y )
            for( int x = 0; x < buffer->getXSize(); ++x )
               m_img_Input->set( x, y, ( ( buffer->get( x, y ) - min ) * scale ) - 1.0f  );

       }
       else
       {
         MDS_CERR('<' << m_sFilename << "> Failed to read input image." << std::endl);
         return false;
       }
    }
    else
    {
        MDS_LOG_NOTE("Wait timeout");
    }

    // Wait for data - psf image
    if( pIChannel1->wait(1000) )
    {

       if( mds::img::loadPNG( *buffer.get(), *pIChannel1 ) )
       {
         // Copy data to the float
         m_img_PSF = new mds::img::CFImage( buffer->getXSize(), buffer->getYSize() );
         float min( mds::img::getMin< float, mds::img::CDImage>( *buffer ) );
         float max( mds::img::getMax< float, mds::img::CDImage>( *buffer ) );

         float scale( 1.0 );
         if( max > min )
            scale = 1.0f / ( max - min );
         else
         {
            if( max != 0 )
            {
               min = 0;
               scale = 1.0f / max;
            }
         }

         for( int y = 0; y < buffer->getYSize(); ++y )
            for( int x = 0; x < buffer->getXSize(); ++x )
               m_img_PSF->set( x, y, ( (buffer->get( x, y ) - min ) * scale) );

         // Create output image
         m_img_Output = new mds::img::CComplexImage( m_img_Input->getXSize(), m_img_Input->getYSize(), 0 );
       }
       else
       {
         MDS_CERR('<' << m_sFilename << "> Failed to read input image." << std::endl);
         return false;
       }
    }
    else
    {
        MDS_LOG_NOTE("Wait timeout");
    }

    // Normalize input images
    normalize( *m_img_Input, 0.0, 1.0 );
    normalize( *m_img_PSF, 0.0, 1.0 );

	if( m_img_Input->getXSize() % m_tile_size != 0 && m_img_Input->getYSize() % m_tile_size != 0 )
   {
	   MDS_CERR( "Wrong tile size - image size (" << m_img_Input->getXSize() << ", " << m_img_Input->getYSize() << ") is not divisible by this number: " << m_tile_size );
	   exit();
   }

	// Filter image
	switch( m_used_filter )
	{
	case LAM_GOODMAN:
		lamGoodman( *m_img_Input, *m_img_PSF );
		break;

	case RL:
		richardsonLucy( *m_img_Input, *m_img_PSF );
		break;

	case MLE:
		maximumLikelihood( *m_img_Input, *m_img_PSF );
		break;

   case DMLE:
      denoisedMLE( *m_img_Input, *m_img_PSF );
      break;

	case DIV_LG:
		m_div_filter.setFilterType( CDividingFilter::FT_LG );
		dividingFilter( *m_img_Input, *m_img_PSF );
		break;

	case DIV_RL:
		m_div_filter.setFilterType( CDividingFilter::FT_RL );
		dividingFilter( *m_img_Input, *m_img_PSF );
		break;

	case DIV_MLE:
		m_div_filter.setFilterType( CDividingFilter::FT_MLE );
		dividingFilter( *m_img_Input, *m_img_PSF );
		break;

   case DIV_DMLE:
		m_div_filter.setFilterType( CDividingFilter::FT_DMLE );
		dividingFilter( *m_img_Input, *m_img_PSF );
		break;

	default:
		MDS_CERR( "Unknown filter type. " << std::endl );
		return false;
	}

    return(true);
}

//==============================================================================
/*
* Function main() which creates and executes the console application.
*/
int main(int argc, char *argv[])
{
   // Creation of the module using smart pointer
   CTestIterativePtr spModule(new CTestIterative(MODULE_DESCRIPTION));

   // Initialize and execute the module
   if( spModule->init(argc, argv) )
   {
      spModule->run();
   }

   // Console application finished
   return 0;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void :::filter()
//
//\brief ! filter images. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CTestIterative::filter(  )
{
	
   
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn mds::tSize :::decryptFilter(const std::string &name)
//
//\brief ! Convert string filter name to the unsigned value. 
//
//\param name  The name. 
//
//\return   . 
////////////////////////////////////////////////////////////////////////////////////////////////////

CTestIterative::EFilter CTestIterative::decryptFilter(const std::string &name)
{
   std::string filter( name );

   // Convert filter name to lower
   std::transform(filter.begin(), filter.end(), filter.begin(), tolower);

	EFilter f;
   int c( 0 );

   // Test filter name
   for( f = EFilter_Min; f <= EFilter_Max; f++, ++c )
   {
      if( FILTERS[ c ] == filter )
         return f;
   }

   return EFilter_Min;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void :::savePNG(const mds::img::CComplexImage &image, mds::mod::CChannel *channel)
//
//\brief ! Save PNG - normalize and save image. 
//
//\param image             The image. 
//\param [in,out] channel  If non-null, the channel. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CTestIterative::savePNG(const mds::img::CFImage &image, mds::mod::CChannel *channel)
{
   // Get image sizes
   mds::tSize w( image.getXSize() ), h( image.getYSize() );

   mds::img::CFImagePtr imgCopy( new mds::img::CFImage( w, h ) );

   mds::tSize x, y;

   float min, max;

   min = max = image.get( 0, 0 );
   // Copy data
   for( y = 0; y < h; ++y )
   {
      for( x = 0; x < w; ++x )
      {
         float v = image.get( x, y );

         min = ( v < min ) ? v : min;
         max = ( v > max ) ? v : max;

         imgCopy->set( x, y, v );
      }
   }
   
   float scale(0.0);

   // Compute normalization factor
   if( min < max )
      scale = ( 2.0f / float( max - min ) );

   // Normalize image
   *imgCopy -= min;
   *imgCopy *= scale;
   *imgCopy -= 1.0f;

   // Save png file
   if( ! mds::img::savePNG( *imgCopy.get(), *channel) )
    {
         MDS_CERR('<' << m_sFilename << "> Failed to write output image." << std::endl);
    }

}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void :::savePNG(const mds::img::CComplexImage &image, mds::mod::CChannel *channel)
//
//\brief ! Save complex image as png picture. 
//
//\param image             The image. 
//\param [in,out] channel  If non-null, the channel. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CTestIterative::savePNG(const mds::img::CComplexImage &image, mds::mod::CChannel *channel)
{
// Get image sizes
   mds::tSize w( image.getXSize() ), h( image.getYSize() );

   mds::img::CFImagePtr imgCopy( new mds::img::CFImage( w, h ) );

   mds::tSize x, y;

   float min, max;

   min = max = image.get( 0, 0 ).getNorm();
   // Copy data
   for( y = 0; y < h; ++y )
   {
      for( x = 0; x < w; ++x )
      {
         float v = image.get( x, y ).getNorm();

         min = ( v < min ) ? v : min;
         max = ( v > max ) ? v : max;

         imgCopy->set( x, y, v );
      }
   }
   
   float scale(0.0);

   // Compute normalization factor
   if( min < max )
      scale = ( 2.0f / float( max - min ) );

   // Normalize image
   *imgCopy -= min;
   *imgCopy *= scale;
   *imgCopy -= 1.0f;

   // Save png file
   if( ! mds::img::savePNG( *imgCopy.get(), *channel) )
    {
         MDS_CERR('<' << m_sFilename << "> Failed to write output image." << std::endl);
    }	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void :::savePNG(const mds::img::CComplexImage &image, const std::string &fname)
//
//\brief ! Save complex image as a png picture - fname version. 
//
//\param image The image. 
//\param fname Filename of the file. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CTestIterative::savePNG(const mds::img::CComplexImage &image, const std::string &fname)
{
   mds::mod::CFileChannel channel( mds::mod::CH_OUT, fname );

   if( channel.connect() )
   {
      savePNG( image, &channel );
   }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void :::lamGoodman(const mds::img::CFImage &input, const mds::img::CFImage &psf)
//
//\brief ! Filter by Lam-Goodman method. 
//
//\param input The input. 
//\param psf   The psf. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CTestIterative::lamGoodman(const mds::img::CFImage &input, const mds::img::CFImage &psf)
{
    m_lg_filter.init( input, psf );
   
#ifdef _DEBUG
	   mds::img::CComplexImagePtr noise( new mds::img::CComplexImage( m_lg_filter.getPaddedWidth(), m_lg_filter.getPaddedHeight(), 0 ) );
	   m_lg_filter.getNoiseEstimation( *noise );
      savePNG( *noise, "outputNoise.png" );
#endif

   MDS_LOG_NOTE( "Starting filter: Lam-Goodman. Steps: " << m_steps << std::endl );

   if( !m_silent )
      MDS_CERR( "Starting filter: Lam-Goodman. Steps: " << m_steps << std::endl );

	for( int i = 0; i < m_steps; i++ )
    {
      std::stringstream ss;
      ss.width( 3 );
      ss.fill( '_' );
      ss << "output_" << i << ".png";

      MDS_LOG_NOTE( "Writing file " << i + 1 << " of " << m_steps << " : " << ss.str() << std::endl );

       // Write info
       if( ! m_silent )
       {
          MDS_CERR( "Writing file " << i + 1 << " of " << m_steps << " : " << ss.str() << std::endl );
       }

      m_lg_filter.filter(1);
      m_lg_filter.getOutput( *m_img_Output );
      savePNG( *m_img_Output, ss.str() );
    }
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void :::richardsonLucy(const mds::img::CFImage &input, const mds::img::CFImage &psf)
//
//\brief ! Filter by Richardson-Lucy method. 
//
//\param input The input. 
//\param psf   The psf. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CTestIterative::richardsonLucy(const mds::img::CFImage &input, const mds::img::CFImage &psf)
{
#ifdef _DEBUG
       m_rl_filter.init( input, psf );
       m_rl_filter.getOutput( *m_img_Output );
      savePNG( *m_img_Output, "/output/outputPSF.png" );
#endif

    MDS_LOG_NOTE( "Starting filter: Richardson-Lucy. Steps: " << m_steps << std::endl );

    if( ! m_silent )
       MDS_CERR( "Starting filter: Richardson-Lucy. Steps: " << m_steps << std::endl );

    for( int i = 0; i < m_steps; i++ )
    {
      std::stringstream ss;
      ss.width( 3 );
      ss.fill( '_' );
      ss << "output_" << i << ".png";

      MDS_LOG_NOTE( "Writing file " << i + 1 << " of " << m_steps << " : " << ss.str() << std::endl );

      // Write info
      if( ! m_silent )
      {
         MDS_CERR( "Writing file " << i + 1 << " of " << m_steps << " : " << ss.str() << std::endl );
      }

      m_rl_filter.filter(1);
      m_rl_filter.getOutput( *m_img_Output );
      savePNG( *m_img_Output, ss.str() );
    }
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void :::maximumLikelihood(const mds::img::CFImage &input, const mds::img::CFImage &psf)
//
//\brief ! Filter by maximum-likelihood method. 
//
//\param input The input. 
//\param psf   The psf. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CTestIterative::maximumLikelihood(const mds::img::CFImage &input, const mds::img::CFImage &psf)
{
       m_mle_filter.init( input, psf );

#ifdef _DEBUG
       m_mle_filter.getOutput( *m_img_Output );
       savePNG( *m_img_Output, "/output/outputPSF.png" );
#endif

    MDS_LOG_NOTE( "Starting filter: MLE. Steps: " << m_steps << std::endl );
    
    if( ! m_silent )
       MDS_CERR( "Starting filter: MLE. Steps: " << m_steps << std::endl );

    for( int i = 0; i < m_steps; i++ )
    {
      std::stringstream ss;
      ss.width( 3 );
      ss.fill( '_' );
      ss << "output_" << i << ".png";

      MDS_LOG_NOTE( "Writing file " << i + 1 << " of " << m_steps << " : " << ss.str() << std::endl );

      // Write info
      if( ! m_silent )
      {
         MDS_CERR( "Writing file " << i + 1 << " of " << m_steps << " : " << ss.str() << std::endl );
      }

      m_mle_filter.filter(1);
      m_mle_filter.getOutput( *m_img_Output );
      savePNG( *m_img_Output, ss.str() );
    }

}

////////////////////////////////////////////////////////////////////////////////////////////////////
//\fn void :::dividingFilter(const mds::img::CFImage &input, const mds::img::CFImage &psf)
//
//\brief ! Use dividing filter. 
//
//\param input The input. 
//\param psf   The psf. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CTestIterative::dividingFilter(const mds::img::CFImage &input, const mds::img::CFImage &psf)
{
       m_div_filter.init( input, psf );

#ifdef _DEBUG
       m_div_filter.getOutput( *m_img_Output );
       savePNG( *m_img_Output, "/output/outputPSF.png" );
#endif

    std::stringstream lognote;

    lognote << "Starting filter: Tiling. Subtype: ";

    switch( m_used_filter )
    {
    case DIV_LG:
       lognote << "Lam-Goodman" << std::endl;
		break;

    case DIV_RL:
       lognote << "Richardson-Lucy" << std::endl;
	   break;

    case DIV_MLE:
       lognote << "MLE" << std::endl;
      break;

    case DIV_DMLE:
       lognote << "Denoised MLE" << std::endl;
      break;
    default:
	   MDS_CERR( "Unknown filter type. " << std::endl );
      return;
	 };

    lognote << "Steps: " << m_steps << std::endl;
    lognote << "Used tile size: " << m_tile_size << std::endl;

    MDS_LOG_NOTE( lognote.str() );
    if( ! m_silent )
    {
       MDS_CERR( lognote.str() );
    }

    for( int i = 0; i < m_steps; i++ )
    {
      std::stringstream ss;
      ss.width( 3 );
      ss.fill( '_' );
      ss << "output_" << i << ".png";

      MDS_LOG_NOTE( "Writing file " << i + 1 << " of " << m_steps << " : " << ss.str() << std::endl );

      // Write info
      if( ! m_silent )
      {
         MDS_CERR( "Writing file " << i + 1 << " of " << m_steps << " : " << ss.str() << std::endl );
      }

      m_div_filter.filter(1);
      m_div_filter.getOutput( *m_img_Output );
      savePNG( *m_img_Output, ss.str() );
    }

}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief	! Convert string window name to the enum. 
//!
//!\param	name	The name. 
//!
//!\return	. 
////////////////////////////////////////////////////////////////////////////////////////////////////
CTestIterative::EWindow CTestIterative::decryptWindow(const std::string &name)
{
	std::string window( name );

   // Convert window name to lower
   std::transform(window.begin(), window.end(), window.begin(), tolower);

   EWindow w;
   int c( 0 );

   // Test filter name
   for( w = EWindow_Min; w <= EWindow_Max; w++, c++ )
   {
      if( FILTERS[ c ] == window )
         return w;
   }

   return EWindow_Min;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
//!\brief   ! Filter by denoising mle method. 
//!
//!\param   input The input. 
//!\param   psf   The psf. 
////////////////////////////////////////////////////////////////////////////////////////////////////
void CTestIterative::denoisedMLE(const mds::img::CFImage &input, const mds::img::CFImage &psf)
{
	m_dmle_filter.init( input, psf );

#ifdef _DEBUG
       m_dmle_filter.getOutput( *m_img_Output );
       savePNG( *m_img_Output, "/output/outputPSF.png" );
#endif

    MDS_LOG_NOTE( "Starting filter: Denoised MLE. Steps: " << m_steps << std::endl );
    
    if( ! m_silent )
       MDS_CERR( "Starting filter: Denoised MLE. Steps: " << m_steps << std::endl );

    for( int i = 0; i < m_steps; i++ )
    {
      std::stringstream ss;
      ss.width( 3 );
      ss.fill( '_' );
      ss << "output_" << i << ".png";

      MDS_LOG_NOTE( "Writing file " << i + 1 << " of " << m_steps << " : " << ss.str() << std::endl );

      // Write info
      if( ! m_silent )
      {
         MDS_CERR( "Writing file " << i + 1 << " of " << m_steps << " : " << ss.str() << std::endl );
      }

      m_dmle_filter.filter(1);
      m_dmle_filter.getOutput( *m_img_Output );
      savePNG( *m_img_Output, ss.str() );
    }
}

