/** @file packet.cpp
* zdrojovy soubor s implemenaci zpracovani paketu
* @author Radek Hranicky
*/

/*
LIS Noise Cleaner - Tento nastroj slouzi k filtraci sumu vytvoreneho programem LDP
Copyright (C) 2012 Radek Hranicky

Tento program je svobodny software: muzete jej sirit a upravovat podle ustanoveni Obecne verejne licence GNU (GNU General Public License),
vydavane Free Software Foundation a to bud podle 3. verze teto Licence, nebo (podle vaseho uvazeni) kterekoli pozdejsi verze.

Tento program je rozsirovan v nadeji, ze bude uzitecny, avas BEZ JAKEKOLIV ZARUKY. Neposkytuji se ani odvozene zaruky PRODEJNOSTI
anebo VHODNOSTI PRO URCITY UCEL. Dalsi podrobnosti hledejte v Obecne verejne licenci GNU.

Kopii Obecne verejne licence GNU jste meli obdrzet spolu s timto programem. Pokud se tak nestalo, najdete ji zde: <http://www.gnu.org/licenses/>.
*/

#include <iostream>
#include <arpa/inet.h>
#include <netinet/if_ether.h>
#include <netinet/ip6.h>
#include <netinet/tcp.h>

#include "errors.h"
#include "config.h"
#include "packet.h"

/**
 * Zpracovani paketu v ramci prvni faze - nacteni informaci o jednobytovych TCP segmentech
 * a v ramci koliznich vyber segmentu s nejvyssim Hop Limitem
 * @param pcap hlavicka
 * @param paket
 * @param konfigurace programu
 * @return informace o paketu
 */
int processPacket_phase1(pcap_pkthdr *pcap_header, u_char *packet, TConfig &cfg) {
  int errNum = OK;
  cfg.pktCount++;
  TpacketInfo packetInfo = analyzePacket(pcap_header, packet); // analyza paketu

  if (packetInfo.srcPort == 0) { // nejde o TCP segment
    return errNum;
  }

  cfg.tcpCount++;

  if (packetInfo.tcpPayload != 1) { // zajimaji nas pouze jednobytove TCP segmenty
    return errNum;
  }

  cfg.byteTcpCount++; // jde o jednobytovy TCP segment

  // hledame zda jiz nemame ulozen kolizni TCP segment
  bool collision_exists = false;
  list<TpacketInfo>::iterator i;
  for(i=cfg.selected_packets.begin(); i !=cfg.selected_packets.end(); ++i) {
    if (compareIp6Addr(packetInfo.srcAddr, i->srcAddr)
        && compareIp6Addr(packetInfo.dstAddr, i->dstAddr)
        && packetInfo.srcPort == i->srcPort
        && packetInfo.dstPort == i->dstPort
        && packetInfo.tcpPayload == i->tcpPayload
        && packetInfo.seqNum == i->seqNum)
    {
      collision_exists = true;
      // ma nove nalezeny vyssi hop limit a obsahuje jiny znak?
      if (packetInfo.hopLimit > i->hopLimit && packetInfo.secretByte != i->secretByte) {
        *i = packetInfo;
      }
      break;
    }
  }

  // pokud zatim neexistuje kolizni segment, pridame tento do seznamu
  if (!collision_exists) {
    cfg.selected_packets.push_back(packetInfo);
  }
  return errNum;
}


/**
 * Zpracovani paketu v ramci druhe faze - zapis vybranych paketu do vystupniho souboru
 * @param pcap hlavicka
 * @param paket
 * @param konfigurace programu
 * @return informace o paketu
 */
int processPacket_phase2(pcap_pkthdr *pcap_header, u_char *packet, TConfig &cfg) {
  int errNum = OK;
  TpacketInfo packetInfo = analyzePacket(pcap_header, packet);   // analyza paketu

  // zjistime, zda jde o jednobytovy TCP segment
  if (packetInfo.tcpPayload == 1) {
    // zjistime, zda budeme tento paket zapisovat na vystup
    list<TpacketInfo>::iterator i;
    // zjistime zda je znak pro toto sekvencni cislo spravny
    for(i=cfg.selected_packets.begin(); i !=cfg.selected_packets.end(); ++i) {
      if (compareIp6Addr(packetInfo.srcAddr, i->srcAddr)
        && compareIp6Addr(packetInfo.dstAddr, i->dstAddr)
        && packetInfo.srcPort == i->srcPort
        && packetInfo.dstPort == i->dstPort
        && packetInfo.tcpPayload == i->tcpPayload
        && packetInfo.secretByte == i->secretByte
        && packetInfo.seqNum == i->seqNum) {
        pcap_dump((u_char *)cfg.pcapDump, pcap_header, packet); // zapis na vystup
        cfg.pktWrittenCount++;
        break;
      }
    }
  } else {
    // nejde o jednobytovy tcp paket
    pcap_dump((u_char *)cfg.pcapDump, pcap_header, packet);      // zapis na vystup
    cfg.pktWrittenCount++;
  }
  return errNum;
}


/**
 * Analyza paketu
 * @param pcap hlavicka
 * @param paket
 * @return informace o paketu
 */
TpacketInfo analyzePacket(pcap_pkthdr *pcap_header, u_char *packet) {
  // vytvoreni struktury s informacemi o paketu
  TpacketInfo packetInfo;
  packetInfo.srcPort = 0;
  packetInfo.dstPort = 0;
  packetInfo.seqNum = 0;
  packetInfo.hopLimit = 0;
  packetInfo.tcpPayload = 0;
  packetInfo.secretByte = 0;

  unsigned int eth_frameLen = pcap_header->len;
  struct ether_header *ether;             // ukazatel na strukturu Ethernetoveho headeru
  ether = (struct ether_header *) packet; // nastavime jej na zacatek naseho paketu

  if (ntohs(ether->ether_type) != ETHERTYPE_IPV6) { // zajimaji nas pouze IPv6 datagramy
    return packetInfo;
  }

  // =========================================== ANALYZA IPV6 ===========================================
  uint8_t *ipv6_datagram;                     // ukazatel na IPv6 datagram
  struct ip6_hdr *ipv6_header;                // ukazatel na strukturu IPv6 headeru
  u_int ipv6_datagramLen = 0;                 // delka IPv6 datagramu

  ipv6_datagram = (uint8_t*)packet + sizeof(struct ether_header);  // ziskame zkazatel na IPv6 datagram
  ipv6_header = (struct ip6_hdr*)ipv6_datagram;                    // ziskame ukazatel na IPv6 hlavicku
  ipv6_datagramLen = eth_frameLen - sizeof(struct ether_header);   // vypocet delky IPv6 datagramu

  // ulozeni informaci z IPv6 hlavicky
  packetInfo.srcAddr = ipv6_header->ip6_src;
  packetInfo.dstAddr = ipv6_header->ip6_dst;
  packetInfo.hopLimit = ipv6_header->ip6_ctlun.ip6_un1.ip6_un1_hlim;

  // ============================================ ANALYZA TCP ===========================================

  if (ipv6_header->ip6_ctlun.ip6_un1.ip6_un1_nxt != NEXT_HEADER_TCP) { // nejde o TCP segment
    //cout << "non-TCP" << endl;
    return packetInfo;
  }
  struct tcphdr *tcp_header;
  tcp_header = (struct tcphdr *)((uint8_t *)ipv6_header + IPV6_FIXED_HEADER_LEN);

  // ulozeni informaci z TCP hlavicky
  packetInfo.srcPort = ntohs(tcp_header->source);
  packetInfo.dstPort = ntohs(tcp_header->dest);
  packetInfo.seqNum = ntohl(tcp_header->seq);

  unsigned short int data_offset = tcp_header->doff;

  packetInfo.tcpPayload = ntohs(ipv6_header->ip6_ctlun.ip6_un1.ip6_un1_plen) - data_offset * 4;

  // Jde o jednobytovy TCP segment?
  if (packetInfo.tcpPayload == 1) { // muze obsahovat znak tajne zpravy
    unsigned int secretbyteOffset = sizeof(struct ether_header) + IPV6_FIXED_HEADER_LEN + data_offset * 4;
    packetInfo.secretByte = packet[secretbyteOffset];
  }
  return packetInfo;
}


/**
 * Porovnani dvou IPv6 adres
 * @param prvni adresa
 * @param druha adresa
 * @return shoda
 */
bool compareIp6Addr(in6_addr &addr1, in6_addr &addr2) {
  bool match = true;
  for (unsigned int octet = 0; octet < 16; octet++) {
    if (addr1.s6_addr[octet] != addr2.s6_addr[octet]) {
      match = false;
      break;
    }
  }
  return match;
}
