----------------------------------------------------------------------------------
-- Detektor objektu
-- Copyright (C) 2012 Brno University of Technology,
--                        Faculty of Information Technology
-- Author(s): Petr Musil <xmusilpetr AT fit.vutbr.cz>
--  
-- LICENSE TERMS
--  
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 	1. Redistributions of source code must retain the above copyright
-- 	notice, this list of conditions and the following disclaimer.
-- 	2. Redistributions in binary form must reproduce the above copyright
-- 	notice, this list of conditions and the following disclaimer in
-- 	the documentation and/or other materials provided with the
-- 	distribution.
-- 	3. All advertising materials mentioning features or use of this software
-- 	or firmware must display the following acknowledgement:
--   
-- 	This product includes software developed by the University of
-- 	Technology, Faculty of Information Technology, Brno and its
-- 	contributors.
--  
-- 	4. Neither the name of the Company nor the names of its contributors
-- 	may be used to endorse or promote products derived from this
-- 	software without specific prior written permission.
--  
-- This software or firmware is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
-- 
--      $Id$
----------------------------------------------------------------------------------


library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.std_logic_unsigned.all;
use work.configure.all;		-- my data types
use work.data_types.all;		-- my data types


entity extractor is
	port(
		CLK 		: in std_logic;
		OPTMODE	: in feature_mode;
		
		TABLE_A	: in std_logic_vector(9 downto 0);	--použitá tabulka
		LBPMASK_A	: in std_logic_vector(2 downto 0);	--maska pro LBP příznak, 8 možností
		RANKA_A		: in std_logic_vector(3 downto 0);
		RANKB_A		: in std_logic_vector(3 downto 0);
		
		TABLE_B	: in std_logic_vector(9 downto 0);	--použitá tabulka
		LBPMASK_B	: in std_logic_vector(2 downto 0);	--maska pro LBP příznak, 8 možností
		RANKA_B		: in std_logic_vector(3 downto 0);
		RANKB_B		: in std_logic_vector(3 downto 0);
		
		DATA_IN_A	: in dsp_out;
		DATA_IN_B	: in dsp_out;
		
		DATA_OUT_A	: out std_logic_vector(8 downto 0);
		DATA_OUT_B	: out std_logic_vector(8 downto 0)		
	);
end extractor;

architecture Behavioral of extractor is

	component lbp is 
		port(
			CLK 		: in std_logic;
			LBPMASK 	: in std_logic_vector(2 downto 0);			
			DATA_IN	: in dsp_out;			
			DATA_OUT	: out std_logic_vector(6 downto 0)
		);
	end component;
	
	component lrd is
		port(
			CLK 		: in std_logic;
			RANK_POS_A	: in std_logic_vector(3 downto 0); -- pozice RankA (0-8)
			RANK_POS_B	: in std_logic_vector(3 downto 0);			
			DATA_IN	: in dsp_out;			
			DATA_OUT	: out std_logic_vector(6 downto 0)
		);
	end component;
	
	component lrp is
		port(
			CLK 		: in std_logic;
			RANK_POS_A	: in std_logic_vector(3 downto 0); -- pozice RankA (0-8)
			RANK_POS_B	: in std_logic_vector(3 downto 0);			
			DATA_IN	: in dsp_out;			
			DATA_OUT	: out std_logic_vector(6 downto 0)
		);
	end component;
	
	component lrd_table is
		port(
			CLK	: in std_logic;		
			TABLE_A	: in std_logic_vector(9 downto 0);
			INDEX_A	: in std_logic_vector(4 downto 0);
			OUT_A		: out std_logic_vector(8 downto 0);		
			TABLE_B	: in std_logic_vector(9 downto 0);
			INDEX_B	: in std_logic_vector(4 downto 0);
			OUT_B		: out std_logic_vector(8 downto 0)
		);
	end component;
		
	signal lbp_feature_A, lbp_feature_B 	: std_logic_vector(6 downto 0);
	signal lrd_feature_A, lrd_feature_B		: std_logic_vector(6 downto 0);
	signal lrp_feature_A, lrp_feature_B		: std_logic_vector(6 downto 0);
	
	signal index_A			: std_logic_vector(6 downto 0);
	signal index_B			: std_logic_vector(6 downto 0);
	signal index_lrd_A	: std_logic_vector(4 downto 0);
	signal index_lrd_B	: std_logic_vector(4 downto 0);
	signal table			: std_logic_vector(9 downto 0);
	
	signal next_table_A	: std_logic_vector(9 downto 0);
	signal next2_table_A	: std_logic_vector(9 downto 0);
	signal next_table_B	: std_logic_vector(9 downto 0);
	signal next2_table_B	: std_logic_vector(9 downto 0);
	
	
begin

SYN_PROC: process (CLK)
	begin
		if( CLK'event and CLK = '1') then			
			next_table_A  <= TABLE_A;
			next2_table_A <= next_table_A;
			
			next_table_B  <= TABLE_B;
			next2_table_B <= next_table_B;
		end if;
end process;

	WITH OPTMODE  SELECT
		index_A		 <=	lrd_feature_A 	WHEN F_LRD,
								lrp_feature_A 	WHEN F_LRP,
								lbp_feature_A 	WHEN F_LBP,
								"0000000" 		WHEN F_NO,
								"0000000"   	WHEN OTHERS;	

	WITH OPTMODE  SELECT
		index_B		 <=	lrd_feature_B 	WHEN F_LRD,
								lrp_feature_B 	WHEN F_LRP,
								lbp_feature_B 	WHEN F_LBP,
								"0000000" 		WHEN F_NO,
								"0000000"   	WHEN OTHERS;
	
	index_lrd_A <= index_A(4 downto 0);	
	index_lrd_B <= index_B(4 downto 0);

	map_lbp_A: lbp	port map(
		CLK => CLK,
		LBPMASK => LBPMASK_A,		
		DATA_IN => DATA_IN_A,		
		DATA_OUT	=>lbp_feature_A
	);
	
	map_lbp_B: lbp	port map(
		CLK => CLK,
		LBPMASK => LBPMASK_B,		
		DATA_IN => DATA_IN_B,		
		DATA_OUT	=>lbp_feature_B
	);
	
	map_lrd_A: lrd port map(
		CLK => CLK,
		RANK_POS_A => RANKA_A,
		RANK_POS_B => RANKB_A,		
		DATA_IN => DATA_IN_A,		
		DATA_OUT => lrd_feature_A
	);
	
	map_lrd_B: lrd port map(
		CLK => CLK,
		RANK_POS_A => RANKA_B,
		RANK_POS_B => RANKB_B,		
		DATA_IN => DATA_IN_B,		
		DATA_OUT => lrd_feature_B
	);
	
	map_lrp_A: lrp port map(
		CLK => CLK,
		RANK_POS_A => RANKA_A,
		RANK_POS_B => RANKB_A,		
		DATA_IN => DATA_IN_A,		
		DATA_OUT => lrp_feature_A
	);	
	
	map_lrp_B: lrp port map(
		CLK => CLK,
		RANK_POS_A => RANKA_B,
		RANK_POS_B => RANKB_B,		
		DATA_IN => DATA_IN_B,		
		DATA_OUT => lrp_feature_B
	);	
	
	lrd_table_map: lrd_table port map(
		CLK		=> CLK,		
		TABLE_A	=> next2_table_A,
		INDEX_A	=> index_lrd_A,
		OUT_A		=> DATA_OUT_A,	
		TABLE_B	=> next2_table_B,
		INDEX_B	=> index_lrd_B,
		OUT_B		=> DATA_OUT_B
	);
	
end architecture;
 
